﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using CBRobot;

namespace CBRobotConsole {

    /** This class represents a copy of a program which has been compiled into
     *  a path that can be executed by the robot.
     */
    class CompiledProgram : Path {

        private List<CompiledProgramEntry> entries;
        private int lastExecutedEntryIndex;
        private bool loop;

        private class CompiledProgramEntry {
            public double EndTime;          /* End time of the entry */
            public int Index;               /* Index of the entry in the original program */
            public ArmPosition TargetPosition;  /* Target position for this entry */
            public ArmSpeed Speed;          /* Speed for this entry */
        }

        public CompiledProgram(SavedProgram program) {
            lastExecutedEntryIndex = -1;

            // Compile the program into an array of target positions / speeds
            entries = new List<CompiledProgramEntry>();
            ArmPosition lastPosition = null;
            double endTime = 0.0;
            for (int index = 0; index < program.Entries.Count; index ++) {
                Robot robot = MainWindow.Instance.Robot;
                ProgramEntry originalEntry = program.Entries[index];

                // Figure out how long it will take to reach the target position,
                // and the per-motor speed that will cause each motor to arrive
                // at the same time.
                ArmPosition position = originalEntry.PositionTransformer.Position;
                if (lastPosition == null) { lastPosition = position; }
                double speed = originalEntry.SpeedTransformer.Speed;
                ArmSpeed speedVector = null;
                double timeToReach = 0.0;
                ComputeSpeedVectorAndTime(position, lastPosition, speed, robot, out speedVector, out timeToReach);
                endTime += timeToReach + originalEntry.Pause;

                CompiledProgramEntry entry = new CompiledProgramEntry();
                entry.EndTime = endTime;
                entry.Index = index;
                entry.TargetPosition = position;
                entry.Speed = speedVector;
                entries.Add(entry);

                lastPosition = position;
            }
        }
        
        /** Gets the index of the last executed entry in the original program */
        public int LastExecutedEntryIndex {
            get { return lastExecutedEntryIndex; }
            set {
                if (lastExecutedEntryIndex != value) {
                    lastExecutedEntryIndex = value;
                    OnLastExecutedEntryIndexChanged();
                }
            }
        }

        /** Fired when the LastExecutedEntryIndex property changes */
        public event EventHandler LastExecutedEntryIndexChanged;
        
        /** Fires the LastExecutedEntryIndexChanged event */
        private void OnLastExecutedEntryIndexChanged() {
            if (LastExecutedEntryIndexChanged != null) {
                LastExecutedEntryIndexChanged(this, EventArgs.Empty);
            }
        }
        
        // (inherited from Path)
        public ArmPosition StartPosition {
            get {
                if (entries.Count > 0) {
                    return entries[0].TargetPosition;
                }
                else {
                    return ArmPosition.Zero;
                }
            }
        }

        // (inherited from Path)
        public void GetParametersAtTime(double time, out ArmPosition position, out ArmSpeed speed) {
            CompiledProgramEntry entry = GetEntryAtTime(time);
            position = entry != null ? entry.TargetPosition : ArmPosition.Zero;
            speed = entry != null ? entry.Speed : ArmSpeed.Zero;
        }

        // (inherited from Path)
        public double PathLength {
            get {
                if (entries.Count == 0) {
                    return 0.0;
                }
                else {
                    return entries[entries.Count - 1].EndTime;
                }
            }
        }

        // (inherited from Path)
        public bool Loop {
            get { return loop; }
            set { loop = value; }
        }

        // (inherited from Path)
        public void SetPathTime(double time) {
            CompiledProgramEntry entry = GetEntryAtTime(time);
            if (entry != null) { LastExecutedEntryIndex = entry.Index; }
        }

        /** Returns the entry associated with the given time value.  Returns the
         *  first entry if the specified time is less than zero, or the last entry if
         *  the specified time is beyond the end of the path.
         */
        private CompiledProgramEntry GetEntryAtTime(double time) {
            if (entries.Count == 0) {
                return null;
            }
            if (time < 0) {
                return entries[0];
            }
            // Return the first entry with an end time that's not already passed
            foreach (CompiledProgramEntry entry in entries) {
                if (entry.EndTime - time > 0) {
                    return entry;
                }
            }
            return entries[entries.Count - 1];
        }

        /** Computes the amount of time that will be required to reach the
         *  given target position at the given "overall speed", and also
         *  returns the per-motor speed vector that should be used to have
         *  all motors arrive at the same time.
         */
        private static void ComputeSpeedVectorAndTime(ArmPosition targetPosition,
            ArmPosition startPosition, double speed, Robot robot,
            out ArmSpeed speedVector, out double timeToReach) {
            Debug.Assert(targetPosition != null);
            Debug.Assert(startPosition != null);
            Debug.Assert(robot != null);

            // Compute how long it should take to reach the target position at
            // the given overall speed.
            DofVector targetTip = targetPosition.TipPosition.ConvertToDofPoint(robot);
            DofVector startTip = startPosition.TipPosition.ConvertToDofPoint(robot);
            double dM1 = targetTip.M1 - startTip.M1;
            double dM2 = targetTip.M2 - startTip.M2;
            double dM3 = targetTip.M3 - startTip.M3;
            double dM4 = targetPosition.M4 - startPosition.M4;
            double magnitude = Math.Sqrt(dM1 * dM1 + dM2 * dM2 + dM3 * dM3 + dM4 * dM4);
            double time = magnitude / speed;

            if (time > 0) {
                speedVector = new ArmSpeed(Math.Abs(dM1 / time),
                    Math.Abs(dM2 / time), Math.Abs(dM3 / time),
                    Math.Abs(dM4 / time));
            }
            else {
                speedVector = ArmSpeed.Zero;
            }
            timeToReach = time;
        }
    }
}
