﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using CBRobot;

namespace CBRobotConsole {

    public partial class MainWindow : Form {

        private const int PositionUpdateInterval = 50;  /* Update the current position at this interval */
        private const double SpinnerRate = 5.0;         /* Number of degrees to change per spinner tick */

        private Robot robot;        /* The main robot instance */
        private Timer updateTimer;  /* Timer for updating the position display */
        private int preventRobotUpdate;         /* > 0 to prevent robot updates */
        private int preventTransformerUpdate;   /* > 0 to prevent position / speed from being updated by text field changes */
        private int preventTextUpdate;          /* > 0 to prevent text fields from being updated by position changes */
        private SavedProgram savedProgram;          /* The current saved program */
        private CompiledProgram compiledProgram;    /* The saved program, compiled so it can run */
        private Dictionary<ProgramEntry, DataGridViewRow> rowMap;   /* Map from entry to row */
        private string savePath;    /* Path to the saved program */
        private ArmSpeed speedBeforeRunningProgram;     /* Arm speed before running the program */

        private PositionTransformer currentPositionTransformer; /* Stores / translates current position */
        private PositionTransformer targetPositionTransformer;  /* Stores / translates target position */
        private SpeedTransformer speedTransformer;              /* Stores / translates target speed */

        private ContextMenuStrip contextMenu;
        private ToolStripMenuItem goToMenuItem;
        private ToolStripMenuItem moveUpMenuItem;
        private ToolStripMenuItem moveDownMenuItem;
        private ToolStripMenuItem removeMenuItem;

        private static MainWindow instance;     /* The shared main window instance */

        /** Returns the single main window instance */
        public static MainWindow Instance {
            get {
                if (instance == null) { instance = new MainWindow(); }
                return instance;
            }
        }

        public MainWindow() {
            InitializeComponent();
        }
        
        /** Called when the main window is first loaded */
        private void MainWindow_Load(object sender, EventArgs e) {

            // Override the default font of the DataGridView - this can't be
            // done in the designer because there is a bug preventing it if
            // the form itself has a default font set.
            DataGridViewCellStyle f = new DataGridViewCellStyle(grid.DefaultCellStyle);
            f.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F);
            f.Alignment = DataGridViewContentAlignment.MiddleRight;
            grid.ColumnHeadersDefaultCellStyle = f;
            grid.DefaultCellStyle = f;

            aboutLabel.Text = aboutLabel.Text.Replace("{$v}", Application.ProductVersion);

            robot = new Robot(Handle);
            robot.ConnectionStatusChanged += robot_ConnectionStatusChanged;
            robot.TargetPositionChanged += robot_TargetPositionChanged;
            robot.SpeedChanged += robot_SpeedChanged;
            robot.PathChanged += robot_PathChanged;
            currentPositionTransformer = new PositionTransformer();
            currentPositionTransformer.PositionChanged += currentPositionTransformer_Changed;
            currentPositionTransformer.UnitChanged += currentPositionTransformer_Changed;
            targetPositionTransformer = new PositionTransformer();
            targetPositionTransformer.UnitChanged += targetPositionTransformer_UnitChanged;
            targetPositionTransformer.PositionChanged += targetPositionTransformer_PositionChanged;
            targetPositionTransformer.ComponentChanged += targetPositionTransformer_ComponentChanged;
            speedTransformer = new SpeedTransformer();
            speedTransformer.Speed = robot.Speed;
            speedTransformer.UnitChanged += speedTransformer_UnitChanged;
            speedTransformer.SpeedChanged += speedTransformer_SpeedChanged;
            speedTransformer.ComponentChanged += speedTransformer_ComponentChanged;
            statusControl.PausedChanged += statusControl_PausedChanged;
            rowMap = new Dictionary<ProgramEntry, DataGridViewRow>();
            contextMenu = new ContextMenuStrip();
            goToMenuItem = new ToolStripMenuItem("&Go to this Position", null, contextMenu_GoToPosition);
            moveUpMenuItem = new ToolStripMenuItem("Move &Up", null, contextMenu_MoveUp);
            moveDownMenuItem = new ToolStripMenuItem("Move &Down", null, contextMenu_MoveDown);
            removeMenuItem = new ToolStripMenuItem("&Remove", null, contextMenu_Remove);

            updateTimer = new Timer();
            updateTimer.Interval = PositionUpdateInterval;
            updateTimer.Tick += updateTimer_Tick;
            updateTimer.Enabled = true;

            // Set the initial unit
            UnitButton_CheckedChanged(this, EventArgs.Empty);

            // Set the initial connection / paused status display
            UpdateStatusControl();

            SetProgram(new SavedProgram());
        }

        /** Windows messages must be forwarded to the robot for device
         *  connection / disconnection detection.
         */
        protected override void WndProc(ref Message m) {
            if (robot != null) {
                robot.HandleWindowsMessage(ref m);
            }
            base.WndProc(ref m);
        }

        /** Called after the form is closed */
        private void MainWindow_FormClosed(object sender, FormClosedEventArgs e) {
            if (updateTimer != null) { updateTimer.Dispose(); updateTimer = null; }
            if (robot != null) { robot.Dispose(); robot = null; }
        }

        /** Gets the robot instance associated with the window */
        public Robot Robot {
            get { return robot; }
        }

        /** Updates the status control's paused / connected state */
        private void UpdateStatusControl() {
            statusControl.Connected = robot.Connected;
            statusControl.Paused = robot.Paused;
        }

        /** Called when the robot's connection status changes */
        private void robot_ConnectionStatusChanged(object sender, EventArgs e) {
            UpdateStatusControl();
        }

        /** Called when the robot's target position changes */
        private void robot_TargetPositionChanged(object sender, EventArgs e) {
            if (preventTransformerUpdate == 0) {    // (are we allowed to update the transformer?)
                preventRobotUpdate ++;              // (prevent bounce back to the robot)
                try {
                    targetPositionTransformer.Position = robot.TargetPosition;
                }
                finally {
                    preventRobotUpdate --;
                }
            }
        }

        /** Called when the robot's target speed changes */
        private void robot_SpeedChanged(object sender, EventArgs e) {
            if (preventTransformerUpdate == 0) {    // (are we allowed to update the transformer?)
                preventRobotUpdate++;               // (prevent bounce back to the robot)
                try {
                    speedTransformer.Speed = robot.Speed;
                }
                finally {
                    preventRobotUpdate--;
                }
            }
        }

        /** Validates all fields that should contain a double value */
        private void TargetField_Validating(object sender, CancelEventArgs e) {
            TextBox textBox = (TextBox)sender;
            string text = textBox.Text;
            double result;
            if (text == null || !Double.TryParse(text, out result)) {
                MessageBox.Show("Please enter a valid number");
                e.Cancel = true;
            }
        }

        /** Called when the text of one of the target fields changes */
        private void TargetField_Validated(object sender, EventArgs e) {
            if (preventTransformerUpdate == 0) {    // (are we allowed to update the transformer?)
                TextBox box = (TextBox)sender;
                int component = Int32.Parse((string)box.Tag);
                double value;
                if (double.TryParse(box.Text, out value)) {
                    preventTextUpdate ++;           // (prevent bounce back to the text field)
                    try {
                        targetPositionTransformer.SetComponent(component, value);
                    }
                    finally {
                        preventTextUpdate --;
                    }
                }
                else {
                    MessageBox.Show("Please enter a valid number");
                }
            }
        }

        /** Called when a key is pressed on one of the target fields */
        private void TargetField_KeyDown(object sender, KeyEventArgs e) {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return) {

                // Fake validation, so the value is accepted
                CancelEventArgs validateArgs = new CancelEventArgs();
                TargetField_Validating(sender, validateArgs);
                if (!validateArgs.Cancel) {
                    TargetField_Validated(sender, EventArgs.Empty);
                }
                ((TextBox)sender).SelectAll();

                e.SuppressKeyPress = true;  // (prevent the beeping)
            }
        }

        /** Called when a key is pressed on one of the speed fields */
        private void SpeedField_KeyDown(object sender, KeyEventArgs e) {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return) {
                e.SuppressKeyPress = true;  // (prevent the beeping)
                // (Acceptance of the value already occurs on enter key, so no special handling is needed)
            }
        }

        /** Called when the text of one of the speed fields changes */
        private void SpeedField_ValueChanged(object sender, EventArgs e) {
            if (preventTransformerUpdate == 0) {    // (are we allowed to update the transformer?)
                NumericUpDown box = (NumericUpDown)sender;
                int component = Int32.Parse((string)box.Tag);
                double value;
                if (double.TryParse(box.Text, out value)) {
                    preventTextUpdate ++;           // (prevent bounce back to the text field)
                    try {
                        speedTransformer.SetComponent(component, value);
                    }
                    finally {
                        preventTextUpdate --;
                    }
                }
            }
        }

        /** Called when the position update timer fires */
        private void updateTimer_Tick(object sender, EventArgs e) {
            
            // Update the robot's position from the device.
            robot.UpdateFromDevice();

            // Update the displayed position.
            currentPositionTransformer.Position = robot.CurrentPosition;
        }

        /** Called when the current position (or its unit) changes */
        private void currentPositionTransformer_Changed(object sender, EventArgs e) {
            currentPosLabel1.Text = String.Format("{0:0.##}\n{1}", currentPositionTransformer.Component1, GetUnitString(currentPositionTransformer.Unit1));
            currentPosLabel2.Text = String.Format("{0:0.##}\n{1}", currentPositionTransformer.Component2, GetUnitString(currentPositionTransformer.Unit2));
            currentPosLabel3.Text = String.Format("{0:0.##}\n{1}", currentPositionTransformer.Component3, GetUnitString(currentPositionTransformer.Unit3));
            currentPosLabel4.Text = String.Format("{0:0.##}\n{1}", currentPositionTransformer.Component4, GetUnitString(currentPositionTransformer.Unit4));
        }

        /** Called when the target position transformer's value changes */
        private void targetPositionTransformer_PositionChanged(object sender, EventArgs e) {
            if (preventRobotUpdate == 0) {      // (are we allowed to update the robot?)
                preventTransformerUpdate ++;      // (prevent bounce back to the transformer)
                try {
                    robot.TargetPosition = targetPositionTransformer.Position;
                }
                finally {
                    preventTransformerUpdate --;
                }
            }
        }

        /** Called when one of the target position transformer's components changes */
        private void targetPositionTransformer_ComponentChanged(object sender, EventArgs e) {
            if (preventTextUpdate == 0) {       // (are we allowed to update the text field?)
                preventTransformerUpdate ++;    // (prevent bounce back to the transformer)
                try {
                    targetPosField1.Text = String.Format("{0:0.##}", targetPositionTransformer.Component1);
                    targetPosField2.Text = String.Format("{0:0.##}", targetPositionTransformer.Component2);
                    targetPosField3.Text = String.Format("{0:0.##}", targetPositionTransformer.Component3);
                    targetPosField4.Text = String.Format("{0:0.##}", targetPositionTransformer.Component4);
                }
                finally {
                    preventTransformerUpdate --;
                }
            }
        }

        /** Called when the target position transformer's unit changes */
        private void targetPositionTransformer_UnitChanged(object sender, EventArgs e) {
            targetPosUnitLabel1.Text = GetUnitString(targetPositionTransformer.Unit1);
            targetPosUnitLabel2.Text = GetUnitString(targetPositionTransformer.Unit2);
            targetPosUnitLabel3.Text = GetUnitString(targetPositionTransformer.Unit3);
            targetPosUnitLabel4.Text = GetUnitString(targetPositionTransformer.Unit4);
        }

        /** Called when the speed transformer's value changes */
        private void speedTransformer_SpeedChanged(object sender, EventArgs e) {
            if (preventRobotUpdate == 0) {      // (are we allowed to update the robot?)
                preventTransformerUpdate ++;    // (prevent bounce back to the transformer)
                try {
                    robot.Speed = speedTransformer.Speed;
                }
                finally {
                    preventTransformerUpdate --;
                }
            }
        }

        /** Called when one of the speed transformer's components changes */
        private void speedTransformer_ComponentChanged(object sender, EventArgs e) {
            if (preventTextUpdate == 0) {
                preventTransformerUpdate ++;
                try {
                    speedField1.Value = new Decimal(speedTransformer.Component1);
                    speedField2.Value = new Decimal(speedTransformer.Component2);
                    speedField3.Value = new Decimal(speedTransformer.Component3);
                    speedField4.Value = new Decimal(speedTransformer.Component4);
                }
                finally {
                    preventTransformerUpdate --;
                }
            }
        }

        /** Called when the speed transformer's unit changes */
        private void speedTransformer_UnitChanged(object sender, EventArgs e) {
            speedUnitLabel1.Text = GetUnitString(speedTransformer.Unit1);
            speedUnitLabel2.Text = GetUnitString(speedTransformer.Unit2);
            speedUnitLabel3.Text = GetUnitString(speedTransformer.Unit3);
            speedUnitLabel4.Text = GetUnitString(speedTransformer.Unit4);
        }

        /** Returns the given unit as a lowercase string */
        private static string GetUnitString(Unit unit) {
            switch (unit) {
                case Unit.Degrees:
                    return "degrees";
                case Unit.Radians:
                    return "radians";
                case Unit.Steps:
                    return "steps";
                default:
                    return "???";
            }
        }

        /** Called when the checked state of one of the unit buttons changes */
        private void UnitButton_CheckedChanged(object sender, EventArgs e) {
            Unit newUnit = Unit.Unknown;
            if (degreesButton.Checked) {
                newUnit = Unit.Degrees;
            }
            else if (radiansButton.Checked) {
                newUnit = Unit.Radians;
            }
            else if (stepsButton.Checked) {
                newUnit = Unit.Steps;
            }
            currentPositionTransformer.DofUnit = newUnit;
            targetPositionTransformer.DofUnit = newUnit;
            speedTransformer.DofUnit = newUnit;
            ApplyUnitToProgram();
        }

        /** Called whenever one of the spinner buttons ticks */
        private void SpinnerButton_Tick(object sender, EventArgs e) {
            BigSpinnerButton button = (BigSpinnerButton)sender;
            int componentIndex = Int32.Parse((string)button.Tag);   // (0 through 3)
            double value = targetPositionTransformer.Position.GetComponent(componentIndex, CoordinateSystem.Dof, robot);
            value += (button.Direction ? -1.0 : 1.0) * SpinnerRate * Math.PI / 180.0;
            targetPositionTransformer.Position = targetPositionTransformer.Position.SetComponent(componentIndex, value, CoordinateSystem.Dof, robot);
        }

        /** Called when the pause state of the status control changes */
        private void statusControl_PausedChanged(object sender, EventArgs e) {
            robot.Paused = statusControl.Paused;
        }

        /** Called when the "Set Home Position" button is clicked */
        private void setHomeButton_Click(object sender, EventArgs e) {
            robot.SetHomePosition();
            targetPositionTransformer.Position = ArmPosition.Zero;
        }

        /** Called when the "Edit > Undo" menu item is clicked */
        private void undoMenuItem_Click(object sender, EventArgs e) {
            savedProgram.UndoManager.Undo();
        }

        /** Called when the "Edit > Redo" menu item is clicked */
        private void redoMenuItem_Click(object sender, EventArgs e) {
            savedProgram.UndoManager.Redo();
        }

        /** Called when the undo / redo state changes */
        private void savedProgram_UndoStateChanged(object sender, EventArgs e) {
            string nextUndo = savedProgram.UndoManager.NextUndoName;
            string nextRedo = savedProgram.UndoManager.NextRedoName;
            
            if (nextUndo != null && robot.Path == null) {
                undoMenuItem.Enabled = true;
                undoMenuItem.Text = "Undo " + nextUndo;
            }
            else {
                undoMenuItem.Enabled = false;
                undoMenuItem.Text = "Undo";
            }
            
            if (nextRedo != null && robot.Path == null) {
                redoMenuItem.Enabled = true;
                redoMenuItem.Text = "Redo " + nextRedo;
            }
            else {
                redoMenuItem.Enabled = false;
                redoMenuItem.Text = "Redo";
            }
        }

        /** Sets a new program */
        public void SetProgram(SavedProgram program) {
            if (program == null) { throw new ArgumentNullException("program"); }
            if (savedProgram != null) {
                
                savedProgram.EntryInserted -= savedProgram_EntryInserted;
                savedProgram.EntryRemoved -= savedProgram_EntryRemoved;
                savedProgram.UndoManager.StateChanged -= savedProgram_UndoStateChanged;
                while (grid.Rows.Count > 0) {
                    RemoveRowFromGrid(grid.Rows[0]);
                }

            }

            savedProgram = program;
            ApplyUnitToProgram();

            savedProgram.EntryInserted += savedProgram_EntryInserted;
            savedProgram.EntryRemoved += savedProgram_EntryRemoved;
            savedProgram.UndoManager.StateChanged += savedProgram_UndoStateChanged;
            IList<ProgramEntry> entries = savedProgram.Entries;
            for (int i = 0; i < entries.Count; i++) {
                AddEntryToGrid(i, entries[i]);
            }

            // Immediately update the undo menu items
            savedProgram_UndoStateChanged(this, EventArgs.Empty);
        }

        /** Applies the current unit to the current saved program */
        private void ApplyUnitToProgram() {
            if (savedProgram != null) {
                foreach (ProgramEntry entry in savedProgram.Entries) {
                    ApplyUnitToEntry(entry);
                }
            }
        }

        /** Applies the current unit to the given program entry */
        private void ApplyUnitToEntry(ProgramEntry entry) {
            entry.PositionTransformer.DofUnit = currentPositionTransformer.DofUnit;
            entry.SpeedTransformer.DofUnit = speedTransformer.DofUnit != Unit.Steps ? speedTransformer.DofUnit : Unit.Degrees;
            DataGridViewRow row;
            if (rowMap.TryGetValue(entry, out row)) {
                grid.InvalidateRow(row.Index);  // (force a formatting refresh, in case value didn't change)
                UpdateRowValues(row);
            }
        }

        /** Called when a program entry is added */
        private void savedProgram_EntryInserted(object sender, InsertEventArgs e) {
            AddEntryToGrid(e.Index, e.Entry);
        }

        /** Called when a program entry is removed */
        private void savedProgram_EntryRemoved(object sender, RemoveEventArgs e) {
            RemoveRowFromGrid(grid.Rows[e.Index]);
        }

        /** Begins watching for changes to a program entry */
        private void AddEntryToGrid(int index, ProgramEntry entry) {
            DataGridViewRow row = new DataGridViewRow();
            row.CreateCells(grid);
            row.Tag = entry;
            UpdateRowValues(row);
            grid.Rows.Insert(index, row);
            rowMap.Add(entry, row);
            entry.ValueChanged += ProgramEntry_ValueChanged;
        }

        /** Stops watching for changes to a program entry */
        private void RemoveRowFromGrid(DataGridViewRow row) {
            ProgramEntry entry = (ProgramEntry)row.Tag;
            entry.ValueChanged -= ProgramEntry_ValueChanged;
            rowMap.Remove(entry);
            grid.Rows.Remove(row);
        }

        /** Updates the values of a row based on its corresponding entry */
        private void UpdateRowValues(DataGridViewRow row) {
            ProgramEntry entry = (ProgramEntry)row.Tag;
            if (preventTextUpdate == 0) {       // (are we allowed to update the text fields?)
                preventTransformerUpdate ++;    // (prevent bounce back to the transformer)
                try {
                    row.SetValues(
                        entry.PositionTransformer.Component1,
                        entry.PositionTransformer.Component2,
                        entry.PositionTransformer.Component3,
                        entry.PositionTransformer.Component4,
                        entry.SpeedTransformer.TransformedSpeed,
                        entry.Pause);
                }
                finally {
                    preventTransformerUpdate --;
                }
            }
        }

        /** Called when the "Add Point" button is clicked */
        private void addButton_Click(object sender, EventArgs e) {
            ProgramEntry entry = new ProgramEntry(savedProgram.UndoManager);
            robot.UpdateFromDevice();   // (get the very latest position)
            entry.PositionTransformer.Position = robot.CurrentPosition;
            ApplyUnitToEntry(entry);
            savedProgram.InsertProgramEntry(savedProgram.Entries.Count, entry);
        }

        /** Called when the "Remove Point" button is clicked */
        private void removeButton_Click(object sender, EventArgs e) {
            if (grid.SelectedRows.Count > 0) {
                List<int> selectedIndices = new List<int>();
                foreach (DataGridViewRow row in grid.SelectedRows) { selectedIndices.Add(row.Index); }
                selectedIndices.Sort();

                // Step backward through the selected rows
                savedProgram.UndoManager.BeginUndoGroup(selectedIndices.Count == 1 ? "Remove Row" : "Remove Rows");
                try {
                    for (int i = selectedIndices.Count - 1; i >= 0; i --) {
                        int index = selectedIndices[i];
                        savedProgram.RemoveProgramEntry(index);
                    }
                }
                finally {
                    savedProgram.UndoManager.EndUndoGroup();
                }
            }
        }

        /** Called when one of the values on a program entry is changed by the user */
        private void grid_CellValueChanged(object sender, DataGridViewCellEventArgs e) {
            if (preventTransformerUpdate == 0) {    // Are we allowed to update the transformer?
                preventTextUpdate ++;
                try {
                    if (e.RowIndex >= 0 && e.ColumnIndex >= 0) {
                        DataGridViewRow row = grid.Rows[e.RowIndex];
                        DataGridViewCell cell = row.Cells[e.ColumnIndex];
                        ProgramEntry entry = (ProgramEntry)row.Tag;
                        double? doubleValue = cell.Value as double?;
                        if (doubleValue.HasValue) {
                            if (e.ColumnIndex == M1Column.Index) {
                                entry.SetPositionComponent(0, doubleValue.Value);
                            }
                            else if (e.ColumnIndex == M2Column.Index) {
                                entry.SetPositionComponent(1, doubleValue.Value);
                            }
                            else if (e.ColumnIndex == M3Column.Index) {
                                entry.SetPositionComponent(2, doubleValue.Value);
                            }
                            else if (e.ColumnIndex == M4Column.Index) {
                                entry.SetPositionComponent(3, doubleValue.Value);
                            }
                            else if (e.ColumnIndex == SpeedColumn.Index) {
                                entry.SetSpeed(doubleValue.Value);
                            }
                            else if (e.ColumnIndex == PauseColumn.Index) {
                                entry.SetPause(doubleValue.Value);
                            }
                        }
                    }
                }
                finally {
                    preventTextUpdate --;
                }
            }
        }

        /** Called when one of the values on a program entry is changed */
        private void ProgramEntry_ValueChanged(object sender, EventArgs e) {
            ProgramEntry entry = (ProgramEntry)sender;
            DataGridViewRow row = rowMap[entry];
            UpdateRowValues(row);
        }

        /** Applies formatting to a cell */
        private void grid_CellFormatting(object sender, DataGridViewCellFormattingEventArgs e) {
            if (e.RowIndex >= 0 && e.ColumnIndex >= 0) {
                ProgramEntry entry = (ProgramEntry)grid.Rows[e.RowIndex].Tag;
                if (e.ColumnIndex == SpeedColumn.Index) {
                    e.Value = String.Format("{0:0.##} {1}", e.Value, GetSpeedAbbreviation(entry.SpeedTransformer.DofUnit));
                    e.FormattingApplied = true;
                }
                else if (e.ColumnIndex == PauseColumn.Index) {
                    e.Value = String.Format("{0:0.##} s", e.Value);
                    e.FormattingApplied = true;
                }
                else {
                    e.Value = String.Format("{0:0.##}", e.Value);
                    e.FormattingApplied = true;
                }
            }
        }

        /** Returns speed column abbreviation for the given unit */
        private string GetSpeedAbbreviation(Unit unit) {
            switch (unit) {
                case Unit.Degrees:
                    return "deg/s";
                case Unit.Radians:
                    return "rad/s";
                default:
                    throw new InvalidOperationException(String.Format("Unsupported unit: {0}", unit));
            }
        }

        /** Parses the formatting out of a cell's value */
        private void grid_CellParsing(object sender, DataGridViewCellParsingEventArgs e) {
            Match m = Regex.Match(e.Value.ToString(), "^[-+]?[0-9.]+");
            e.Value = String.IsNullOrEmpty(m.Value) ? 0.0 : double.Parse(m.Value);
            e.ParsingApplied = true;
        }

        /** Called when the user clicks File > Exit */
        private void exitMenuItem_Click(object sender, EventArgs e) {
            Close();
        }

        /** Called when the user clicks File > New */
        private void newProgramMenuItem_Click(object sender, EventArgs e) {
            savePath = null;
            SetProgram(new SavedProgram());
        }

        /** Called when the user clicks File > Save */
        private void saveProgramMenuItem_Click(object sender, EventArgs e) {
            if (savePath != null) {
                savedProgram.Save(savePath);
            }
            else {
                saveProgramAsMenuItem_Click(this, EventArgs.Empty);
            }
        }

        /** Called when the user clicks File > Save As */
        private void saveProgramAsMenuItem_Click(object sender, EventArgs e) {
            SaveFileDialog dlg = new SaveFileDialog();
            dlg.InitialDirectory = System.IO.Path.GetDirectoryName(savePath);
            dlg.FileName = System.IO.Path.GetFileName(savePath);
            dlg.Filter = "CSV Files (*.csv)|*.csv|All Files (*.*)|*.*";
            if (DialogResult.OK == dlg.ShowDialog(this)) {
                savePath = dlg.FileName;
                savedProgram.Save(savePath);
            }
        }

        /** Called when the user clicks File > Open */
        private void openProgramMenuItem_Click(object sender, EventArgs e) {
            OpenFileDialog dlg = new OpenFileDialog();
            dlg.InitialDirectory = System.IO.Path.GetDirectoryName(savePath);
            dlg.Filter = "CSV Files (*.csv)|*.csv|All Files (*.*)|*.*";
            if (DialogResult.OK == dlg.ShowDialog(this)) {
                SavedProgram newProgram = SavedProgram.Open(dlg.FileName);
                if (newProgram != null) {
                    savePath = dlg.FileName;
                    SetProgram(newProgram);
                }
            }
        }

        /** Called before the form is closed, to give the user a chance to save
         *  their changes.
         */
        private void MainWindow_FormClosing(object sender, FormClosingEventArgs e) {
            if (savedProgram != null && savedProgram.UndoManager.NextUndoName != null) {
                DialogResult r = MessageBox.Show("Would you like to save your changes to the program before exiting?",
                    "Save?", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
                if (r == DialogResult.Cancel) { e.Cancel = true; }
                if (r == DialogResult.Yes) {
                    saveProgramMenuItem_Click(this, EventArgs.Empty);
                    if (savePath == null) { e.Cancel = true; }  // (user canceled the save as)
                }
            }
        }

        /** Called when the user clicks the "Run Program" button */
        private void runProgramButton_Click(object sender, EventArgs e) {
            if (robot.Path == null) {
                // Select the first item while we move to the start position
                grid.ClearSelection();
                if (grid.Rows.Count > 0) { grid.Rows[0].Selected = true; }

                // Save the speed just before running the program
                speedBeforeRunningProgram = robot.Speed;

                // Compile the program, and tell the robot to run it.
                if (compiledProgram != null) {
                    compiledProgram.LastExecutedEntryIndexChanged -= compiledProgram_LastExecutedEntryIndexChanged;
                }
                compiledProgram = new CompiledProgram(savedProgram);
                compiledProgram.LastExecutedEntryIndexChanged += compiledProgram_LastExecutedEntryIndexChanged;
                compiledProgram.Loop = loopBox.Checked;
                robot.Path = compiledProgram;
            }
            else {
                // Stop the currently running program
                robot.Path = null;
            }
        }

        /** Called when the robot's path changes */
        private void robot_PathChanged(object sender, EventArgs e) {

            // Disable the position / speed controls if running a program
            UpdateControlEnableStates();
            runProgramButton.Text = robot.Path == null ? "Run Program" : "Stop Program";

            // When stopping a program, set the target position to the current
            // position, and the speed to whatever it was before the program
            // was started.
            if (robot.Path == null) {
                Debug.Assert(speedBeforeRunningProgram != null);
                robot.UpdateFromDevice();   // (get the very latest position)
                robot.TargetPosition = robot.CurrentPosition;
                if (speedBeforeRunningProgram != null) { robot.Speed = speedBeforeRunningProgram; }
                grid.ClearSelection();
            }
        }

        /** Called when a new program step is executed */
        private void compiledProgram_LastExecutedEntryIndexChanged(object sender, EventArgs e) {
            if (compiledProgram != null) {
                int i = compiledProgram.LastExecutedEntryIndex;
                if (i >= 0 && i < grid.Rows.Count) {
                    grid.ClearSelection();
                    grid.Rows[i].Selected = true;
                }
            }
        }

        /** Enables or disables the controls on the form */
        private void UpdateControlEnableStates() {
            bool value = robot.Path == null;
            upButton1.Enabled = value;
            upButton2.Enabled = value;
            upButton3.Enabled = value;
            upButton4.Enabled = value;
            downButton1.Enabled = value;
            downButton2.Enabled = value;
            downButton3.Enabled = value;
            downButton4.Enabled = value;
            targetPosField1.Enabled = value;
            targetPosField2.Enabled = value;
            targetPosField3.Enabled = value;
            targetPosField4.Enabled = value;
            speedField1.Enabled = value;
            speedField2.Enabled = value;
            speedField3.Enabled = value;
            speedField4.Enabled = value;
            setHomeButton.Enabled = value;
            saveButton.Enabled = value;
            loadButton.Enabled = value;
            addButton.Enabled = value;
            removeButton.Enabled = value;

            newProgramMenuItem.Enabled = value;
            openProgramMenuItem.Enabled = value;
            saveProgramMenuItem.Enabled = value;
            saveProgramAsMenuItem.Enabled = value;

            savedProgram_UndoStateChanged(this, EventArgs.Empty);
        }

        /** Called when the user clicks the hyperlink */
        private void linkLabel1_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e) {
            Process.Start("http://www.thecardboardrobot.com");
        }

        /** Called when the user picks "Move Up" from the context menu */
        private void contextMenu_MoveUp(object sender, EventArgs e) {
            if (grid.SelectedRows.Count > 0) {
                List<int> selectedIndices = new List<int>();
                foreach (DataGridViewRow row in grid.SelectedRows) { selectedIndices.Add(row.Index); }
                selectedIndices.Sort();

                if (!grid.Rows[0].Selected) {   // (can't move up if first one selected)
                    savedProgram.UndoManager.BeginUndoGroup("Move Up");
                    try {
                        for (int i = 0; i < selectedIndices.Count; i ++) {
                            int selectedIndex = selectedIndices[i];
                            ProgramEntry entry = savedProgram.Entries[selectedIndex];
                            savedProgram.RemoveProgramEntry(selectedIndex);
                            savedProgram.InsertProgramEntry(selectedIndex - 1, entry);
                            
                        }
                        grid.ClearSelection();
                        for (int i = 0; i < selectedIndices.Count; i ++) {
                            grid.Rows[selectedIndices[i] - 1].Selected = true;
                        }
                    }
                    finally {
                        savedProgram.UndoManager.EndUndoGroup();
                    }
                }
            }
        }

        /** Called when the user picks "Move Down" from the context menu */
        private void contextMenu_MoveDown(object sender, EventArgs e) {
            if (grid.SelectedRows.Count > 0) {
                List<int> selectedIndices = new List<int>();
                foreach (DataGridViewRow row in grid.SelectedRows) { selectedIndices.Add(row.Index); }
                selectedIndices.Sort();

                if (!grid.Rows[grid.Rows.Count - 1].Selected) {   // (can't move down if last one selected)
                    savedProgram.UndoManager.BeginUndoGroup("Move Down");
                    try {
                        for (int i = selectedIndices.Count - 1; i >= 0; i--) {
                            int selectedIndex = selectedIndices[i];
                            ProgramEntry entry = savedProgram.Entries[selectedIndex];
                            savedProgram.RemoveProgramEntry(selectedIndex);
                            savedProgram.InsertProgramEntry(selectedIndex + 1, entry);

                        }
                        grid.ClearSelection();
                        for (int i = 0; i < selectedIndices.Count; i++) {
                            grid.Rows[selectedIndices[i] + 1].Selected = true;
                        }
                    }
                    finally {
                        savedProgram.UndoManager.EndUndoGroup();
                    }
                }
            }
        }

        /** Called when the user picks "Remove" from the context menu */
        private void contextMenu_Remove(object sender, EventArgs e) {
            removeButton_Click(this, EventArgs.Empty);
        }

        /** Called when the user picks "Go to this Position" from the context menu */
        private void contextMenu_GoToPosition(object sender, EventArgs e) {
            if (grid.SelectedRows.Count == 1) {
                ProgramEntry entry = (ProgramEntry)grid.SelectedRows[0].Tag;
                robot.TargetPosition = entry.PositionTransformer.Position;
            }
        }

        /** Called when the user lifts the mouse anywhere on the grid */
        private void grid_MouseUp(object sender, MouseEventArgs e) {
            if (e.Button == MouseButtons.Right && robot.Path == null) {
                DataGridView.HitTestInfo hitTest = grid.HitTest(e.X, e.Y);
                if (hitTest.Type == DataGridViewHitTestType.Cell) {
                    // Select the row that was clicked on
                    DataGridViewRow row = grid.Rows[hitTest.RowIndex];
                    if (!row.Selected) {
                        grid.ClearSelection();
                        row.Selected = true;
                    }

                    // Generate and show the context menu
                    GenerateContextMenu();
                    contextMenu.Show(grid, e.Location);
                }
            }
        }

        /** Updates the contextMenu member as appropriate */
        private void GenerateContextMenu() {
            contextMenu.Items.Clear();
            goToMenuItem.Enabled = grid.SelectedRows.Count == 1;
            contextMenu.Items.Add(goToMenuItem);
            contextMenu.Items.Add(new ToolStripSeparator());
            contextMenu.Items.Add(moveUpMenuItem);
            contextMenu.Items.Add(moveDownMenuItem);
            contextMenu.Items.Add(new ToolStripSeparator());
            contextMenu.Items.Add(removeMenuItem);
        }

        private void loopBox_Click(object sender, EventArgs e) {
            if (compiledProgram != null) {
                compiledProgram.Loop = loopBox.Checked;
            }
        }
    }
}
