﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Windows.Forms;
using CBRobot;

namespace CBRobotConsole {

    public class InsertEventArgs : EventArgs {
        public int Index;
        public ProgramEntry Entry;
    }

    public class RemoveEventArgs : EventArgs {
        public int Index;
    }

    /** Represents an entire saved program within the CBRobotConsole application */
    public class SavedProgram {
        private UndoManager undoManager;
        private List<ProgramEntry> entries;

        public SavedProgram() {
            undoManager = new UndoManager();
            entries = new List<ProgramEntry>();
        }

        /** Returns the undo manager for this document */
        public UndoManager UndoManager {
            get { return undoManager; }
        }

        /** Gets the entries of this program as a read-only collection */
        public ReadOnlyCollection<ProgramEntry> Entries {
            get { return entries.AsReadOnly(); }
        }
        
        /** Fired when an entry is added to the program */
        public event EventHandler<InsertEventArgs> EntryInserted;
        
        /** Fires the EntryInserted event */
        private void OnEntryInserted(int index, ProgramEntry entry) {
            if (EntryInserted != null) {
                EntryInserted(this, new InsertEventArgs() {
                    Index = index, Entry = entry} );
            }
        }

        /** Fired when an entry is removed from the program */
        public event EventHandler<RemoveEventArgs> EntryRemoved;
        
        /** Fires the EntryRemoved event */
        private void OnEntryRemoved(int index) {
            if (EntryRemoved != null) {
                EntryRemoved(this, new RemoveEventArgs() { Index = index });
            }
        }

        /** Inserts an entry into the program at the specified index */
        public void InsertProgramEntry(int index, ProgramEntry entry) {
            InsertProgramEntryImpl(index, entry);
        }
        
        private void InsertProgramEntryImpl(object arg1, object arg2) {
            int index = (int)arg1;
            ProgramEntry entry = (ProgramEntry)arg2;

            entries.Insert(index, entry);
            undoManager.AddInverseOperation("Add Row",
                new InverseFunc(RemoveProgramEntryImpl), index, entry);
            
            OnEntryInserted(index, entry);
        }

        /** Removes the entry at the specified index from the program */
        public void RemoveProgramEntry(int index) {
            RemoveProgramEntryImpl(index, null);
        }

        private void RemoveProgramEntryImpl(object arg1, object arg2) {
            int index = (int)arg1;

            ProgramEntry entry = entries[index];
            entries.RemoveAt(index);
            undoManager.AddInverseOperation("Remove Row",
                new InverseFunc(InsertProgramEntryImpl), index, entry);

            OnEntryRemoved(index);
        }

        /** Saves the document to the given CSV file.  Displays an appropriate
         *  error message on failure.
         */
        public void Save(string path) {
            try {
                using (StreamWriter w = File.CreateText(path)) {
                    PositionTransformer pt = new PositionTransformer();
                    ScalarSpeedTransformer st = new ScalarSpeedTransformer();
                    pt.DofUnit = Unit.Degrees;
                    st.DofUnit = Unit.Degrees;

                    foreach (ProgramEntry entry in entries) {
                        pt.Position = entry.PositionTransformer.Position;
                        st.Speed = entry.SpeedTransformer.Speed;
                        string line = String.Format("{0:F6}, {1:F6}, {2:F6}, {3:F6}, {4:F6}, {5:F6}",
                            pt.Component1, pt.Component2, pt.Component3, pt.Component4,
                            st.TransformedSpeed, entry.Pause);
                        w.WriteLine(line);
                    }
                }
            }
            catch (Exception ex) {
                MessageBox.Show(String.Format("An error occurred while saving the file.\n\nError details:\n{0}", ex),
                    "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /** Reads the CSV file at the given path and returns a new program
         *  instance.  Displays an appropriate error message and returns null
         *  if the program failed to load.
         */
        public static SavedProgram Open(string path) {
            try {
                SavedProgram program = new SavedProgram();
                char[] separators = new char[] { ',' };
                double radsPerDeg = Math.PI / 180.0;
                double defaultSpeed = Robot.DefaultSpeed.M1Speed;

                using (StreamReader r = new StreamReader(File.OpenRead(path))) {
                    while (true) {
                        string line = r.ReadLine();
                        if (line == null) { break; }
                        if (line.Trim().Length == 0) { continue; }  // (skip blank lines)

                        string[] parts = line.Split(separators);
                        double m1 = 0.0, m2 = 0.0, m3 = 0.0, m4 = 0.0;
                        double speed = defaultSpeed, pause = 0.0;
                        if (parts.Length >= 1) { Double.TryParse(parts[0].Trim(), out m1); }
                        if (parts.Length >= 2) { Double.TryParse(parts[1].Trim(), out m2); }
                        if (parts.Length >= 3) { Double.TryParse(parts[2].Trim(), out m3); }
                        if (parts.Length >= 4) { Double.TryParse(parts[3].Trim(), out m4); }
                        if (parts.Length >= 5) { Double.TryParse(parts[4].Trim(), out speed); }
                        if (parts.Length >= 6) { Double.TryParse(parts[5].Trim(), out pause); }
                        
                        ProgramEntry entry = new ProgramEntry(program.UndoManager);
                        
                        entry.SpeedTransformer.Speed = speed * radsPerDeg;

                        entry.PositionTransformer.Position = new ArmPosition(new DofVector(
                            m1 * radsPerDeg, m2 * radsPerDeg, m3 * radsPerDeg), m4 * radsPerDeg);

                        entry.Pause = pause;

                        program.InsertProgramEntry(program.Entries.Count, entry);
                    }
                }

                // Don't undo past a load / revert operation
                program.UndoManager.Reset();

                return program;
            }
            catch (Exception ex) {
                MessageBox.Show(String.Format("An error occurred while reading the file.\n\nError details:\n{0}", ex),
                    "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return null;
            }
        }
    }
}
