﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;

namespace CBRobotConsole {

    /** Converts a speed value (measured in radians / sec) into various units */
    public class ScalarSpeedTransformer {

        private Unit dofUnit;
        private double speed;
        private double transformedSpeed;

        public ScalarSpeedTransformer() {
            dofUnit = Unit.Radians;
        }

        /** Gets or sets the unit for DOF components */
        public Unit DofUnit {
            get { return dofUnit; }
            set {
                if (dofUnit != value) {
                    dofUnit = value;
                    UpdateTransformedSpeed();
                    OnUnitChanged();
                    OnTransformedSpeedChanged();
                }
            }
        }

        /** Gets or sets the un-transformed speed value */
        public double Speed {
            get { return speed; }
            set {
                if (speed != value) {
                    speed = value;
                    UpdateTransformedSpeed();
                    OnTransformedSpeedChanged();
                }
            }
        }

        /** Gets or sets the transformed speed value (updates the original) */
        public double TransformedSpeed {
            get { return transformedSpeed; }
            set {
                if (transformedSpeed != value) {
                    transformedSpeed = value;
                    speed = value / GetDofConversionFactor();
                    OnSpeedChanged();
                }
            }
        }

        /** Updates the transformed speed value */
        private void UpdateTransformedSpeed() {
            transformedSpeed = speed * GetDofConversionFactor();
        }

        /** Returns the conversion factor (radians to target unit) */
        private double GetDofConversionFactor() {
            switch (dofUnit) {
                case Unit.Radians:
                    return 1.0;
                case Unit.Degrees:
                    return 180.0 / Math.PI;
                default:
                    throw new InvalidOperationException(String.Format("Unknown unit: {0}", dofUnit));
            }
        }

        /** Fired when the Speed property changes */
        public event EventHandler SpeedChanged;
        
        /** Fires the SpeedChanged event */
        private void OnSpeedChanged() {
            if (SpeedChanged != null) {
                SpeedChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the TransformedSpeed property changes */
        public event EventHandler TransformedSpeedChanged;
        
        /** Fires the TransformedSpeedChanged event */
        private void OnTransformedSpeedChanged() {
            if (TransformedSpeedChanged != null) {
                TransformedSpeedChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when a unit change is made */
        public event EventHandler UnitChanged;
        
        /** Fires the UnitChanged event */
        private void OnUnitChanged() {
            if (UnitChanged != null) {
                UnitChanged(this, EventArgs.Empty);
            }
        }
    }
}
