﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Collections.Generic;

namespace CBRobotConsole {

    public delegate void InverseFunc(object arg1, object arg2);

    /** This is a simple undo / redo stack manager.  Whenever an undoable
     *  operation is performed, call AddInverseOperation() to add the
     *  appropriate inverse operation to the undo / redo stack.
     */
    public class UndoManager {
        private Stack<UndoOperation> undoStack;     /* Undo stack */
        private Stack<UndoOperation> redoStack;     /* Redo stack */
        private bool isUndoing;                     /* Are we undoing an operation? */
        private bool isRedoing;                     /* Are we redoing an operation? */
        private string undoName;                    /* Name of the operation we're undoing */
        private UndoGroup undoGroup;                /* Current undo group, null if no group is open */

        private class UndoOperation {
            public string Name;         /* Name of what we're undoing */
            public InverseFunc Func;    /* Function to perform the undo */
            public object Arg1;         /* First argument passed to the function */
            public object Arg2;         /* Second argument passed to the function */
        }

        private class UndoGroup {
            public string Name;                         /* Name of the operation this undo group is for */
            public Stack<UndoOperation> UndoStack;      /* Undo stack within the undo group */
        }

        public UndoManager() {
            undoStack = new Stack<UndoOperation>();
            redoStack = new Stack<UndoOperation>();
        }

        /** Empties the undo / redo stack */
        public void Reset() {
            undoStack.Clear();
            redoStack.Clear();
            isUndoing = false;
            isRedoing = false;
            undoName = null;
            undoGroup = null;
            OnStateChanged();
        }

        /** Undoes one operation on the undo stack.  Does nothing if the undo
         *  stack is empty.
         */
        public void Undo() {
            if (isUndoing || isRedoing) { throw new InvalidOperationException("Undo within an undo / redo is unsupported"); }
            if (undoGroup != null) { throw new InvalidOperationException("Cannot undo while an undo group is open"); }

            if (undoStack.Count > 0) {
                isUndoing = true;
                try {
                    UndoOperation op = undoStack.Pop();
                    undoName = op.Name;
                    op.Func(op.Arg1, op.Arg2);
                }
                finally {
                    isUndoing = false;
                }
                OnStateChanged();
            }
        }

        /** Redoes one operation on the redo stack.  Does nothing if the redo
         *  stack is empty.
         */
        public void Redo() {
            if (isUndoing || isRedoing) { throw new InvalidOperationException("Redo within an undo / redo is unsupported"); }
            if (undoGroup != null) { throw new InvalidOperationException("Cannot redo while an undo group is open"); }

            if (redoStack.Count > 0) {
                isRedoing = true;
                try {
                    UndoOperation op = redoStack.Pop();
                    op.Func(op.Arg1, op.Arg2);
                }
                finally {
                    isRedoing = false;
                }
                OnStateChanged();
            }
        }

        /** Adds the specified operation to the undo stack, unless an undo
         *  is currently in progress, in which case it is added to the redo
         *  stack instead.
         */
        public void AddInverseOperation(string name, InverseFunc func, object arg1, object arg2) {
            UndoOperation op = new UndoOperation();
            op.Name = name;
            op.Func = func;
            op.Arg1 = arg1;
            op.Arg2 = arg2;

            // Special functionality if an undo group is open
            if (undoGroup != null) {
                undoGroup.UndoStack.Push(op);
            }
            else {
                // Push to either the undo stack or the redo stack
                if (isUndoing) {
                    op.Name = undoName;     // (use the name of the undone operation instead)
                    redoStack.Push(op);
                }
                else {
                    // The redo stack must be cleared anytime a new operation is
                    // added to the undo stack (unless, of course, we're redoing).
                    if (!isRedoing) {
                        redoStack.Clear();
                    }

                    undoStack.Push(op);
                }
                OnStateChanged();
            }
        }

        /** Returns the name of the next operation to be undone, or null if the
         *  undo stack is empty.
         */
        public string NextUndoName {
            get {
                if (undoStack.Count == 0) { return null; }
                return undoStack.Peek().Name;
            }
        }

        /** Returns the name of the next operation to be redone, or null if the
         *  redo stack is empty.
         */
        public string NextRedoName {
            get {
                if (redoStack.Count == 0) { return null; }
                return redoStack.Peek().Name;
            }
        }

        /** Called when the undo / redo state changes */
        public event EventHandler StateChanged;
        
        /** Fires the StateChanged event */
        private void OnStateChanged() {
            if (StateChanged != null) {
                StateChanged(this, EventArgs.Empty);
            }
        }

        /** Begins a new undo group with the given name */
        public void BeginUndoGroup(string name) {
            if (name == null) { throw new ArgumentNullException("name"); }
            if (undoGroup != null) { throw new InvalidOperationException("Nested undo groups are unsupported"); }

            undoGroup = new UndoGroup() {
                Name = name,
                UndoStack = new Stack<UndoOperation>()
            };
        }

        /** Ends the current undo group */
        public void EndUndoGroup() {
            if (undoGroup == null) { throw new InvalidOperationException("No undo group was opened"); }

            UndoGroup g = undoGroup;
            undoGroup = null;   // (do this BEFORE calling AddInverseOperation)
            AddInverseOperation(g.Name, UndoUndoGroup, g, null);
        }

        /** Function to undo an undo group */
        public void UndoUndoGroup(object arg1, object arg2) {
            UndoGroup g = (UndoGroup)arg1;
            BeginUndoGroup(g.Name);
            while (g.UndoStack.Count > 0) {
                UndoOperation op = g.UndoStack.Pop();
                op.Func(op.Arg1, op.Arg2);
            }
            EndUndoGroup();
        }
    }
}
