﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Drawing;
using System.Windows.Forms;

namespace CBRobot.StatusControl {

    /** A status control that can be used to display the robot's connection
     *  state (connected + running, connected + paused, disconnected).
     */
    public partial class StatusControl : UserControl {

        private bool connected = false;
        private bool paused = true;
        private bool mouseOver = false;
        private bool mousePressed = false;

        /** Creates a new StatusControl */
        public StatusControl() {
            InitializeComponent();

            // Nullify the default button highlighting
            button.FlatAppearance.MouseOverBackColor = button.BackColor;
            button.FlatAppearance.MouseDownBackColor = button.BackColor;
            
            UpdateButtonAndText();
        }

        /** Gets or sets whether the button indicates that the robot is in a connected state */
        public bool Connected {
            get { return connected; }
            set {
                if (connected != value) {
                    connected = value;
                    UpdateButtonAndText();
                }
            }
        }

        /** Gets or sets whether the button indicates that the robot is in a paused state */
        public bool Paused {
            get { return paused; }
            set {
                if (paused != value) {
                    paused = value;
                    UpdateButtonAndText();
                    OnPausedChanged();
                }
            }
        }

        /** Fired when the Paused property changes */
        public event EventHandler PausedChanged;
        
        /** Fires the PausedChanged event */
        private void OnPausedChanged() {
            if (PausedChanged != null) {
                PausedChanged(this, EventArgs.Empty);
            }
        }

        /** Updates the state of the button to match the current connected / running states */
        private void UpdateButtonAndText() {
            if (connected) {
                button.Enabled = true;
                if (paused) {
                    button.Image = GetRedImage();
                    largeText.Text = "Paused";
                    smallText.Text = "Click to Start";
                }
                else {
                    button.Image = GetGreenImage();
                    largeText.Text = "Running";
                    smallText.Text = "Click to Stop";
                }
            }
            else {
                button.Enabled = false;
                button.Image = GetRedImage();
                largeText.Text = "Not Connected";
                smallText.Text = "Plug in the Robot!";
            }
        }

        /** Returns the green image corresponding to the mouseover / pressed state */
        private Bitmap GetGreenImage() {
            return mousePressed ? Properties.Resources.CircleButton_Green_Pressed :
                (mouseOver ? Properties.Resources.CircleButton_Green_Hover :
                Properties.Resources.CircleButton_Green_Normal);
        }

        /** Returns the red image corresponding to the mouseover / pressed state */
        private Bitmap GetRedImage() {
            return mousePressed ? Properties.Resources.CircleButton_Red_Pressed :
                (mouseOver ? Properties.Resources.CircleButton_Red_Hover :
                Properties.Resources.CircleButton_Red_Normal);
        }

        private void button_MouseEnter(object sender, EventArgs e) {
            mouseOver = true;
            UpdateButtonAndText();
        }

        private void button_MouseLeave(object sender, EventArgs e) {
            mouseOver = false;
            UpdateButtonAndText();
        }

        private void button_MouseDown(object sender, MouseEventArgs e) {
            mousePressed = true;
            UpdateButtonAndText();
        }

        private void button_MouseUp(object sender, MouseEventArgs e) {
            mousePressed = false;
            UpdateButtonAndText();
        }

        private void button_Click(object sender, EventArgs e) {
            if (Connected) {
                Paused = !Paused;
            }
        }
    }
}
