﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Threading;
using Microsoft.Win32.SafeHandles;

namespace CBRobot {

    /* The DeviceConnection class manages the connection to the robot at a very
     * low level.  It doesn't handle any auto-connect or auto-disconnect
     * features - just basic operations like TryConnect(), Disconnect(), and
     * read / write.
     */
    internal class DeviceConnection : IDisposable {

        private const string targetHardwareID = "Vid_04d8&Pid_FC8B";    // VID / PID to search for

        private SafeFileHandle writeHandle = null;  /* Write handle to the USB device, null if disconnected */
        private SafeFileHandle readHandle = null;   /* Read handle to the USB device, null if disconnected */
        private volatile bool threadCanceled = false;   /* Has the read / write thread been canceled? */

        private DataOut dataOut;                    /* Data to write / last written */
        private DataIn dataIn;                      /* Data last read */
        private bool resetHomePositionFlag;         /* Reset the home position on next write? */

        private ManualResetEvent hasDataEvent;      /* Signals that data is available for writing */
        private ManualResetEvent dataSentEvent;     /* Signals that the read / write thread has completed a send */

        private object dataMutex;                   /* Data mutex (lock to change / prevent changes to data) */
        private object connectionMutex;             /* Connection mutex (lock to change / prevent connection status changes) */

        /* Data sent to the device */
        private struct DataOut {
            public short m1Target, m2Target, m3Target, m4Target;
            public short m1Speed, m2Speed, m3Speed, m4Speed;
            public bool paused;
        }

        /* Data read from the device */
        private struct DataIn {
            public short m1Current, m2Current, m3Current, m4Current;
        }

        #region Safe Handles
        /* Safe (auto-releasing) HDEVINFO handle */
        private class DevInfoSafeHandle : SafeHandleZeroOrMinusOneIsInvalid {
            public DevInfoSafeHandle() : base(true) { }
            protected override bool ReleaseHandle() {
                return SetupDiDestroyDeviceInfoList(handle);
            }
        }

        /* Safe (auto-releasing) HGlobal handle */
        private class HGlobalSafeHandle : SafeHandleZeroOrMinusOneIsInvalid {
            public HGlobalSafeHandle(IntPtr p) : base(true) { SetHandle(p); }
            protected override bool ReleaseHandle() {
                Marshal.FreeHGlobal(handle);
                return true;
            }
            public static implicit operator IntPtr(HGlobalSafeHandle h) {
                return h.handle;
            }
        }
        #endregion Safe Handles

        #region Imports
        /* ---- Imported functions ---- */
        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern DevInfoSafeHandle SetupDiGetClassDevs(
            out Guid ClassGuid,     /* [in,opt] const GUID* */
            IntPtr Enumerator,      /* [in,opt] PCTSTR */
            IntPtr HwndParent,      /* [in,opt] HWND */
            uint Flags);            /* [in] DWORD */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiDestroyDeviceInfoList(
            IntPtr DeviceInfoSet);  /* [in] HDEVINFO */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiEnumDeviceInfo(
            DevInfoSafeHandle DeviceInfoSet,        /* [in] HDEVINFO */
            uint MemberIndex,                       /* [in] DWORD */
            ref SP_DEVINFO_DATA DeviceInfoData);    /* [out] PSP_DEVINFO_DATA (ref is used to pass in cbSize) */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiGetDeviceRegistryProperty(
            DevInfoSafeHandle DeviceInfoSet,    /* [in] HDEVINFO */
            ref SP_DEVINFO_DATA DeviceInfoData, /* [in] PSP_DEVINFO_DATA */
            uint Property,                      /* [in] DWORD */
            out uint PropertyRegDataType,       /* [out,opt] PDWORD */
            IntPtr PropertyBuffer,              /* [out,opt] PBYTE (NULL)*/
            uint PropertyBufferSize,            /* [in] DWORD */
            out uint RequiredSize);             /* [out,opt] PDWORD */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiGetDeviceRegistryProperty(
            DevInfoSafeHandle DeviceInfoSet,    /* [in] HDEVINFO */
            ref SP_DEVINFO_DATA DeviceInfoData, /* [in] PSP_DEVINFO_DATA */
            uint Property,                      /* [in] DWORD */
            out uint PropertyRegDataType,       /* [out,opt] PDWORD */
            HGlobalSafeHandle PropertyBuffer,   /* [out,opt] PBYTE */
            uint PropertyBufferSize,            /* [in] DWORD (0) */
            IntPtr RequiredSize);               /* [out,opt] PDWORD (NULL) */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiEnumDeviceInterfaces(
            DevInfoSafeHandle DeviceInfoSet,    /* [in] HDEVINFO */
            ref SP_DEVINFO_DATA DeviceInfoData, /* [in,opt] PSP_DEVINFO_DATA */
            ref Guid InterfaceClassGuid,        /* [in] const GUID* */
            uint MemberIndex,                   /* [in] DWORD */
            ref SP_DEVICE_INTERFACE_DATA DeviceInterfaceData);  /* [out] PSP_DEVICE_INTERFACE_DATA (ref is used to pass in cbSize) */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiGetDeviceInterfaceDetail(
            DevInfoSafeHandle DeviceInfoSet,    /* [in] HDEVINFO */
            ref SP_DEVICE_INTERFACE_DATA DeviceInterfaceData,   /* [in] PSP_DEVICE_INTERFACE_DATA */
            IntPtr DeviceInterfaceDetailData,   /* [out,opt] PSP_DEVICE_INTERFACE_DETAIL_DATA */
            uint DeviceInterfaceDetailDataSize, /* [in] DWORD */
            out uint RequiredSize,              /* [out,opt] PDWORD */
            IntPtr DeviceInfoData);             /* [out,opt] PSP_DEVINFO_DATA */

        [DllImport("setupapi.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool SetupDiGetDeviceInterfaceDetail(
            DevInfoSafeHandle DeviceInfoSet,    /* [in] HDEVINFO */
            ref SP_DEVICE_INTERFACE_DATA DeviceInterfaceData,   /* [in] PSP_DEVICE_INTERFACE_DATA */
            IntPtr DeviceInterfaceDetailData,   /* [out,opt] PSP_DEVICE_INTERFACE_DETAIL_DATA */
            uint DeviceInterfaceDetailDataSize, /* [in] DWORD */
            IntPtr RequiredSize,                /* [out,opt] PDWORD */
            IntPtr DeviceInfoData);             /* [out,opt] PSP_DEVINFO_DATA */

        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern SafeFileHandle CreateFile(
            string lpFileName,              /* [in] LPCTSTR */
            uint dwDesiredAccess,           /* [in] DWORD */
            uint dwShareMode,               /* [in] DWORD */
            IntPtr lpSecurityAttributes,    /* [in,opt] LPSECURITY_ATTRIBUTES */
            uint dwCreationDisposition,     /* [in] DWORD */
            uint dwFlagsAndAttributes,      /* [in] DWORD */
            IntPtr hTemplateFile);          /* [in,opt] HANDLE */

        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool WriteFile(
            SafeFileHandle hFile,               /* [in] HANDLE */
            byte[] lpBuffer,                    /* [in] LPCVOID */
            uint nNumberOfBytesToWrite,         /* [in] DWORD */
            out uint lpNumberOfBytesWritten,    /* [out,opt] LPDWORD */
            IntPtr lpOverlapped);               /* [inout,opt] LPOVERLAPPED */

        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool ReadFile(
            SafeFileHandle hFile,               /* [in] HANDLE */
            IntPtr lpBuffer,                    /* [out] LPVOID */
            uint nNumberOfBytesToRead,          /* [in] DWORD */
            out uint lpNumberOfBytesRead,       /* [out] LPDWORD */
            IntPtr lpOverlapped);               /* [in] LPOVERLAPPED */

        /* ---- Imported structures ---- */
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode, Pack = 1)]
        private struct SP_DEVINFO_DATA {
            public uint cbSize;     /* DWORD */
            public Guid ClassGuid;  /* GUID */
            public uint DevInst;    /* DWORD */
            public IntPtr Reserved; /* ULONG_PTR */
        }   /* size is 28 on x86, 32 on x64; Pack = 1 happens to achieve this */

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode, Pack = 1)]
        private struct SP_DEVICE_INTERFACE_DATA {
            public uint cbSize;             /* DWORD */
            public Guid InterfaceClassGuid; /* GUID */
            public uint Flags;              /* DWORD */
            public IntPtr Reserved;         /* ULONG_PTR */
        }   /* size is 28 on x86, 32 on x64; Pack = 1 happens to achieve this */

        /* ---- Imported constants ---- */
        private const uint ERROR_SUCCESS = 0x00;                /* WIN32 error code */
        private const uint ERROR_NO_MORE_ITEMS = 0x00000103;    /* WIN32 error code */
        private const uint DIGCF_PRESENT = 0x02;                /* for SetupDiGetClassDevs() */
        private const uint DIGCF_DEVICEINTERFACE = 0x10;        /* for SetupDiGetClassDevs() */
        private const uint SPDRP_HARDWAREID = 0x00000001;       /* for SetupDiGetDeviceRegistryProperty() */
        private const uint REG_MULTI_SZ = 7;                    /* for SetupDiGetDeviceRegistryProperty() */
        private const uint GENERIC_WRITE = 0x40000000;          /* for CreateFile() */
        private const uint GENERIC_READ = 0x80000000;           /* for CreateFile() */
        private const uint OPEN_EXISTING = 3;                   /* for CreateFile() */
        private const uint FILE_SHARE_READ = 0x00000001;        /* for CreateFile() */
        private const uint FILE_SHARE_WRITE = 0x00000002;       /* for CreateFile() */

        /* GUID for HID class devices */
        private static Guid HIDClassGuid = new Guid(0x4d1e55b2, 0xf16f, 0x11cf, 0x88, 0xcb, 0x00, 0x11, 0x11, 0x00, 0x00, 0x30);

        /* ---- End of Imports ---- */
        #endregion Imports

        public DeviceConnection() {

            dataIn = new DataIn();
            dataOut = new DataOut();
            resetHomePositionFlag = false;
            hasDataEvent = new ManualResetEvent(false);
            dataSentEvent = new ManualResetEvent(false);
            dataMutex = new object();
            connectionMutex = new object();

            // Create and start the read / write thread
            Thread thread = new Thread(ReadWriteThread);
            thread.Start();
        }

        ~DeviceConnection() {
            Dispose(false);
        }
        
        public void Dispose() {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
        
        protected virtual void Dispose(bool disposing) {
            if (!threadCanceled) {
                threadCanceled = true;
                hasDataEvent.Set();     // (unblock the read / write thread)
            }
        }

        /** Gets whether the device is currently connected */
        public bool Connected {
            get { return writeHandle != null; }
        }

        /** Attempts to connect to the device, if not already connected.
         *  The Connected property will be set to true if the connection was
         *  successful.
         */
        public void TryConnect() {

            if (writeHandle == null) {
                /* Obtain the device path (null if not connected) */
                string devicePath = GetDevicePath();
                if (devicePath != null) {

                    /* Open a file handle to the device */
                    SafeFileHandle writeHandleToSave = CreateFile(devicePath, GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, IntPtr.Zero, OPEN_EXISTING, 0, IntPtr.Zero);
                    SafeFileHandle readHandleToSave = CreateFile(devicePath, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, IntPtr.Zero, OPEN_EXISTING, 0, IntPtr.Zero);

                    if (writeHandleToSave != null && !writeHandleToSave.IsInvalid &&
                        readHandleToSave != null && !readHandleToSave.IsInvalid) {
                        
                        // Save the handle.
                        lock (connectionMutex) {
                            writeHandle = writeHandleToSave;
                            readHandle = readHandleToSave;
                        }

                        // Tell the writer thread to write some initial data to the
                        // device, and to initially set the home position.
                        lock (dataMutex) {
                            resetHomePositionFlag = true;
                            hasDataEvent.Set();
                        }
                    }
                }
            }
        }

        /** Disconnects from the device.  Does nothing if not connected. */
        public void Disconnect() {
            lock (connectionMutex) {
                writeHandle.Close();
                readHandle.Close();
                writeHandle = null;
                readHandle = null;
            }
        }

        /** Forces a re-send of the previous data, to test if we're still connected.
         *  Sets the Connected property to false if the send fails.
         */
        public void TestConnection() {
            dataSentEvent.Reset();

            // Set the "has data" flag to force the read / write thread to re-send the data.
            hasDataEvent.Set();

            // Wait for the write to finish.  If it failed, the Connection property
            // should have been set to false.
            dataSentEvent.WaitOne();
        }

        /** Updates the given robot with the latest data received from the USB device,
         *  and queues new data to send.
         */
        public void UpdateRobot(Robot robot) {
            lock (dataMutex) {  // (prevent concurrent access)

                // Update the robot with the last-read data.
                UpdateRobotFromInput(robot, ref dataIn);

                // Update the output structure, to be written by the background thread.
                FillOutStruct(ref dataOut, robot);

                // Read & reset the robot's flag to reset the home position.
                resetHomePositionFlag |= robot.ResetHomePositionFlag;
                robot.ResetHomePositionFlag = false;

                hasDataEvent.Set();     // (signal write thread)
            }
        }

        /** Returns the path to the USB device, or NULL if it's not connected or an
         *  error occurs.
         */
        private string GetDevicePath() {

            DevInfoSafeHandle deviceSet = null;
            HGlobalSafeHandle pProperty = null;
            HGlobalSafeHandle pInterfaceDetails = null;

            try {
                SP_DEVINFO_DATA deviceData = new SP_DEVINFO_DATA();
                SP_DEVICE_INTERFACE_DATA interfaceData = new SP_DEVICE_INTERFACE_DATA();

                /* Create a list of all plugged in devices */
                deviceSet = SetupDiGetClassDevs(out HIDClassGuid, IntPtr.Zero, IntPtr.Zero, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

                if (!deviceSet.IsInvalid) {
                    for (uint deviceIndex = 0; deviceIndex < 1000000; deviceIndex++) {  /* 1000000 is a failsafe */

                        /* Set up an SP_DEVINFO_DATA structure so we can obtain device properties */
                        deviceData.cbSize = (uint)Marshal.SizeOf(deviceData);
                        if (SetupDiEnumDeviceInfo(deviceSet, deviceIndex, ref deviceData) == false) {
                            if (Marshal.GetLastWin32Error() == ERROR_NO_MORE_ITEMS) {
                                break;  /* End of the list */
                            }
                            continue;   /* Try the next device */
                        }

                        /* Obtain the hardware ID of the device.  Note that we have to query for the size of the property before we
                         * can actually obtain it. */
                        uint propType;
                        uint propSize;
                        SetupDiGetDeviceRegistryProperty(deviceSet, ref deviceData, SPDRP_HARDWAREID, out propType, IntPtr.Zero, 0, out propSize);
                        if (propType != REG_MULTI_SZ || propSize == 0) { continue; }
                        pProperty = new HGlobalSafeHandle(Marshal.AllocHGlobal((int)propSize));
                        SetupDiGetDeviceRegistryProperty(deviceSet, ref deviceData, SPDRP_HARDWAREID, out propType, pProperty, propSize, IntPtr.Zero);
                        string hardwareID = Marshal.PtrToStringUni(pProperty);
                        pProperty.Close();

                        /* Only continue if we find the correct VID / PID */
                        if (hardwareID.ToLowerInvariant().Contains(targetHardwareID.ToLowerInvariant())) {

                            /* Obtain the first HID class interface available to the device found */
                            interfaceData.cbSize = (uint)Marshal.SizeOf(interfaceData);
                            if (SetupDiEnumDeviceInterfaces(deviceSet, ref deviceData, ref HIDClassGuid, 0, ref interfaceData) == false) {
                                continue;   /* Try the next device */
                            }

                            /* Obtain the path to the interface.  Note that we have to query for the length of the structure before we
                             * can actually obtain it.
                             * 
                             * NOTE: sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA) is 6 on x86 and 8 on x64.  SetupDiGetDeviceInterfaceDetail
                             * will fail with ERROR_INVALID_USER_BUFFER if the size given is incorrect for the current architecture.
                             */
                            uint detailSize;
                            uint minDetailSize = (IntPtr.Size == 8) ? 8U : 6U;
                            SetupDiGetDeviceInterfaceDetail(deviceSet, ref interfaceData, IntPtr.Zero, 0, out detailSize, IntPtr.Zero);
                            if (detailSize < minDetailSize) { continue; }
                            pInterfaceDetails = new HGlobalSafeHandle(Marshal.AllocHGlobal((int)detailSize));
                            Marshal.WriteInt32(pInterfaceDetails, (int)minDetailSize); /* Fill in cbSize (DWORD); do not include the string length! */
                            if (SetupDiGetDeviceInterfaceDetail(deviceSet, ref interfaceData, pInterfaceDetails, detailSize, IntPtr.Zero, IntPtr.Zero)) {
                                return Marshal.PtrToStringUni((IntPtr)(((IntPtr)pInterfaceDetails).ToInt64() + 4));
                            }
                            pInterfaceDetails.Close();
                        }
                    }
                }
                return null;
            }
            catch (OutOfMemoryException e) {    /* Silently fail the connection attempt if out of memory */
                Debug.Fail(e.Message);
                return null;
            }
            finally {
                /* Close the handles now, instead of waiting for garbage collection */
                if (pProperty != null) { pProperty.Close(); }
                if (pInterfaceDetails != null) { pInterfaceDetails.Close(); }
                if (deviceSet != null) { deviceSet.Close(); }
            }
        }

        /** Simple wrapper around ReadFile() that works in managed code */
        private static bool ReadFileManaged(SafeFileHandle hFile, byte[] buffer,
            uint nNumberOfBytesToRead, out uint lpNumberOfBytesRead,
            IntPtr lpOverlapped) {

            IntPtr bufferPtr = IntPtr.Zero;
            try {
                bufferPtr = Marshal.AllocHGlobal((int)nNumberOfBytesToRead);
                if (ReadFile(hFile, bufferPtr, nNumberOfBytesToRead, out lpNumberOfBytesRead, lpOverlapped)) {
                    Marshal.Copy(bufferPtr, buffer, 0, (int)lpNumberOfBytesRead);
                    return true;
                }
                return false;
            }
            finally {
                if (bufferPtr != IntPtr.Zero) {
                    Marshal.FreeHGlobal(bufferPtr);
                }
            }
        }

        /** Main function for the read / write thread */
        private void ReadWriteThread() {
            while (!threadCanceled) {

                DataOut localDataOut;
                DataIn localDataIn = new DataIn();
                bool localResetHomePositionFlag;

                //// DATA MUTEX ///////////////////////////////////////////////////

                // Wait for data to arrive, then take a snapshot of it so it
                // doesn't change during the write.  Don't hold the data mutex
                // DURING the write, however!
                hasDataEvent.WaitOne();
                lock (dataMutex) {
                    localDataOut = dataOut;
                    localResetHomePositionFlag = resetHomePositionFlag;
                    resetHomePositionFlag = false;
                    hasDataEvent.Reset();
                }

                //// CONNECTION MUTEX /////////////////////////////////////////////

                lock (connectionMutex) {

                    if (writeHandle != null) {
                        if (!DoReadWrite(writeHandle, readHandle, ref localDataOut, out localDataIn, localResetHomePositionFlag)) {
                            writeHandle.Close();
                            readHandle.Close();
                            writeHandle = null;    // (disconnected)
                            readHandle = null;
                        }
                    }

                    // (signal completion regardless of whether the send was actually successful)
                    dataSentEvent.Set();
                }

                //// DATA MUTEX ///////////////////////////////////////////////////

                // Save the data read from the device.
                lock (dataMutex) {
                    dataIn = localDataIn;

                    // If we're still connected and not at the target location,
                    // signal that there is still more data to read!
                    if (writeHandle != null &&
                        (localDataIn.m1Current != localDataOut.m1Target ||
                        localDataIn.m2Current != localDataOut.m2Target ||
                        localDataIn.m3Current != localDataOut.m3Target ||
                        localDataIn.m4Current != localDataOut.m4Target)) {

                        hasDataEvent.Set();
                    }
                }

                // Wait a minimum amount of time before doing another read / write.
                Thread.Sleep(5);
            }
        }

        /** Performs the actual read / write.  Returns true if the write was
         *  successful, or false if it was not.
         */
        private static bool DoReadWrite(SafeFileHandle writeHandle, SafeFileHandle readHandle,
            ref DataOut outStruct, out DataIn inStruct,
            bool resetHomePosition) {

            Debug.Assert(writeHandle != null && readHandle != null, "Argument was null");

            // Initialize the output in case we fail.
            inStruct = new DataIn();

            // Fill the output buffer with data.
            byte[] outBuffer = new byte[65];
            FillOutputBuffer(outBuffer, ref outStruct, resetHomePosition);

            // Write the output buffer.
            uint bytesWritten = 0;
            if (WriteFile(writeHandle, outBuffer, (uint)outBuffer.Length, out bytesWritten, IntPtr.Zero) && bytesWritten == outBuffer.Length) {

                // Read the response.
                uint bytesRead;
                byte[] inBuffer = new byte[65];
                if (ReadFileManaged(readHandle, inBuffer, (uint)inBuffer.Length, out bytesRead, IntPtr.Zero) && bytesRead == inBuffer.Length) {
                    
                    // Fill the input structure from the input buffer.
                    if (ParseInStruct(out inStruct, inBuffer)) {
                        return true;
                    }
                    else {
                        Debug.Fail("Input buffer was invalid");
                    }
                }
            }

            return false;
        }

        /** Fills an output buffer with the contents of the given structure */
        private static void FillOutputBuffer(byte[] buffer, ref DataOut outStruct, bool resetHomePosition) {
            Debug.Assert(buffer != null, "Argument was null");
            if (buffer.Length != 65) { throw new ArgumentException("buffer is wrong size"); }

            // Initialize all unused bytes to 0xFF to lower EMI and power consumption
            for (int i = 0; i < buffer.Length; i ++) { buffer[i] = 0xFF; }

            buffer[0] = 0x00;   // (report ID, always zero)
            buffer[1] = 0x83;   // (0x83 = "send current position")

            buffer[2] = (byte)(outStruct.m1Target & 0xFF);
            buffer[3] = (byte)(outStruct.m1Target >> 8);
            buffer[4] = (byte)(outStruct.m2Target & 0xFF);
            buffer[5] = (byte)(outStruct.m2Target >> 8);
            buffer[6] = (byte)(outStruct.m3Target & 0xFF);
            buffer[7] = (byte)(outStruct.m3Target >> 8);
            buffer[8] = (byte)(outStruct.m4Target & 0xFF);
            buffer[9] = (byte)(outStruct.m4Target >> 8);

            buffer[10] = (byte)(outStruct.paused ? 0x00 : 0xFF);    // byte 10: emergency power shut-off
            buffer[11] = (byte)(resetHomePosition ? 0xFF : 0x00);   // byte 11: set home position

            buffer[12] = (byte)(outStruct.m1Speed & 0xFF);
            buffer[13] = (byte)(outStruct.m1Speed >> 8);
            buffer[14] = (byte)(outStruct.m2Speed & 0xFF);
            buffer[15] = (byte)(outStruct.m2Speed >> 8);
            buffer[16] = (byte)(outStruct.m3Speed & 0xFF);
            buffer[17] = (byte)(outStruct.m3Speed >> 8);
            buffer[18] = (byte)(outStruct.m4Speed & 0xFF);
            buffer[19] = (byte)(outStruct.m4Speed >> 8);

            buffer[20] = (byte)(outStruct.paused ? 0x00 : 0xFF);    // byte 20: pause function (counter only)
        }

        /** Parses an input structure from the contents of the given input buffer.
         *  Returns true if successful, false if the contents of the buffer were invalid.
         */
        private static bool ParseInStruct(out DataIn inStruct, byte[] buffer) {
            Debug.Assert(buffer != null, "Argument was null");
            if (buffer.Length != 65) { throw new ArgumentException("buffer is wrong size"); }

            // Initialize the structure to zero, in case we miss anything.
            inStruct = new DataIn();

            // Do some basic validation
            if (buffer[1] != 0x83) {    // (0x83 = echo of "send current position")
                return false;
            }

            inStruct.m1Current = (short)((buffer[3] << 8) | buffer[2]);
            inStruct.m2Current = (short)((buffer[5] << 8) | buffer[4]);
            inStruct.m3Current = (short)((buffer[7] << 8) | buffer[6]);
            inStruct.m4Current = (short)((buffer[9] << 8) | buffer[8]);

            return true;
        }

        /** Updates a robot object with the values of the given structure */
        private static void UpdateRobotFromInput(Robot robot, ref DataIn inStruct) {
            Debug.Assert(robot != null, "Argument was null");

            DofVector tipPosition = new DofVector(
                robot.ConvertStepsToAngle(inStruct.m1Current, 1),
                robot.ConvertStepsToAngle(inStruct.m2Current, 2),
                robot.ConvertStepsToAngle(inStruct.m3Current, 3));

            double m4Position = robot.ConvertStepsToAngle(inStruct.m4Current, 4);

            ArmPosition newPosition = new ArmPosition(tipPosition, m4Position);

            robot.CurrentPosition = newPosition;
        }

        /** Fills an output structure with the parameters of the given robot */
        private static void FillOutStruct(ref DataOut outStruct, Robot robot) {
            Debug.Assert(robot != null, "Argument was null");

            // Convert to DOF
            DofVector targetTipPos = robot.TargetPosition.TipPosition.ConvertToDofPoint(robot);
            double targetM4Pos = robot.TargetPosition.M4;

            if (targetTipPos != null) {
                outStruct.m1Target = (short)Math.Max(Int16.MinValue, Math.Min(Int16.MaxValue, Math.Round(robot.ConvertAngleToSteps(targetTipPos.M1, 1))));
                outStruct.m2Target = (short)Math.Max(Int16.MinValue, Math.Min(Int16.MaxValue, Math.Round(robot.ConvertAngleToSteps(targetTipPos.M2, 2))));
                outStruct.m3Target = (short)Math.Max(Int16.MinValue, Math.Min(Int16.MaxValue, Math.Round(robot.ConvertAngleToSteps(targetTipPos.M3, 3))));
            }

            outStruct.m4Target = (short)Math.Max(Int16.MinValue, Math.Min(Int16.MaxValue, Math.Round(robot.ConvertAngleToSteps(targetM4Pos, 4))));

            double stepsPerRads1 = robot.GetMotorParameters(1).StepsPerRadian;
            double stepsPerRads2 = robot.GetMotorParameters(2).StepsPerRadian;
            double stepsPerRads3 = robot.GetMotorParameters(3).StepsPerRadian;
            double stepsPerRads4 = robot.GetMotorParameters(4).StepsPerRadian;

            // Speed[steps/s] = 60000 / Period
            // => Period = 60000 / Speed[steps/s]
            // => Period = 60000 / (Speed[rads/s] * StepsPerRad[steps/rads])
            double period1 = 60000.0 / (robot.Speed.M1Speed * stepsPerRads1);
            double period2 = 60000.0 / (robot.Speed.M2Speed * stepsPerRads2);
            double period3 = 60000.0 / (robot.Speed.M3Speed * stepsPerRads3);
            double period4 = 60000.0 / (robot.Speed.M4Speed * stepsPerRads4);

            outStruct.m1Speed = (short)Math.Max(0, Math.Min(short.MaxValue, Math.Round(period1)));
            outStruct.m2Speed = (short)Math.Max(0, Math.Min(short.MaxValue, Math.Round(period2)));
            outStruct.m3Speed = (short)Math.Max(0, Math.Min(short.MaxValue, Math.Round(period3)));
            outStruct.m4Speed = (short)Math.Max(0, Math.Min(short.MaxValue, Math.Round(period4)));

            outStruct.paused = robot.Paused;
        }
    }
}
