﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;

namespace CBRobot {
    
    /** Represents a vector composed of motor positions (M1, M2, M3), measured
     *  in RADIANS, that defines the position or change in position of the robot
     *  arm's tip.
     *
     *  When interpreted as a point, the vector (0, 0, 0) generally represents
     *  the "home position" of the arm.
     */
    public class DofVector : Vector {

        private double m1, m2, m3;
        private static DofVector zero;

        static DofVector() {
            zero = new DofVector(0.0, 0.0, 0.0);
        }

        /** Creates a new DofVector with the specified motor positions (M1,
         *  M2, M3).
         */
        public DofVector(double m1, double m2, double m3) {
            this.m1 = m1;
            this.m2 = m2;
            this.m3 = m3;
        }

        /** Gets motor #1's position, in radians */
        public double M1 {
            get { return m1; }
            // (intentionally read-only)
        }

        /** Gets motor #2's position, in radians */
        public double M2 {
            get { return m2; }
        }

        /** Gets motor #3's position, in radians */
        public double M3 {
            get { return m3; }
        }

        /** Returns the DOF vector (0, 0, 0) */
        public static DofVector Zero {
            get { return zero; }
        }

        /** Returns the value of the component with the given index (0, 1, or 2) */
        public double GetComponent(int componentIndex) {
            switch (componentIndex) {
                case 0:
                    return m1;
                case 1:
                    return m2;
                case 2:
                    return m3;
                default:
                    throw new ArgumentOutOfRangeException("componentIndex",
                        componentIndex, "Component index was out of range (0 - 2)");
            }
        }

        /** Returns a new vector with the specified component (0, 1, or 2) set
         *  to the given value.
         */
        public DofVector SetComponent(int componentIndex, double value) {
            switch (componentIndex) {
                case 0:
                    return new DofVector(value, m2, m3);
                case 1:
                    return new DofVector(m1, value, m3);
                case 2:
                    return new DofVector(m1, m2, value);
                default:
                    throw new ArgumentOutOfRangeException("componentIndex",
                        componentIndex, "Component index was out of range (0 - 2)");
            }
        }

        // Inherited from Vector
        public override DofVector ConvertToDofPoint(Robot robot) {
            return this;
        }

        /** Returns a string representation of this point */
        public override string ToString() {
            return String.Format("({0:N6}, {1:N6}, {2:N6})", m1, m2, m3);
        }
    }
}
