﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Diagnostics;
using System.Windows.Forms;

namespace CBRobot {
    
    /** This class is responsible for "running" a path - it takes a CBPath as
     *  input, and feeds position and speed values to the robot based on the
     *  current position along the path.
     */
    internal class PathExecutor : IDisposable {
        private Robot robot;
        private Path path;
        private Timer updateTimer;      /* Timer to update the robot's position / speed */
        private double pathTime;        /* Time parameter for current path position */
        private Stopwatch stopwatch;    /* Stopwatch to measure time with */
        private double lastTime;                /* Time at the last iteration of the update timer */
        private ArmPosition lastPosition;       /* Arm position at last update */
        private bool waitingForStartPosition;   /* Are we currently waiting to arrive at the start position? */
        private double startDelayRemaining;     /* Remaining start delay */

        private const int PathUpdateInterval = 100; /* Interval (milliseconds) at which to compute and send new position / speed values to the robot */
        private const double TargetPositionThreshold = 2.0; /* Threshold (in steps) used to determine if the robot is already at the target position */
        private const double StartDelay = 0.25;     /* Delay (in seconds) after reaching the start position before starting the program */

        public PathExecutor(Robot robot) {
            this.robot = robot;
            lastPosition = ArmPosition.Zero;
            stopwatch = new Stopwatch();
        }

        ~PathExecutor() {
            Dispose(false);
        }
        
        public void Dispose() {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
        
        protected virtual void Dispose(bool disposing) {
            if (disposing) {
                if (updateTimer != null) { updateTimer.Dispose(); updateTimer = null; }
            }
        }

        /** Specifies the path to be executed.  Set to null to stop executing
         *  the current path.
         */
        public Path Path {
            get { return path; }
            set {
                if (path != value) {
                    // Stop the timer if there is no path.
                    if (value == null) { updateTimer.Dispose(); updateTimer = null; }

                    path = value;
                    pathTime = 0.0;
                    stopwatch.Stop();
                    stopwatch.Reset();
                    stopwatch.Start();
                    lastTime = 0.0;
                    UpdatePositionValue(ref lastPosition, robot);
                    waitingForStartPosition = true;

                    // Start the timer if there is a path to run.
                    if (value != null && updateTimer == null) {
                        updateTimer = new Timer();
                        updateTimer.Interval = PathUpdateInterval;
                        updateTimer.Tick += updateTimer_Tick;
                        updateTimer.Enabled = true;
                    }
                }
            }
        }

        /** Called when the update timer elapses */
        private void updateTimer_Tick(object sender, EventArgs e) {
            if (path != null) {

                // Update the robot's CurrentPosition property
                robot.UpdateFromDevice();

                // Update the current time
                double realTimeDifference = UpdateTimeValue(ref lastTime, stopwatch);
                double timeDifferenceByMovement = UpdatePositionValue(ref lastPosition, robot);

                if (waitingForStartPosition) {

                    // Tell the robot to go to the start of the path
                    ArmPosition startPosition = path.StartPosition;
                    Debug.Assert(startPosition != null, "Path returned a null start position");
                    if (startPosition != null) { robot.TargetPosition = startPosition; }
                    // (user-specified speed value is used when moving to start position)

                    if (robot.IsWithinThresholdOfTarget(TargetPositionThreshold)) {
                        waitingForStartPosition = false;
                        startDelayRemaining = StartDelay;
                    }
                }
                else {

                    // Wait a short amount of time after reaching the start position
                    if (startDelayRemaining > 0) {
                        startDelayRemaining -= realTimeDifference;
                    }
                    else {
                        // Increment by real time if the robot's motors are already at their
                        // target position.  Otherwise, increment based on how far the arm has
                        // moved (to minimize drift between the time parameter and the robot's
                        // actual position)
                        if (robot.IsWithinThresholdOfTarget(TargetPositionThreshold)) {
                            pathTime += realTimeDifference;
                        }
                        else {
                            pathTime += timeDifferenceByMovement;
                        }
                    }

                    // Notify the path that the time parameter has changed.
                    path.SetPathTime(pathTime);

                    // Ask the path for the target position / speed for the current time value.
                    ArmPosition pos;
                    ArmSpeed speed;
                    path.GetParametersAtTime(pathTime, out pos, out speed);
                    Debug.Assert(pos != null, "Path returned a null position");
                    Debug.Assert(speed != null, "Path returned a null speed");

                    // Update the robot's target position / speed.
                    if (pos != null) { robot.TargetPosition = pos; }
                    if (speed != null) { robot.Speed = speed; }

                    // Stop executing if the path is finished
                    if (pathTime >= path.PathLength) {
                        Path oldPath = robot.Path;
                        robot.Path = null;          // (stop)
                        if (oldPath.Loop) {
                            robot.Path = oldPath;   // (restart)
                        }
                    }
                }
            }
        }

        /** Updates the given time value, then returns the number of seconds
         *  since the previous value.
         */
        private static double UpdateTimeValue(ref double lastTime, Stopwatch stopwatch) {
            double newTime = stopwatch.Elapsed.TotalSeconds;
            double result = newTime - lastTime;
            lastTime = newTime;
            return result;
        }

        /** Updates the given position value to the current position, and computes a
         *  time difference based on how far the arm has traveled in the direction of
         *  the target position (using DOF coordinates and DOF speed - may introduce a
         *  lot of error if applied to a path based on cartesian coordinates).
         */
        private static double UpdatePositionValue(ref ArmPosition lastPosition, Robot robot) {
            ArmPosition currentPos = robot.CurrentPosition;
            ArmPosition targetPos = robot.TargetPosition;
            DofVector currentTip = currentPos.TipPosition.ConvertToDofPoint(robot);
            DofVector targetTip = targetPos.TipPosition.ConvertToDofPoint(robot);
            DofVector lastTip = lastPosition.TipPosition.ConvertToDofPoint(robot);

            // Compute a vector representing the robot's expected velocity from its last position
            double v1 = (targetTip.M1 >= lastTip.M1 ? 1.0 : -1.0) * robot.Speed.M1Speed;
            double v2 = (targetTip.M2 >= lastTip.M2 ? 1.0 : -1.0) * robot.Speed.M2Speed;
            double v3 = (targetTip.M3 >= lastTip.M3 ? 1.0 : -1.0) * robot.Speed.M3Speed;
            double v4 = (targetPos.M4 >= lastPosition.M4 ? 1.0 : -1.0) * robot.Speed.M4Speed;

            // (zero out motors that aren't actually going to move)
            if (robot.IsMotorWithinThresholdOfTarget(1, TargetPositionThreshold)) { v1 = 0; }
            if (robot.IsMotorWithinThresholdOfTarget(2, TargetPositionThreshold)) { v2 = 0; }
            if (robot.IsMotorWithinThresholdOfTarget(3, TargetPositionThreshold)) { v3 = 0; }
            if (robot.IsMotorWithinThresholdOfTarget(4, TargetPositionThreshold)) { v4 = 0; }
            double vMagSq = v1 * v1 + v2 * v2 + v3 * v3 + v4 * v4;

            // Compute the dot product between the travel vector and the expected velocity vector
            double t1 = currentTip.M1 - lastTip.M1;
            double t2 = currentTip.M2 - lastTip.M2;
            double t3 = currentTip.M3 - lastTip.M3;
            double t4 = currentPos.M4 - lastPosition.M4;
            double dotProduct = t1 * v1 + t2 * v2 + t3 * v3 + t4 * v4;

            // Divide the result by the length of the expected velocity vector SQUARED
            // to yield the distance traveled along that vector, RELATIVE to the length
            // of that vector
            double timeTraveled = vMagSq != 0.0 ? dotProduct / vMagSq : 0.0;    // (will be exactly zero if all motors at target position)

            lastPosition = robot.CurrentPosition;
            return timeTraveled;
        }
    }
}
