
/*
**  WiFi Web Server for the DeRose Family Entertainment Center
**  
**  The starting point was the WifiWebServer Arduino example.
** 
**  (c) 428 Industries, June, 2013
 */

#include <WiFi.h>
#include <SD.h>
#include <LiquidCrystal.h>
#include <LED.h>
#include <RGB_LED.h>

#include "EcSignalData.h"

//----------------------------- 
//        Pin assignments
//-----------------------------

// Pins 10-13 used by the WiFi shield on an Uno
// Pins 50-52 used by the WiFi shield on the Mega

// LCD pin assignments           
#define RS            22
#define EN            24
#define DB4           26
#define DB5           28
#define DB6           30
#define DB7           32
#define IRledPin      34                // IR LEDs connected to this pin

// Shelf and floor LED pin assignments
#define FLOOR_RED     42
#define FLOOR_GREEN   44
#define FLOOR_BLUE    46

//#define DEBUG

// Define one of these
//#define UNO
#define MEGA

#ifdef UNO
  #define IR_DELAY 10
#endif
#ifdef MEGA
  #define IR_DELAY 8
#endif

//-----------------------------
//      Types
//-----------------------------

// What state is the system in?
typedef enum {
    S_ITUNES,
    S_TV,
    S_APPLE_TV,
    S_KQED,
    S_DVD,
    S_OFF,
} StateType;

//----------------------------- 
//      Global variables
//----------------------------- 

RGB_LED         FloorLight( FLOOR_RED, FLOOR_GREEN, FLOOR_BLUE, true /* isOneWhenHigh */);   // Controls floor lighting
const char      *HeartBeatStr[] = {"*", " "};
int             HeartBeat = false;
LiquidCrystal   Lcd(RS, EN, DB4, DB5, DB6, DB7);
long int        LastMillis = 0;
boolean         IsVideoOn = false;       // Keep track of the on/off state of video device
boolean         IsDVDOn   = false;       // Keep track of the on/off state of the DVD player
boolean         IsAudioOn = false;       // Keep track of the on/off state of the audio amp
boolean         IsMuted   = false;       // True if audio is muted
boolean         IsSpeakerBOn = false;
IPAddress       MyAddress;
int             NewVolume;               // What to set the volume to at the next opportunity
char            Pass[] = "Airport*wave";
int             SDChipSelect = 4;        // For communicating with the SD card
char            Ssid[] = "DeRose Family Network";
StateType       State = S_OFF;
int             Status = WL_IDLE_STATUS;
int             Volume    = 0;           // A percentage, [0 ... 100]
WiFiServer      WifiServer(80);

int availableMemory() {
  int size = 1024; // Use 2048 with ATmega328
  byte *buf;

  while ((buf = (byte *) malloc(--size)) == NULL)
    ;

  free(buf);

  return size;
}

void printAvailableMemory()
{
  Serial.print("Available memory: ");
  Serial.println(availableMemory());
}

// This procedure sends a 38KHz pulse to the IRledPin 
// for a certain # of microseconds. We'll use this whenever we need to send codes
// Copied from adafru.it/157
void pulseIR(long microsecs) {
  // we'll count down from the number of microseconds we are told to wait
   cli();  // this turns off any background interrupts
 
   while (microsecs > 0) {
       // 38 kHz is about 13 microseconds high and 13 microseconds low
       digitalWrite(IRledPin, HIGH);  // this takes a few microseconds to happen
       delayMicroseconds(IR_DELAY);   // hang out for IR_DELAY microseconds so that digitalWrite + IR_DELAY = 13 microsecond
       digitalWrite(IRledPin, LOW);   // this also a few microseconds
       delayMicroseconds(IR_DELAY);   // hang out for IR_DELAY microseconds so that digitalWrite + IR_DELAY = 13 microsecond
       // so 26 microseconds altogether
       microsecs -= 26;
  }
 
  sei();  // this turns them back on
}

void sendSignal(int *signal)
{
    int i = 0;  // Index into signal data array
    
    while (true) {
        pulseIR( 10 * signal[i]);
        delayMicroseconds( 10 * signal[i+1]);
        if (signal[i+1] == 0) return;
        i += 2;
    }
}

void setup() {
  // set up the LCD's number of columns and rows: 
  Lcd.begin(16, 2);
  Lcd.setCursor(0, 1);
  Lcd.print("Initializing");
  
  //Initialize serial and wait for port to open:
  Serial.begin(9600); 
  while (!Serial) {
    ; // wait for serial port to connect. Needed for Leonardo only
  }
  printAvailableMemory();
  
  pinMode(IRledPin, OUTPUT);
  digitalWrite(IRledPin, HIGH);
  
  // The hardware CS pin (pin 10 on the Uno) must be set as an output
  pinMode( 10, OUTPUT);
  
  // Outputs used for Shelf and Floor RGB LED lighting
  pinMode( 38, OUTPUT); digitalWrite( 38, LOW);
  pinMode( 40, OUTPUT); digitalWrite( 40, LOW);
  pinMode( 42, OUTPUT); digitalWrite( 42, LOW);
  pinMode( 44, OUTPUT); digitalWrite( 44, LOW);
  pinMode( 46, OUTPUT); digitalWrite( 46, LOW);
  pinMode( 48, OUTPUT); digitalWrite( 48, LOW);
 
  FloorLight.SetSmoothDelay(10);
  FloorLight.SetRGB( 0, 255, 255);
 
  // see if the card is present and can be initialized:
  if (!SD.begin(SDChipSelect)) {
    Lcd.setCursor(0,0);
    Lcd.print("SD failed              ");
    // don't do anything more:
    return;
  }

  // check for the presence of the shield:
  if (WiFi.status() == WL_NO_SHIELD) {
    Lcd.setCursor(0,1);
    Lcd.print("No WiFi shield"); 
    // don't continue:
    while(true);
  } 
  
  // attempt to connect to Wifi network:
  while ( Status != WL_CONNECTED) {
    Lcd.setCursor(0, 1);
    Lcd.print(Ssid);
    // Connect to WPA/WPA2 network. Change this line if using open or WEP network:    
    Status = WiFi.begin(Ssid, Pass);

    // wait a few seconds for connection:
    delay(5000);
  } 
  WifiServer.begin();
  // you're connected now, so print out the status:
  printWifiStatus();
  
  Lcd.setCursor(0,1);
  Lcd.print("IP: ");
  Lcd.print(MyAddress);
  Lcd.print("           ");
  
  Lcd.setCursor(0,0);
  Lcd.print("Initializing SD");
  // make sure that the default chip select pin is set to
  // output, even if you don't use it:
  pinMode(10, OUTPUT);
  
  Lcd.setCursor(0,0);
  Lcd.print("Ready                 ");
}

// Read a line of input from the Wifi client into buffer. Replace the newline with
// \0 and return the length of the returned string
int readLine(WiFiClient client, char *buffer)
{
    int i = 0;
    while (client.connected()) {
        // Wait for data
        if (client.available()) {
            char c = client.read();
#ifdef DEBUG
            Serial.write(c);
#endif
            if (c == '\n') {
                // End of line
                break;
            } else if (c != '\r') {
                buffer[i++] = c;
            }
        }
    }
    buffer[i] = '\0';
    return i;   
}

// Read a line of input from the file on the SD card into buffer.
// Replace newline with \0 and return the length of the returned string
int sdReadLine( File file, char *buffer)
{
   int i = 0;
   while (file.available()) {
       char c = file.read();
       if (c == '\n') {
               // End of line
               break;
       } else if (c != '\r') {
               buffer[i++] = c;
       }
   }
   buffer[i] = '\0';
   return i;
}

//
// Read the data stored in fileName and send it as a
// signal.
//
void processFile(const char *fileName)
{
   //
   // Read the corresponding data file
   //
   
   #define MAX_DATA_SIZE 256
   int signalData[MAX_DATA_SIZE];
   char buffer[80];
#ifdef DEBUG
   Serial.print("Opening ");
   Serial.print(fileName);
   Serial.print("...");
#endif
   File dataFile = SD.open(fileName);
   if (!dataFile) {
       Serial.print("can't open file ");
       Serial.println(fileName);
       return;
   } else {
#ifdef DEBUG
       Serial.println("ok.");
#endif
   }
   int i = 0;
   while (sdReadLine( dataFile, buffer) > 0) {
       int on, off;
       if (sscanf( buffer, "%d, %d,", &on, &off) != 2) {
           Serial.print("Error reading line ");
           Serial.println(i/2);
           Serial.print("Buffer read: ");
           Serial.println(buffer);
           dataFile.close();
           return;
       }
       signalData[i++] = on;
       signalData[i++] = off;
       if (i >= MAX_DATA_SIZE) {
           Serial.print("Data overflow reading ");
           Serial.println(fileName);
           dataFile.close();
           return;
       }
   }
   dataFile.close();
   sendSignal( signalData);
}

// Map an action to a file name. Return NULL if the action
// can't be found in EcSymTable.
const char *symTableLookup(const char *action)
{
    for (int i = 0; EcSymTable[i].action != NULL; i++) {
        if (strcmp(action, EcSymTable[i].action) == 0) {
            return EcSymTable[i].fileName;
        }
    }
    return NULL;
}

void turnVideoOn()
{
    if (IsVideoOn) return;
    processFile( "vOnOff.txt");
    IsVideoOn = true;
}

void turnVideoOff()
{
    if (!IsVideoOn) return;
    processFile( "vOnOff.txt");
    IsVideoOn = false;
}

void turnDVDOn()
{
    if (IsDVDOn) return;
    processFile( "bOnOff.txt");
    IsDVDOn = true;
}

void turnDVDOff()
{
    if (!IsDVDOn) return;
    processFile( "bOnOff.txt");
    IsDVDOn = false;
}

void setVolume(int newVolume)  // newVolume is a percentage [0..100]
{
    // Compute the number of signals that need to be sent. Each signal corresponds to
    // about 5 percent change.
    if (newVolume < 0) newVolume = 0;
    if (newVolume > 100) newVolume = 100;
    
    int nSignals = (newVolume - Volume)/5;
    if (nSignals > 0) {
        // Increase the volume
        for (int i = 0; i < nSignals; ++i) {
            processFile( "aVolUp.txt");
            delay(300);
        }
    } else {
        // Descrease the volume
        nSignals *= -1.0;
        for (int i = 0; i < nSignals; ++i) {
            processFile( "aVolDown.txt");
            delay(300);
        }
    }
    Volume = newVolume;
}

void turnAudioOn()
{
    if (IsAudioOn) return;
    processFile("aOn.txt");
    IsAudioOn = true;
    // Wait a few seconds for the device to power up
    delay(10000);
}

void turnAudioOff()
{
    if (!IsAudioOn) return;
    setVolume( 10);
    processFile("aOff.txt");
    IsAudioOn = false;
}

void turnSpeakerBOn()
{
    if (IsSpeakerBOn) return;
    delay(100);
    processFile("aSpB.txt");
    IsSpeakerBOn = true;
    delay(3000);
}

void turnSpeakerBOff()
{
    if (!IsSpeakerBOn) return;
    delay(100);
    processFile("aSpB.txt");
    IsSpeakerBOn = false;
    delay(3000);
}

// Go to the specified channel number. E.g. "56"
void gotoChannel(const char *channelNumber)
{
    for (int i = 0; channelNumber[i] != '\0'; i++) {
        char fileName[16];
        sprintf( fileName, "v%c.txt", channelNumber[i]);
        processFile( fileName);
        delay(10);
    }
}


void updateStatus( char *action)
{
    // Make state changes but don't take the time to send the IR codes since
    // that takes long enough that it causes the connection to time out.
    if (strcmp(action, "mute") == 0) {
        IsMuted = !IsMuted;
    } else if (strcmp(action, "audioVolumeUp") == 0) {
        // Turn volume up by 10 percent
        NewVolume = Volume + 10;
    } else if (strcmp(action, "audioVolumeDown") == 0) {
        // Turn volume down by 10 percent
        NewVolume = Volume - 10;
    } else if (sscanf(action, "volume%d", &NewVolume) == 1) {
        // Nothing else needs to be done
    } else if (strcmp(action, "audioSpeakersB") == 0) {
        IsSpeakerBOn = !IsSpeakerBOn;
    } else if (strcmp(action, "off") == 0) {
        NewVolume = 10;
    }
}

void processAction(const char *action)
{
   int pin, state, red, green, blue;
   char channelNumber[8];
   
   Lcd.setCursor(0,0);
   Lcd.print(action);
   Lcd.print("                        ");

   // Do special processing for compound actions that initiate multiple
   // device control steps. Simple primitive actions such as turning
   // up the volume on the audio device are processed by sending a single
   // signal who's code is stored on the SD card in a file that is looked
   // up in EcSymTable.
   if (strcmp(action, "iTunes") == 0) {
       // Listen to iTunes
       if (State == S_ITUNES) return;
       State = S_ITUNES;
       turnVideoOff();
       turnAudioOn();
       processFile("aAux.txt");          // Set the input to aux -- it's connected to the airport
       turnSpeakerBOff();
       FloorLight.SetMode( RGB_LED::RAINBOW);
   } else if (strcmp(action, "mute") == 0) {
       // Keep track of the audio muted state
       processFile("aMute.txt");
   } else if (strcmp(action, "tv") == 0) {
       // Watch TV
       if (State == S_TV) return;
       State = S_TV;
       turnVideoOn();                   // Turn on video
       turnAudioOn();
       processFile("aVid.txt");       // Change to the video input on the audio device
       turnSpeakerBOff();
       processFile("sIn1.txt");         // TV is on the first switcher input
       FloorLight.SetMode( RGB_LED::FADE_OFF);
   } else if (strcmp(action, "appleTV") == 0) {
       // Watch Apple TV
       if (State == S_APPLE_TV) return;
       State = S_APPLE_TV;
       turnVideoOn();
       turnAudioOn();
       processFile("aVid.txt");
       turnSpeakerBOff();
       processFile("sIn2.txt");       // Apple TV is on the second switcher input
       processFile("Amenu.txt");      // Make sure the Apple TV is awake
       FloorLight.SetMode( RGB_LED::FADE_OFF);
   } else if (strcmp(action, "dvd") == 0) {
       // Watch a DVD
       if (State == S_DVD) return;
       State = S_DVD;
       turnVideoOn();
       turnAudioOn();
       processFile("aVid.txt");
       turnSpeakerBOff();
       processFile("sIn3.txt");       // Blu ray player is on the third switcher input
       FloorLight.SetMode( RGB_LED::FADE_OFF);
   } else if (strcmp(action, "kqed") == 0) {
       // Listen to KQED radio
       if (State == S_KQED) return;
       State = S_KQED;
       turnVideoOff();
       turnAudioOn();
       processFile("aCD.txt");        // The KQED feed is on the cd channel
       turnSpeakerBOff();
       FloorLight.SetMode( RGB_LED::RAINBOW);
   } else if (strcmp(action, "videoOn") == 0) {
       // Turn Video On
       turnVideoOn();
   } else if (strcmp(action, "videoOff") == 0) {
       turnVideoOff();
   } else if (strcmp(action, "videoIsOff") == 0) {
       IsVideoOn = false;
   } else if (strcmp(action, "videoIsOn") == 0) {
       IsVideoOn = true;
   } else if (strcmp(action, "audioVolumeUp") == 0) {
       // Turn the audio volume up by a fixed percentage
       setVolume( NewVolume); 
   } else if (strcmp(action, "audioVolumeDown") == 0) {
       // Turn the audio volume down a fixed percentage
       setVolume( NewVolume);
   } else if (sscanf(action, "volume%d", &NewVolume) == 1) {
       setVolume( NewVolume);
   } else if (strcmp(action, "off") == 0) {
       // Turn everything off
       turnSpeakerBOff();
       turnVideoOff();
       turnAudioOff();
       FloorLight.SetMode( RGB_LED::FADE_OFF);
       processFile("arcAny.txt");
       State = S_OFF;
   } else if (sscanf( action, "pin_%d_%d", &pin, &state) == 2) {
       // For debugging lighting
       digitalWrite( pin, state ? HIGH : LOW);
   } else if (strcmp( action, "rainbow") == 0) {
       // Turn rainbow lighting loop on
       FloorLight.SetMode( RGB_LED::RAINBOW);
   } else if (strcmp( action, "lightOff") == 0) {
       FloorLight.SetMode( RGB_LED::FADE_OFF);
   } else if (sscanf( action, "rgb_%d_%d_%d", &red, &green, &blue) == 3) {
       FloorLight.SetMode( RGB_LED::MANUAL);
       FloorLight.SetRGB( red, green, blue);
   } else if (strcmp( action, "cnn") == 0) {
       // CNN is on channel 56
       gotoChannel("56");
   } else if (sscanf( action, "channel%s", channelNumber) == 1) {
       // Go to the specified channel
       gotoChannel( channelNumber);
   } else {
       // Lookup the signal data file name in the symbol table
       const char *fileName = symTableLookup( action);
       if (fileName) {
           // Found it -- read and send the data
           processFile( fileName);
       } else {
           Lcd.setCursor(0,0);
           Lcd.print("Unknown: ");
           Lcd.print(action);
       }
   }
}


void loop() {
  FloorLight.Update();
  
  // listen for incoming clients
  WiFiClient client = WifiServer.available();
  char action[64];
  
  if (client) {
#ifdef DEBUG
    Serial.println("new client");
#endif
    // an http request ends with a blank line
    boolean isFirstLine = true;
    char lineBuffer[128];
    while (client.connected()) {
        if (readLine( client, lineBuffer) == 0) {
            char statusLine[128];
            
            // A blank line indicates the http request has ended.

            // send a standard http response header
            client.println("HTTP/1.1 200 OK");
            client.println("Content-Type: application/json");
            client.println("Connection: close");
            client.println();
            
            updateStatus( action);
            
            // Return a description of the status of the system after processAction is complete
            client.println("{");
            sprintf( statusLine, "  \"mute\" : \"%s\",", IsMuted ? "yes" : "no");
            client.println(statusLine);
            sprintf( statusLine, "  \"volume\" : \"%d\",", NewVolume);
            client.println(statusLine);
            sprintf( statusLine, "  \"speakerB\" : \"%s\",", IsSpeakerBOn ? "yes" : "no");
            client.println(statusLine);
            client.println("}");
            break;
        } else if (isFirstLine) {
            // Parse  the line for the action to take
            int i = 5;    // Skip over first 5 characters: "GET /"
            while (lineBuffer[i] != ' ') {
                action[i-5] = lineBuffer[i];
                i++;
            }
            action[i-5] = '\0';
            isFirstLine = false;
        }
    }
    // give the web browser time to receive the data
    delay(1);
    
    // close the connection:
#ifdef DEBUG
    Serial.println("Closing connection");
#endif
    client.stop();
#ifdef DEBUG
    Serial.println("client disonnected");
    Serial.print("Action: ");
    Serial.println(action);
#endif
    processAction(action);

  }
  // Add a heartbeat indicator
  if (millis() - LastMillis > 1000) {
      // A second has gone by since the last update
      LastMillis = millis();
      HeartBeat = !HeartBeat;
      Lcd.setCursor(14, 1);
      Lcd.print(HeartBeatStr[HeartBeat]);
  }
}


void printWifiStatus() {
  // print the SSID of the network you're attached to:
  Serial.print("SSID: ");
  Serial.println(WiFi.SSID());

  // print your WiFi shield's IP address:
  MyAddress = WiFi.localIP();
  Serial.print("IP Address: ");
  Serial.println(MyAddress);

#ifdef DEBUG
  // print the received signal strength:
  long rssi = WiFi.RSSI();
  Serial.print("signal strength (RSSI):");
  Serial.print(rssi);
  Serial.println(" dBm");
#endif
}

