{{
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SL32 Integer Engine
//
// Author: Kwabena W. Agyeman
// Updated: 8/28/2010
// Designed For: P8X32A
// Version: 1.0
//
// Copyright (c) 2010 Kwabena W. Agyeman
// See end of file for terms of use.
//
// Update History:
//
// v1.0 - Original release - 7/27/2010.
//
// For each included copy of this object only one spin interpreter should access it at a time.
//
// Nyamekye,
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}}

PUB sin(angle, radius) '' 9 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the sin of an angle.
'' //
'' // Returns sin of an angle scaled by the radius. I.E. (radius * sin(angle)).
'' //
'' // Angle - The angle to take the sin of in degrees.
'' // Radius - The radius (hypotenuse) of the polar plane triangle.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return ((sinTable(scaleAngle(angle)) * ((radius <# 32_767) #> 0)) / $FF_FF)

PUB cos(angle, radius) '' 14 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the cos of an angle.
'' //
'' // Returns cos of an angle scaled by the radius. I.E. (radius * cos(angle)).
'' //
'' // Angle - The angle to take the cos of in degrees.
'' // Radius - The radius (hypotenuse) of the polar plane triangle.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return sin((90 + angle), radius)

PUB tan(angle, radius) '' 9 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the tan of an angle.
'' //
'' // Returns tan of an angle scaled by the radius. I.E. (radius * tan(angle)).
'' //
'' // Angle - The angle to take the tan of in degrees.
'' // Radius - The radius (hypotenuse) of the polar plane triangle.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return ((sinTable(scaleAngle(angle)) * ((radius <# 32_767) #> 0)) / sinTable(scaleAngle(90 + angle)))

PUB arcSin(yLength, radius) | mirrorFlag, minValue, maxValue '' 8 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the arc sin of an angle.
'' //
'' // Returns the angle in which ((radius) * sin(angle)) = yLength.
'' //
'' // yLength - The opposite side of a right triangle. Opposite to where the angle is calculated.
'' // Radius - The radius (hypotenuse) of the polar plane triangle.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  radius := ((radius <# 32_767) #> 0)
  yLength := ((yLength <# radius) #> -radius)

  mirrorFlag := (yLength < 0)
  yLength := (((||yLength) * $FF_FF) / radius)

  minValue := 0
  maxValue := 20_48

  repeat

    result := ((minValue + maxValue) >> 1)

    if(yLength => word[$E0_00][result++])

      minValue := result

    else

      maxValue := (result - 2)

  until(minValue > maxValue)

  return (((result * 90) >> 11) * (mirrorFlag | 1))

PUB arcCos(xLength, radius) '' 13 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the arc cos of an angle.
'' //
'' // Returns the angle in which ((radius) * cos(angle)) = xLength.
'' //
'' // xLength - The addjacent side of a right triangle. Addjacent to where the angle is calculated.
'' // Radius - The radius (hypotenuse) of the polar plane triangle.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return (90 - arcSin(xLength, radius))

PUB arcTan(yLength, xLength) '' 13 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the arc tan of an angle.
'' //
'' // Returns the angle in which tan(angle) = (yLength / xLength).
'' //
'' // yLength - The opposite side of a right triangle. Opposite to where the angle is calculated.
'' // xLength - The addjacent side of a right triangle. Addjacent to where the angle is calculated.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return arcSin(yLength, polarRadius(yLength, xLength))

PUB polarAngle(yLength, xLength) '' 18 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the angle of a triangle in the polar plane using its x and y side lengths (arcTan2).
'' //
'' // Returns the absolute angle in which tan(angle) = (yLength / xLength).
'' //
'' // yLength - The opposite side of a right triangle. Opposite to where the angle is calculated.
'' // xLength - The addjacent side of a right triangle. Addjacent to where the angle is calculated.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  if(xLength < 0)

    result := (180 * ((yLength < 0) | 1))

    -xLength

    -yLength

  result += arcTan(yLength, xLength)

PUB polarRadius(yLength, xLength) '' 5 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Computes the radius (hypotenuse) of a triangle in the polar plane using its x and y side lengths (pythagorian theram).
'' //
'' // Returns the radius (hypotenuse) of a triangle in the polar plane.
'' //
'' // yLength - The opposite side of a right triangle. Opposite to where the angle is calculated.
'' // xLength - The addjacent side of a right triangle. Addjacent to where the angle is calculated.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return (^^((xLength * xLength) + (yLength * yLength)))

PUB power(base, exponent) '' 5 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Returns a base rasied by an exponent.
'' //
'' // Base - A positive or negative integer to be raised by an exponent.
'' // Exponent - A zero or positive only power to raise the base by.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  result -= (exponent => 0)
  repeat while(exponent)

    if(exponent & 1)
      result *= base

    base *= base
    exponent >>= 1

PUB random(seedValue, maxValue, minValue) '' 6 Stack Longs

'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
'' // Returns a pseudo random number between a max value and min value calculated from a seed value.
'' //
'' // SeedValue - A seed value to use for randomization. The system counter is a good default choice.
'' // MaxValue - The maximum value the pseudo random number can take on.
'' // MinValue - The minimum value the pseudo random number can take on.
'' ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  return (((||(seedValue?)) // (||(maxValue - minValue))) + (minValue <# maxValue))

PRI scaleAngle(angle) ' 4 Stack Longs

  return (((angle // 360) << 13) / 360)

PRI sinTable(angle) ' 4 Stack Longs

  return (word[((angle * ((not(not(angle & $8_00))) | 1)) | constant($E0_00 >> 1)) << 1] * ((not(not(angle & $10_00))) | 1))

DAT

' //////////////////////Sine Table/////////////////////////////////////////////////////////////////////////////////////////////

  '  word       0       Index - 0
  '  word       50      Index - 1
  '  word       101     Index - 2
  '  word       151     Index - 3
  '  word       201     Index - 4
  '  word       251     Index - 5
  '  word       302     Index - 6
  '  word       352     Index - 7
  '  word       402     Index - 8
  '  word       452     Index - 9
  '  word       503     Index - 10
  '  word       553     Index - 11
  '  word       603     Index - 12
  '  word       653     Index - 13
  '  word       704     Index - 14
  '  word       754     Index - 15
  '  word       804     Index - 16
  '  word       854     Index - 17
  '  word       905     Index - 18
  '  word       955     Index - 19
  '  word       1005    Index - 20
  '  word       1056    Index - 21
  '  word       1106    Index - 22
  '  word       1156    Index - 23
  '  word       1206    Index - 24
  '  word       1257    Index - 25
  '  word       1307    Index - 26
  '  word       1357    Index - 27
  '  word       1407    Index - 28
  '  word       1458    Index - 29
  '  word       1508    Index - 30
  '  word       1558    Index - 31
  '  word       1608    Index - 32
  '  word       1659    Index - 33
  '  word       1709    Index - 34
  '  word       1759    Index - 35
  '  word       1809    Index - 36
  '  word       1860    Index - 37
  '  word       1910    Index - 38
  '  word       1960    Index - 39
  '  word       2010    Index - 40
  '  word       2061    Index - 41
  '  word       2111    Index - 42
  '  word       2161    Index - 43
  '  word       2211    Index - 44
  '  word       2261    Index - 45
  '  word       2312    Index - 46
  '  word       2362    Index - 47
  '  word       2412    Index - 48
  '  word       2462    Index - 49
  '  word       2513    Index - 50
  '  word       2563    Index - 51
  '  word       2613    Index - 52
  '  word       2663    Index - 53
  '  word       2714    Index - 54
  '  word       2764    Index - 55
  '  word       2814    Index - 56
  '  word       2864    Index - 57
  '  word       2914    Index - 58
  '  word       2965    Index - 59
  '  word       3015    Index - 60
  '  word       3065    Index - 61
  '  word       3115    Index - 62
  '  word       3165    Index - 63
  '  word       3216    Index - 64
  '  word       3266    Index - 65
  '  word       3316    Index - 66
  '  word       3366    Index - 67
  '  word       3416    Index - 68
  '  word       3467    Index - 69
  '  word       3517    Index - 70
  '  word       3567    Index - 71
  '  word       3617    Index - 72
  '  word       3667    Index - 73
  '  word       3718    Index - 74
  '  word       3768    Index - 75
  '  word       3818    Index - 76
  '  word       3868    Index - 77
  '  word       3918    Index - 78
  '  word       3968    Index - 79
  '  word       4019    Index - 80
  '  word       4069    Index - 81
  '  word       4119    Index - 82
  '  word       4169    Index - 83
  '  word       4219    Index - 84
  '  word       4269    Index - 85
  '  word       4320    Index - 86
  '  word       4370    Index - 87
  '  word       4420    Index - 88
  '  word       4470    Index - 89
  '  word       4520    Index - 90
  '  word       4570    Index - 91
  '  word       4621    Index - 92
  '  word       4671    Index - 93
  '  word       4721    Index - 94
  '  word       4771    Index - 95
  '  word       4821    Index - 96
  '  word       4871    Index - 97
  '  word       4921    Index - 98
  '  word       4971    Index - 99
  '  word       5022    Index - 100
  '  word       5072    Index - 101
  '  word       5122    Index - 102
  '  word       5172    Index - 103
  '  word       5222    Index - 104
  '  word       5272    Index - 105
  '  word       5322    Index - 106
  '  word       5372    Index - 107
  '  word       5422    Index - 108
  '  word       5472    Index - 109
  '  word       5523    Index - 110
  '  word       5573    Index - 111
  '  word       5623    Index - 112
  '  word       5673    Index - 113
  '  word       5723    Index - 114
  '  word       5773    Index - 115
  '  word       5823    Index - 116
  '  word       5873    Index - 117
  '  word       5923    Index - 118
  '  word       5973    Index - 119
  '  word       6023    Index - 120
  '  word       6073    Index - 121
  '  word       6123    Index - 122
  '  word       6173    Index - 123
  '  word       6223    Index - 124
  '  word       6273    Index - 125
  '  word       6324    Index - 126
  '  word       6374    Index - 127
  '  word       6424    Index - 128
  '  word       6474    Index - 129
  '  word       6524    Index - 130
  '  word       6574    Index - 131
  '  word       6624    Index - 132
  '  word       6674    Index - 133
  '  word       6724    Index - 134
  '  word       6774    Index - 135
  '  word       6824    Index - 136
  '  word       6874    Index - 137
  '  word       6924    Index - 138
  '  word       6974    Index - 139
  '  word       7024    Index - 140
  '  word       7074    Index - 141
  '  word       7123    Index - 142
  '  word       7173    Index - 143
  '  word       7223    Index - 144
  '  word       7273    Index - 145
  '  word       7323    Index - 146
  '  word       7373    Index - 147
  '  word       7423    Index - 148
  '  word       7473    Index - 149
  '  word       7523    Index - 150
  '  word       7573    Index - 151
  '  word       7623    Index - 152
  '  word       7673    Index - 153
  '  word       7723    Index - 154
  '  word       7773    Index - 155
  '  word       7823    Index - 156
  '  word       7873    Index - 157
  '  word       7922    Index - 158
  '  word       7972    Index - 159
  '  word       8022    Index - 160
  '  word       8072    Index - 161
  '  word       8122    Index - 162
  '  word       8172    Index - 163
  '  word       8222    Index - 164
  '  word       8272    Index - 165
  '  word       8321    Index - 166
  '  word       8371    Index - 167
  '  word       8421    Index - 168
  '  word       8471    Index - 169
  '  word       8521    Index - 170
  '  word       8571    Index - 171
  '  word       8620    Index - 172
  '  word       8670    Index - 173
  '  word       8720    Index - 174
  '  word       8770    Index - 175
  '  word       8820    Index - 176
  '  word       8870    Index - 177
  '  word       8919    Index - 178
  '  word       8969    Index - 179
  '  word       9019    Index - 180
  '  word       9069    Index - 181
  '  word       9118    Index - 182
  '  word       9168    Index - 183
  '  word       9218    Index - 184
  '  word       9268    Index - 185
  '  word       9318    Index - 186
  '  word       9367    Index - 187
  '  word       9417    Index - 188
  '  word       9467    Index - 189
  '  word       9517    Index - 190
  '  word       9566    Index - 191
  '  word       9616    Index - 192
  '  word       9666    Index - 193
  '  word       9715    Index - 194
  '  word       9765    Index - 195
  '  word       9815    Index - 196
  '  word       9865    Index - 197
  '  word       9914    Index - 198
  '  word       9964    Index - 199
  '  word       10014   Index - 200
  '  word       10063   Index - 201
  '  word       10113   Index - 202
  '  word       10163   Index - 203
  '  word       10212   Index - 204
  '  word       10262   Index - 205
  '  word       10312   Index - 206
  '  word       10361   Index - 207
  '  word       10411   Index - 208
  '  word       10460   Index - 209
  '  word       10510   Index - 210
  '  word       10560   Index - 211
  '  word       10609   Index - 212
  '  word       10659   Index - 213
  '  word       10708   Index - 214
  '  word       10758   Index - 215
  '  word       10808   Index - 216
  '  word       10857   Index - 217
  '  word       10907   Index - 218
  '  word       10956   Index - 219
  '  word       11006   Index - 220
  '  word       11055   Index - 221
  '  word       11105   Index - 222
  '  word       11154   Index - 223
  '  word       11204   Index - 224
  '  word       11254   Index - 225
  '  word       11303   Index - 226
  '  word       11353   Index - 227
  '  word       11402   Index - 228
  '  word       11452   Index - 229
  '  word       11501   Index - 230
  '  word       11550   Index - 231
  '  word       11600   Index - 232
  '  word       11649   Index - 233
  '  word       11699   Index - 234
  '  word       11748   Index - 235
  '  word       11798   Index - 236
  '  word       11847   Index - 237
  '  word       11897   Index - 238
  '  word       11946   Index - 239
  '  word       11996   Index - 240
  '  word       12045   Index - 241
  '  word       12094   Index - 242
  '  word       12144   Index - 243
  '  word       12193   Index - 244
  '  word       12243   Index - 245
  '  word       12292   Index - 246
  '  word       12341   Index - 247
  '  word       12391   Index - 248
  '  word       12440   Index - 249
  '  word       12489   Index - 250
  '  word       12539   Index - 251
  '  word       12588   Index - 252
  '  word       12637   Index - 253
  '  word       12687   Index - 254
  '  word       12736   Index - 255
  '  word       12785   Index - 256
  '  word       12835   Index - 257
  '  word       12884   Index - 258
  '  word       12933   Index - 259
  '  word       12982   Index - 260
  '  word       13032   Index - 261
  '  word       13081   Index - 262
  '  word       13130   Index - 263
  '  word       13179   Index - 264
  '  word       13229   Index - 265
  '  word       13278   Index - 266
  '  word       13327   Index - 267
  '  word       13376   Index - 268
  '  word       13425   Index - 269
  '  word       13475   Index - 270
  '  word       13524   Index - 271
  '  word       13573   Index - 272
  '  word       13622   Index - 273
  '  word       13671   Index - 274
  '  word       13721   Index - 275
  '  word       13770   Index - 276
  '  word       13819   Index - 277
  '  word       13868   Index - 278
  '  word       13917   Index - 279
  '  word       13966   Index - 280
  '  word       14015   Index - 281
  '  word       14064   Index - 282
  '  word       14113   Index - 283
  '  word       14163   Index - 284
  '  word       14212   Index - 285
  '  word       14261   Index - 286
  '  word       14310   Index - 287
  '  word       14359   Index - 288
  '  word       14408   Index - 289
  '  word       14457   Index - 290
  '  word       14506   Index - 291
  '  word       14555   Index - 292
  '  word       14604   Index - 293
  '  word       14653   Index - 294
  '  word       14702   Index - 295
  '  word       14751   Index - 296
  '  word       14800   Index - 297
  '  word       14849   Index - 298
  '  word       14898   Index - 299
  '  word       14947   Index - 300
  '  word       14996   Index - 301
  '  word       15045   Index - 302
  '  word       15093   Index - 303
  '  word       15142   Index - 304
  '  word       15191   Index - 305
  '  word       15240   Index - 306
  '  word       15289   Index - 307
  '  word       15338   Index - 308
  '  word       15387   Index - 309
  '  word       15436   Index - 310
  '  word       15485   Index - 311
  '  word       15533   Index - 312
  '  word       15582   Index - 313
  '  word       15631   Index - 314
  '  word       15680   Index - 315
  '  word       15729   Index - 316
  '  word       15777   Index - 317
  '  word       15826   Index - 318
  '  word       15875   Index - 319
  '  word       15924   Index - 320
  '  word       15972   Index - 321
  '  word       16021   Index - 322
  '  word       16070   Index - 323
  '  word       16119   Index - 324
  '  word       16167   Index - 325
  '  word       16216   Index - 326
  '  word       16265   Index - 327
  '  word       16313   Index - 328
  '  word       16362   Index - 329
  '  word       16411   Index - 330
  '  word       16459   Index - 331
  '  word       16508   Index - 332
  '  word       16557   Index - 333
  '  word       16605   Index - 334
  '  word       16654   Index - 335
  '  word       16703   Index - 336
  '  word       16751   Index - 337
  '  word       16800   Index - 338
  '  word       16848   Index - 339
  '  word       16897   Index - 340
  '  word       16946   Index - 341
  '  word       16994   Index - 342
  '  word       17043   Index - 343
  '  word       17091   Index - 344
  '  word       17140   Index - 345
  '  word       17188   Index - 346
  '  word       17237   Index - 347
  '  word       17285   Index - 348
  '  word       17334   Index - 349
  '  word       17382   Index - 350
  '  word       17431   Index - 351
  '  word       17479   Index - 352
  '  word       17527   Index - 353
  '  word       17576   Index - 354
  '  word       17624   Index - 355
  '  word       17673   Index - 356
  '  word       17721   Index - 357
  '  word       17769   Index - 358
  '  word       17818   Index - 359
  '  word       17866   Index - 360
  '  word       17915   Index - 361
  '  word       17963   Index - 362
  '  word       18011   Index - 363
  '  word       18060   Index - 364
  '  word       18108   Index - 365
  '  word       18156   Index - 366
  '  word       18205   Index - 367
  '  word       18253   Index - 368
  '  word       18301   Index - 369
  '  word       18349   Index - 370
  '  word       18398   Index - 371
  '  word       18446   Index - 372
  '  word       18494   Index - 373
  '  word       18542   Index - 374
  '  word       18590   Index - 375
  '  word       18639   Index - 376
  '  word       18687   Index - 377
  '  word       18735   Index - 378
  '  word       18783   Index - 379
  '  word       18831   Index - 380
  '  word       18879   Index - 381
  '  word       18928   Index - 382
  '  word       18976   Index - 383
  '  word       19024   Index - 384
  '  word       19072   Index - 385
  '  word       19120   Index - 386
  '  word       19168   Index - 387
  '  word       19216   Index - 388
  '  word       19264   Index - 389
  '  word       19312   Index - 390
  '  word       19360   Index - 391
  '  word       19408   Index - 392
  '  word       19456   Index - 393
  '  word       19504   Index - 394
  '  word       19552   Index - 395
  '  word       19600   Index - 396
  '  word       19648   Index - 397
  '  word       19696   Index - 398
  '  word       19744   Index - 399
  '  word       19792   Index - 400
  '  word       19840   Index - 401
  '  word       19888   Index - 402
  '  word       19936   Index - 403
  '  word       19984   Index - 404
  '  word       20031   Index - 405
  '  word       20079   Index - 406
  '  word       20127   Index - 407
  '  word       20175   Index - 408
  '  word       20223   Index - 409
  '  word       20271   Index - 410
  '  word       20318   Index - 411
  '  word       20366   Index - 412
  '  word       20414   Index - 413
  '  word       20462   Index - 414
  '  word       20509   Index - 415
  '  word       20557   Index - 416
  '  word       20605   Index - 417
  '  word       20653   Index - 418
  '  word       20700   Index - 419
  '  word       20748   Index - 420
  '  word       20796   Index - 421
  '  word       20843   Index - 422
  '  word       20891   Index - 423
  '  word       20939   Index - 424
  '  word       20986   Index - 425
  '  word       21034   Index - 426
  '  word       21081   Index - 427
  '  word       21129   Index - 428
  '  word       21177   Index - 429
  '  word       21224   Index - 430
  '  word       21272   Index - 431
  '  word       21319   Index - 432
  '  word       21367   Index - 433
  '  word       21414   Index - 434
  '  word       21462   Index - 435
  '  word       21509   Index - 436
  '  word       21557   Index - 437
  '  word       21604   Index - 438
  '  word       21652   Index - 439
  '  word       21699   Index - 440
  '  word       21746   Index - 441
  '  word       21794   Index - 442
  '  word       21841   Index - 443
  '  word       21889   Index - 444
  '  word       21936   Index - 445
  '  word       21983   Index - 446
  '  word       22031   Index - 447
  '  word       22078   Index - 448
  '  word       22125   Index - 449
  '  word       22173   Index - 450
  '  word       22220   Index - 451
  '  word       22267   Index - 452
  '  word       22315   Index - 453
  '  word       22362   Index - 454
  '  word       22409   Index - 455
  '  word       22456   Index - 456
  '  word       22503   Index - 457
  '  word       22551   Index - 458
  '  word       22598   Index - 459
  '  word       22645   Index - 460
  '  word       22692   Index - 461
  '  word       22739   Index - 462
  '  word       22786   Index - 463
  '  word       22834   Index - 464
  '  word       22881   Index - 465
  '  word       22928   Index - 466
  '  word       22975   Index - 467
  '  word       23022   Index - 468
  '  word       23069   Index - 469
  '  word       23116   Index - 470
  '  word       23163   Index - 471
  '  word       23210   Index - 472
  '  word       23257   Index - 473
  '  word       23304   Index - 474
  '  word       23351   Index - 475
  '  word       23398   Index - 476
  '  word       23445   Index - 477
  '  word       23492   Index - 478
  '  word       23539   Index - 479
  '  word       23586   Index - 480
  '  word       23633   Index - 481
  '  word       23679   Index - 482
  '  word       23726   Index - 483
  '  word       23773   Index - 484
  '  word       23820   Index - 485
  '  word       23867   Index - 486
  '  word       23914   Index - 487
  '  word       23960   Index - 488
  '  word       24007   Index - 489
  '  word       24054   Index - 490
  '  word       24101   Index - 491
  '  word       24147   Index - 492
  '  word       24194   Index - 493
  '  word       24241   Index - 494
  '  word       24288   Index - 495
  '  word       24334   Index - 496
  '  word       24381   Index - 497
  '  word       24428   Index - 498
  '  word       24474   Index - 499
  '  word       24521   Index - 500
  '  word       24567   Index - 501
  '  word       24614   Index - 502
  '  word       24661   Index - 503
  '  word       24707   Index - 504
  '  word       24754   Index - 505
  '  word       24800   Index - 506
  '  word       24847   Index - 507
  '  word       24893   Index - 508
  '  word       24940   Index - 509
  '  word       24986   Index - 510
  '  word       25033   Index - 511
  '  word       25079   Index - 512
  '  word       25126   Index - 513
  '  word       25172   Index - 514
  '  word       25218   Index - 515
  '  word       25265   Index - 516
  '  word       25311   Index - 517
  '  word       25358   Index - 518
  '  word       25404   Index - 519
  '  word       25450   Index - 520
  '  word       25497   Index - 521
  '  word       25543   Index - 522
  '  word       25589   Index - 523
  '  word       25635   Index - 524
  '  word       25682   Index - 525
  '  word       25728   Index - 526
  '  word       25774   Index - 527
  '  word       25820   Index - 528
  '  word       25866   Index - 529
  '  word       25913   Index - 530
  '  word       25959   Index - 531
  '  word       26005   Index - 532
  '  word       26051   Index - 533
  '  word       26097   Index - 534
  '  word       26143   Index - 535
  '  word       26189   Index - 536
  '  word       26235   Index - 537
  '  word       26281   Index - 538
  '  word       26328   Index - 539
  '  word       26374   Index - 540
  '  word       26420   Index - 541
  '  word       26466   Index - 542
  '  word       26512   Index - 543
  '  word       26557   Index - 544
  '  word       26603   Index - 545
  '  word       26649   Index - 546
  '  word       26695   Index - 547
  '  word       26741   Index - 548
  '  word       26787   Index - 549
  '  word       26833   Index - 550
  '  word       26879   Index - 551
  '  word       26925   Index - 552
  '  word       26970   Index - 553
  '  word       27016   Index - 554
  '  word       27062   Index - 555
  '  word       27108   Index - 556
  '  word       27154   Index - 557
  '  word       27199   Index - 558
  '  word       27245   Index - 559
  '  word       27291   Index - 560
  '  word       27336   Index - 561
  '  word       27382   Index - 562
  '  word       27428   Index - 563
  '  word       27473   Index - 564
  '  word       27519   Index - 565
  '  word       27565   Index - 566
  '  word       27610   Index - 567
  '  word       27656   Index - 568
  '  word       27701   Index - 569
  '  word       27747   Index - 570
  '  word       27792   Index - 571
  '  word       27838   Index - 572
  '  word       27883   Index - 573
  '  word       27929   Index - 574
  '  word       27974   Index - 575
  '  word       28020   Index - 576
  '  word       28065   Index - 577
  '  word       28111   Index - 578
  '  word       28156   Index - 579
  '  word       28201   Index - 580
  '  word       28247   Index - 581
  '  word       28292   Index - 582
  '  word       28337   Index - 583
  '  word       28383   Index - 584
  '  word       28428   Index - 585
  '  word       28473   Index - 586
  '  word       28519   Index - 587
  '  word       28564   Index - 588
  '  word       28609   Index - 589
  '  word       28654   Index - 590
  '  word       28700   Index - 591
  '  word       28745   Index - 592
  '  word       28790   Index - 593
  '  word       28835   Index - 594
  '  word       28880   Index - 595
  '  word       28925   Index - 596
  '  word       28970   Index - 597
  '  word       29015   Index - 598
  '  word       29061   Index - 599
  '  word       29106   Index - 600
  '  word       29151   Index - 601
  '  word       29196   Index - 602
  '  word       29241   Index - 603
  '  word       29286   Index - 604
  '  word       29331   Index - 605
  '  word       29375   Index - 606
  '  word       29420   Index - 607
  '  word       29465   Index - 608
  '  word       29510   Index - 609
  '  word       29555   Index - 610
  '  word       29600   Index - 611
  '  word       29645   Index - 612
  '  word       29690   Index - 613
  '  word       29734   Index - 614
  '  word       29779   Index - 615
  '  word       29824   Index - 616
  '  word       29869   Index - 617
  '  word       29913   Index - 618
  '  word       29958   Index - 619
  '  word       30003   Index - 620
  '  word       30047   Index - 621
  '  word       30092   Index - 622
  '  word       30137   Index - 623
  '  word       30181   Index - 624
  '  word       30226   Index - 625
  '  word       30271   Index - 626
  '  word       30315   Index - 627
  '  word       30360   Index - 628
  '  word       30404   Index - 629
  '  word       30449   Index - 630
  '  word       30493   Index - 631
  '  word       30538   Index - 632
  '  word       30582   Index - 633
  '  word       30627   Index - 634
  '  word       30671   Index - 635
  '  word       30716   Index - 636
  '  word       30760   Index - 637
  '  word       30804   Index - 638
  '  word       30849   Index - 639
  '  word       30893   Index - 640
  '  word       30937   Index - 641
  '  word       30982   Index - 642
  '  word       31026   Index - 643
  '  word       31070   Index - 644
  '  word       31114   Index - 645
  '  word       31159   Index - 646
  '  word       31203   Index - 647
  '  word       31247   Index - 648
  '  word       31291   Index - 649
  '  word       31335   Index - 650
  '  word       31380   Index - 651
  '  word       31424   Index - 652
  '  word       31468   Index - 653
  '  word       31512   Index - 654
  '  word       31556   Index - 655
  '  word       31600   Index - 656
  '  word       31644   Index - 657
  '  word       31688   Index - 658
  '  word       31732   Index - 659
  '  word       31776   Index - 660
  '  word       31820   Index - 661
  '  word       31864   Index - 662
  '  word       31908   Index - 663
  '  word       31952   Index - 664
  '  word       31995   Index - 665
  '  word       32039   Index - 666
  '  word       32083   Index - 667
  '  word       32127   Index - 668
  '  word       32171   Index - 669
  '  word       32215   Index - 670
  '  word       32258   Index - 671
  '  word       32302   Index - 672
  '  word       32346   Index - 673
  '  word       32390   Index - 674
  '  word       32433   Index - 675
  '  word       32477   Index - 676
  '  word       32521   Index - 677
  '  word       32564   Index - 678
  '  word       32608   Index - 679
  '  word       32651   Index - 680
  '  word       32695   Index - 681
  '  word       32738   Index - 682
  '  word       32782   Index - 683
  '  word       32826   Index - 684
  '  word       32869   Index - 685
  '  word       32912   Index - 686
  '  word       32956   Index - 687
  '  word       32999   Index - 688
  '  word       33043   Index - 689
  '  word       33086   Index - 690
  '  word       33130   Index - 691
  '  word       33173   Index - 692
  '  word       33216   Index - 693
  '  word       33260   Index - 694
  '  word       33303   Index - 695
  '  word       33346   Index - 696
  '  word       33389   Index - 697
  '  word       33433   Index - 698
  '  word       33476   Index - 699
  '  word       33519   Index - 700
  '  word       33562   Index - 701
  '  word       33605   Index - 702
  '  word       33649   Index - 703
  '  word       33692   Index - 704
  '  word       33735   Index - 705
  '  word       33778   Index - 706
  '  word       33821   Index - 707
  '  word       33864   Index - 708
  '  word       33907   Index - 709
  '  word       33950   Index - 710
  '  word       33993   Index - 711
  '  word       34036   Index - 712
  '  word       34079   Index - 713
  '  word       34122   Index - 714
  '  word       34165   Index - 715
  '  word       34208   Index - 716
  '  word       34251   Index - 717
  '  word       34293   Index - 718
  '  word       34336   Index - 719
  '  word       34379   Index - 720
  '  word       34422   Index - 721
  '  word       34465   Index - 722
  '  word       34507   Index - 723
  '  word       34550   Index - 724
  '  word       34593   Index - 725
  '  word       34635   Index - 726
  '  word       34678   Index - 727
  '  word       34721   Index - 728
  '  word       34763   Index - 729
  '  word       34806   Index - 730
  '  word       34848   Index - 731
  '  word       34891   Index - 732
  '  word       34934   Index - 733
  '  word       34976   Index - 734
  '  word       35019   Index - 735
  '  word       35061   Index - 736
  '  word       35104   Index - 737
  '  word       35146   Index - 738
  '  word       35188   Index - 739
  '  word       35231   Index - 740
  '  word       35273   Index - 741
  '  word       35315   Index - 742
  '  word       35358   Index - 743
  '  word       35400   Index - 744
  '  word       35442   Index - 745
  '  word       35485   Index - 746
  '  word       35527   Index - 747
  '  word       35569   Index - 748
  '  word       35611   Index - 749
  '  word       35654   Index - 750
  '  word       35696   Index - 751
  '  word       35738   Index - 752
  '  word       35780   Index - 753
  '  word       35822   Index - 754
  '  word       35864   Index - 755
  '  word       35906   Index - 756
  '  word       35948   Index - 757
  '  word       35990   Index - 758
  '  word       36032   Index - 759
  '  word       36074   Index - 760
  '  word       36116   Index - 761
  '  word       36158   Index - 762
  '  word       36200   Index - 763
  '  word       36242   Index - 764
  '  word       36284   Index - 765
  '  word       36326   Index - 766
  '  word       36367   Index - 767
  '  word       36409   Index - 768
  '  word       36451   Index - 769
  '  word       36493   Index - 770
  '  word       36535   Index - 771
  '  word       36576   Index - 772
  '  word       36618   Index - 773
  '  word       36660   Index - 774
  '  word       36701   Index - 775
  '  word       36743   Index - 776
  '  word       36785   Index - 777
  '  word       36826   Index - 778
  '  word       36868   Index - 779
  '  word       36909   Index - 780
  '  word       36951   Index - 781
  '  word       36992   Index - 782
  '  word       37034   Index - 783
  '  word       37075   Index - 784
  '  word       37117   Index - 785
  '  word       37158   Index - 786
  '  word       37199   Index - 787
  '  word       37241   Index - 788
  '  word       37282   Index - 789
  '  word       37324   Index - 790
  '  word       37365   Index - 791
  '  word       37406   Index - 792
  '  word       37447   Index - 793
  '  word       37489   Index - 794
  '  word       37530   Index - 795
  '  word       37571   Index - 796
  '  word       37612   Index - 797
  '  word       37653   Index - 798
  '  word       37694   Index - 799
  '  word       37736   Index - 800
  '  word       37777   Index - 801
  '  word       37818   Index - 802
  '  word       37859   Index - 803
  '  word       37900   Index - 804
  '  word       37941   Index - 805
  '  word       37982   Index - 806
  '  word       38023   Index - 807
  '  word       38064   Index - 808
  '  word       38105   Index - 809
  '  word       38145   Index - 810
  '  word       38186   Index - 811
  '  word       38227   Index - 812
  '  word       38268   Index - 813
  '  word       38309   Index - 814
  '  word       38350   Index - 815
  '  word       38390   Index - 816
  '  word       38431   Index - 817
  '  word       38472   Index - 818
  '  word       38512   Index - 819
  '  word       38553   Index - 820
  '  word       38594   Index - 821
  '  word       38634   Index - 822
  '  word       38675   Index - 823
  '  word       38715   Index - 824
  '  word       38756   Index - 825
  '  word       38797   Index - 826
  '  word       38837   Index - 827
  '  word       38877   Index - 828
  '  word       38918   Index - 829
  '  word       38958   Index - 830
  '  word       38999   Index - 831
  '  word       39039   Index - 832
  '  word       39080   Index - 833
  '  word       39120   Index - 834
  '  word       39160   Index - 835
  '  word       39200   Index - 836
  '  word       39241   Index - 837
  '  word       39281   Index - 838
  '  word       39321   Index - 839
  '  word       39361   Index - 840
  '  word       39402   Index - 841
  '  word       39442   Index - 842
  '  word       39482   Index - 843
  '  word       39522   Index - 844
  '  word       39562   Index - 845
  '  word       39602   Index - 846
  '  word       39642   Index - 847
  '  word       39682   Index - 848
  '  word       39722   Index - 849
  '  word       39762   Index - 850
  '  word       39802   Index - 851
  '  word       39842   Index - 852
  '  word       39882   Index - 853
  '  word       39922   Index - 854
  '  word       39962   Index - 855
  '  word       40001   Index - 856
  '  word       40041   Index - 857
  '  word       40081   Index - 858
  '  word       40121   Index - 859
  '  word       40161   Index - 860
  '  word       40200   Index - 861
  '  word       40240   Index - 862
  '  word       40280   Index - 863
  '  word       40319   Index - 864
  '  word       40359   Index - 865
  '  word       40398   Index - 866
  '  word       40438   Index - 867
  '  word       40478   Index - 868
  '  word       40517   Index - 869
  '  word       40557   Index - 870
  '  word       40596   Index - 871
  '  word       40635   Index - 872
  '  word       40675   Index - 873
  '  word       40714   Index - 874
  '  word       40754   Index - 875
  '  word       40793   Index - 876
  '  word       40832   Index - 877
  '  word       40872   Index - 878
  '  word       40911   Index - 879
  '  word       40950   Index - 880
  '  word       40989   Index - 881
  '  word       41029   Index - 882
  '  word       41068   Index - 883
  '  word       41107   Index - 884
  '  word       41146   Index - 885
  '  word       41185   Index - 886
  '  word       41224   Index - 887
  '  word       41263   Index - 888
  '  word       41302   Index - 889
  '  word       41341   Index - 890
  '  word       41380   Index - 891
  '  word       41419   Index - 892
  '  word       41458   Index - 893
  '  word       41497   Index - 894
  '  word       41536   Index - 895
  '  word       41575   Index - 896
  '  word       41614   Index - 897
  '  word       41653   Index - 898
  '  word       41691   Index - 899
  '  word       41730   Index - 900
  '  word       41769   Index - 901
  '  word       41808   Index - 902
  '  word       41846   Index - 903
  '  word       41885   Index - 904
  '  word       41924   Index - 905
  '  word       41962   Index - 906
  '  word       42001   Index - 907
  '  word       42039   Index - 908
  '  word       42078   Index - 909
  '  word       42117   Index - 910
  '  word       42155   Index - 911
  '  word       42194   Index - 912
  '  word       42232   Index - 913
  '  word       42270   Index - 914
  '  word       42309   Index - 915
  '  word       42347   Index - 916
  '  word       42385   Index - 917
  '  word       42424   Index - 918
  '  word       42462   Index - 919
  '  word       42500   Index - 920
  '  word       42539   Index - 921
  '  word       42577   Index - 922
  '  word       42615   Index - 923
  '  word       42653   Index - 924
  '  word       42691   Index - 925
  '  word       42730   Index - 926
  '  word       42768   Index - 927
  '  word       42806   Index - 928
  '  word       42844   Index - 929
  '  word       42882   Index - 930
  '  word       42920   Index - 931
  '  word       42958   Index - 932
  '  word       42996   Index - 933
  '  word       43034   Index - 934
  '  word       43071   Index - 935
  '  word       43109   Index - 936
  '  word       43147   Index - 937
  '  word       43185   Index - 938
  '  word       43223   Index - 939
  '  word       43261   Index - 940
  '  word       43298   Index - 941
  '  word       43336   Index - 942
  '  word       43374   Index - 943
  '  word       43411   Index - 944
  '  word       43449   Index - 945
  '  word       43487   Index - 946
  '  word       43524   Index - 947
  '  word       43562   Index - 948
  '  word       43599   Index - 949
  '  word       43637   Index - 950
  '  word       43674   Index - 951
  '  word       43712   Index - 952
  '  word       43749   Index - 953
  '  word       43787   Index - 954
  '  word       43824   Index - 955
  '  word       43861   Index - 956
  '  word       43899   Index - 957
  '  word       43936   Index - 958
  '  word       43973   Index - 959
  '  word       44011   Index - 960
  '  word       44048   Index - 961
  '  word       44085   Index - 962
  '  word       44122   Index - 963
  '  word       44159   Index - 964
  '  word       44197   Index - 965
  '  word       44234   Index - 966
  '  word       44271   Index - 967
  '  word       44308   Index - 968
  '  word       44345   Index - 969
  '  word       44382   Index - 970
  '  word       44419   Index - 971
  '  word       44456   Index - 972
  '  word       44493   Index - 973
  '  word       44529   Index - 974
  '  word       44566   Index - 975
  '  word       44603   Index - 976
  '  word       44640   Index - 977
  '  word       44677   Index - 978
  '  word       44714   Index - 979
  '  word       44750   Index - 980
  '  word       44787   Index - 981
  '  word       44824   Index - 982
  '  word       44860   Index - 983
  '  word       44897   Index - 984
  '  word       44934   Index - 985
  '  word       44970   Index - 986
  '  word       45007   Index - 987
  '  word       45043   Index - 988
  '  word       45080   Index - 989
  '  word       45116   Index - 990
  '  word       45153   Index - 991
  '  word       45189   Index - 992
  '  word       45225   Index - 993
  '  word       45262   Index - 994
  '  word       45298   Index - 995
  '  word       45334   Index - 996
  '  word       45371   Index - 997
  '  word       45407   Index - 998
  '  word       45443   Index - 999
  '  word       45479   Index - 1000
  '  word       45516   Index - 1001
  '  word       45552   Index - 1002
  '  word       45588   Index - 1003
  '  word       45624   Index - 1004
  '  word       45660   Index - 1005
  '  word       45696   Index - 1006
  '  word       45732   Index - 1007
  '  word       45768   Index - 1008
  '  word       45804   Index - 1009
  '  word       45840   Index - 1010
  '  word       45876   Index - 1011
  '  word       45912   Index - 1012
  '  word       45948   Index - 1013
  '  word       45983   Index - 1014
  '  word       46019   Index - 1015
  '  word       46055   Index - 1016
  '  word       46091   Index - 1017
  '  word       46126   Index - 1018
  '  word       46162   Index - 1019
  '  word       46198   Index - 1020
  '  word       46233   Index - 1021
  '  word       46269   Index - 1022
  '  word       46305   Index - 1023
  '  word       46340   Index - 1024
  '  word       46376   Index - 1025
  '  word       46411   Index - 1026
  '  word       46447   Index - 1027
  '  word       46482   Index - 1028
  '  word       46518   Index - 1029
  '  word       46553   Index - 1030
  '  word       46588   Index - 1031
  '  word       46624   Index - 1032
  '  word       46659   Index - 1033
  '  word       46694   Index - 1034
  '  word       46730   Index - 1035
  '  word       46765   Index - 1036
  '  word       46800   Index - 1037
  '  word       46835   Index - 1038
  '  word       46870   Index - 1039
  '  word       46905   Index - 1040
  '  word       46941   Index - 1041
  '  word       46976   Index - 1042
  '  word       47011   Index - 1043
  '  word       47046   Index - 1044
  '  word       47081   Index - 1045
  '  word       47116   Index - 1046
  '  word       47150   Index - 1047
  '  word       47185   Index - 1048
  '  word       47220   Index - 1049
  '  word       47255   Index - 1050
  '  word       47290   Index - 1051
  '  word       47325   Index - 1052
  '  word       47359   Index - 1053
  '  word       47394   Index - 1054
  '  word       47429   Index - 1055
  '  word       47464   Index - 1056
  '  word       47498   Index - 1057
  '  word       47533   Index - 1058
  '  word       47567   Index - 1059
  '  word       47602   Index - 1060
  '  word       47636   Index - 1061
  '  word       47671   Index - 1062
  '  word       47705   Index - 1063
  '  word       47740   Index - 1064
  '  word       47774   Index - 1065
  '  word       47809   Index - 1066
  '  word       47843   Index - 1067
  '  word       47877   Index - 1068
  '  word       47912   Index - 1069
  '  word       47946   Index - 1070
  '  word       47980   Index - 1071
  '  word       48014   Index - 1072
  '  word       48049   Index - 1073
  '  word       48083   Index - 1074
  '  word       48117   Index - 1075
  '  word       48151   Index - 1076
  '  word       48185   Index - 1077
  '  word       48219   Index - 1078
  '  word       48253   Index - 1079
  '  word       48287   Index - 1080
  '  word       48321   Index - 1081
  '  word       48355   Index - 1082
  '  word       48389   Index - 1083
  '  word       48423   Index - 1084
  '  word       48457   Index - 1085
  '  word       48491   Index - 1086
  '  word       48524   Index - 1087
  '  word       48558   Index - 1088
  '  word       48592   Index - 1089
  '  word       48626   Index - 1090
  '  word       48659   Index - 1091
  '  word       48693   Index - 1092
  '  word       48727   Index - 1093
  '  word       48760   Index - 1094
  '  word       48794   Index - 1095
  '  word       48827   Index - 1096
  '  word       48861   Index - 1097
  '  word       48894   Index - 1098
  '  word       48928   Index - 1099
  '  word       48961   Index - 1100
  '  word       48995   Index - 1101
  '  word       49028   Index - 1102
  '  word       49061   Index - 1103
  '  word       49095   Index - 1104
  '  word       49128   Index - 1105
  '  word       49161   Index - 1106
  '  word       49194   Index - 1107
  '  word       49228   Index - 1108
  '  word       49261   Index - 1109
  '  word       49294   Index - 1110
  '  word       49327   Index - 1111
  '  word       49360   Index - 1112
  '  word       49393   Index - 1113
  '  word       49426   Index - 1114
  '  word       49459   Index - 1115
  '  word       49492   Index - 1116
  '  word       49525   Index - 1117
  '  word       49558   Index - 1118
  '  word       49591   Index - 1119
  '  word       49624   Index - 1120
  '  word       49656   Index - 1121
  '  word       49689   Index - 1122
  '  word       49722   Index - 1123
  '  word       49755   Index - 1124
  '  word       49787   Index - 1125
  '  word       49820   Index - 1126
  '  word       49853   Index - 1127
  '  word       49885   Index - 1128
  '  word       49918   Index - 1129
  '  word       49951   Index - 1130
  '  word       49983   Index - 1131
  '  word       50016   Index - 1132
  '  word       50048   Index - 1133
  '  word       50080   Index - 1134
  '  word       50113   Index - 1135
  '  word       50145   Index - 1136
  '  word       50178   Index - 1137
  '  word       50210   Index - 1138
  '  word       50242   Index - 1139
  '  word       50274   Index - 1140
  '  word       50307   Index - 1141
  '  word       50339   Index - 1142
  '  word       50371   Index - 1143
  '  word       50403   Index - 1144
  '  word       50435   Index - 1145
  '  word       50467   Index - 1146
  '  word       50499   Index - 1147
  '  word       50531   Index - 1148
  '  word       50563   Index - 1149
  '  word       50595   Index - 1150
  '  word       50627   Index - 1151
  '  word       50659   Index - 1152
  '  word       50691   Index - 1153
  '  word       50723   Index - 1154
  '  word       50755   Index - 1155
  '  word       50787   Index - 1156
  '  word       50818   Index - 1157
  '  word       50850   Index - 1158
  '  word       50882   Index - 1159
  '  word       50913   Index - 1160
  '  word       50945   Index - 1161
  '  word       50977   Index - 1162
  '  word       51008   Index - 1163
  '  word       51040   Index - 1164
  '  word       51071   Index - 1165
  '  word       51103   Index - 1166
  '  word       51134   Index - 1167
  '  word       51166   Index - 1168
  '  word       51197   Index - 1169
  '  word       51228   Index - 1170
  '  word       51260   Index - 1171
  '  word       51291   Index - 1172
  '  word       51322   Index - 1173
  '  word       51354   Index - 1174
  '  word       51385   Index - 1175
  '  word       51416   Index - 1176
  '  word       51447   Index - 1177
  '  word       51478   Index - 1178
  '  word       51509   Index - 1179
  '  word       51540   Index - 1180
  '  word       51571   Index - 1181
  '  word       51602   Index - 1182
  '  word       51633   Index - 1183
  '  word       51664   Index - 1184
  '  word       51695   Index - 1185
  '  word       51726   Index - 1186
  '  word       51757   Index - 1187
  '  word       51788   Index - 1188
  '  word       51819   Index - 1189
  '  word       51849   Index - 1190
  '  word       51880   Index - 1191
  '  word       51911   Index - 1192
  '  word       51941   Index - 1193
  '  word       51972   Index - 1194
  '  word       52003   Index - 1195
  '  word       52033   Index - 1196
  '  word       52064   Index - 1197
  '  word       52094   Index - 1198
  '  word       52125   Index - 1199
  '  word       52155   Index - 1200
  '  word       52186   Index - 1201
  '  word       52216   Index - 1202
  '  word       52246   Index - 1203
  '  word       52277   Index - 1204
  '  word       52307   Index - 1205
  '  word       52337   Index - 1206
  '  word       52367   Index - 1207
  '  word       52398   Index - 1208
  '  word       52428   Index - 1209
  '  word       52458   Index - 1210
  '  word       52488   Index - 1211
  '  word       52518   Index - 1212
  '  word       52548   Index - 1213
  '  word       52578   Index - 1214
  '  word       52608   Index - 1215
  '  word       52638   Index - 1216
  '  word       52668   Index - 1217
  '  word       52698   Index - 1218
  '  word       52728   Index - 1219
  '  word       52758   Index - 1220
  '  word       52788   Index - 1221
  '  word       52817   Index - 1222
  '  word       52847   Index - 1223
  '  word       52877   Index - 1224
  '  word       52906   Index - 1225
  '  word       52936   Index - 1226
  '  word       52966   Index - 1227
  '  word       52995   Index - 1228
  '  word       53025   Index - 1229
  '  word       53054   Index - 1230
  '  word       53084   Index - 1231
  '  word       53113   Index - 1232
  '  word       53143   Index - 1233
  '  word       53172   Index - 1234
  '  word       53202   Index - 1235
  '  word       53231   Index - 1236
  '  word       53260   Index - 1237
  '  word       53289   Index - 1238
  '  word       53319   Index - 1239
  '  word       53348   Index - 1240
  '  word       53377   Index - 1241
  '  word       53406   Index - 1242
  '  word       53435   Index - 1243
  '  word       53464   Index - 1244
  '  word       53493   Index - 1245
  '  word       53522   Index - 1246
  '  word       53551   Index - 1247
  '  word       53580   Index - 1248
  '  word       53609   Index - 1249
  '  word       53638   Index - 1250
  '  word       53667   Index - 1251
  '  word       53696   Index - 1252
  '  word       53725   Index - 1253
  '  word       53754   Index - 1254
  '  word       53782   Index - 1255
  '  word       53811   Index - 1256
  '  word       53840   Index - 1257
  '  word       53868   Index - 1258
  '  word       53897   Index - 1259
  '  word       53925   Index - 1260
  '  word       53954   Index - 1261
  '  word       53983   Index - 1262
  '  word       54011   Index - 1263
  '  word       54039   Index - 1264
  '  word       54068   Index - 1265
  '  word       54096   Index - 1266
  '  word       54125   Index - 1267
  '  word       54153   Index - 1268
  '  word       54181   Index - 1269
  '  word       54210   Index - 1270
  '  word       54238   Index - 1271
  '  word       54266   Index - 1272
  '  word       54294   Index - 1273
  '  word       54322   Index - 1274
  '  word       54350   Index - 1275
  '  word       54378   Index - 1276
  '  word       54406   Index - 1277
  '  word       54434   Index - 1278
  '  word       54462   Index - 1279
  '  word       54490   Index - 1280
  '  word       54518   Index - 1281
  '  word       54546   Index - 1282
  '  word       54574   Index - 1283
  '  word       54602   Index - 1284
  '  word       54630   Index - 1285
  '  word       54657   Index - 1286
  '  word       54685   Index - 1287
  '  word       54713   Index - 1288
  '  word       54740   Index - 1289
  '  word       54768   Index - 1290
  '  word       54796   Index - 1291
  '  word       54823   Index - 1292
  '  word       54851   Index - 1293
  '  word       54878   Index - 1294
  '  word       54906   Index - 1295
  '  word       54933   Index - 1296
  '  word       54960   Index - 1297
  '  word       54988   Index - 1298
  '  word       55015   Index - 1299
  '  word       55042   Index - 1300
  '  word       55070   Index - 1301
  '  word       55097   Index - 1302
  '  word       55124   Index - 1303
  '  word       55151   Index - 1304
  '  word       55178   Index - 1305
  '  word       55206   Index - 1306
  '  word       55233   Index - 1307
  '  word       55260   Index - 1308
  '  word       55287   Index - 1309
  '  word       55314   Index - 1310
  '  word       55341   Index - 1311
  '  word       55367   Index - 1312
  '  word       55394   Index - 1313
  '  word       55421   Index - 1314
  '  word       55448   Index - 1315
  '  word       55475   Index - 1316
  '  word       55502   Index - 1317
  '  word       55528   Index - 1318
  '  word       55555   Index - 1319
  '  word       55582   Index - 1320
  '  word       55608   Index - 1321
  '  word       55635   Index - 1322
  '  word       55661   Index - 1323
  '  word       55688   Index - 1324
  '  word       55714   Index - 1325
  '  word       55741   Index - 1326
  '  word       55767   Index - 1327
  '  word       55794   Index - 1328
  '  word       55820   Index - 1329
  '  word       55846   Index - 1330
  '  word       55873   Index - 1331
  '  word       55899   Index - 1332
  '  word       55925   Index - 1333
  '  word       55951   Index - 1334
  '  word       55977   Index - 1335
  '  word       56003   Index - 1336
  '  word       56030   Index - 1337
  '  word       56056   Index - 1338
  '  word       56082   Index - 1339
  '  word       56108   Index - 1340
  '  word       56134   Index - 1341
  '  word       56159   Index - 1342
  '  word       56185   Index - 1343
  '  word       56211   Index - 1344
  '  word       56237   Index - 1345
  '  word       56263   Index - 1346
  '  word       56289   Index - 1347
  '  word       56314   Index - 1348
  '  word       56340   Index - 1349
  '  word       56366   Index - 1350
  '  word       56391   Index - 1351
  '  word       56417   Index - 1352
  '  word       56442   Index - 1353
  '  word       56468   Index - 1354
  '  word       56493   Index - 1355
  '  word       56519   Index - 1356
  '  word       56544   Index - 1357
  '  word       56570   Index - 1358
  '  word       56595   Index - 1359
  '  word       56620   Index - 1360
  '  word       56646   Index - 1361
  '  word       56671   Index - 1362
  '  word       56696   Index - 1363
  '  word       56721   Index - 1364
  '  word       56747   Index - 1365
  '  word       56772   Index - 1366
  '  word       56797   Index - 1367
  '  word       56822   Index - 1368
  '  word       56847   Index - 1369
  '  word       56872   Index - 1370
  '  word       56897   Index - 1371
  '  word       56922   Index - 1372
  '  word       56947   Index - 1373
  '  word       56972   Index - 1374
  '  word       56996   Index - 1375
  '  word       57021   Index - 1376
  '  word       57046   Index - 1377
  '  word       57071   Index - 1378
  '  word       57095   Index - 1379
  '  word       57120   Index - 1380
  '  word       57145   Index - 1381
  '  word       57169   Index - 1382
  '  word       57194   Index - 1383
  '  word       57218   Index - 1384
  '  word       57243   Index - 1385
  '  word       57267   Index - 1386
  '  word       57292   Index - 1387
  '  word       57316   Index - 1388
  '  word       57340   Index - 1389
  '  word       57365   Index - 1390
  '  word       57389   Index - 1391
  '  word       57413   Index - 1392
  '  word       57437   Index - 1393
  '  word       57462   Index - 1394
  '  word       57486   Index - 1395
  '  word       57510   Index - 1396
  '  word       57534   Index - 1397
  '  word       57558   Index - 1398
  '  word       57582   Index - 1399
  '  word       57606   Index - 1400
  '  word       57630   Index - 1401
  '  word       57654   Index - 1402
  '  word       57678   Index - 1403
  '  word       57702   Index - 1404
  '  word       57725   Index - 1405
  '  word       57749   Index - 1406
  '  word       57773   Index - 1407
  '  word       57797   Index - 1408
  '  word       57820   Index - 1409
  '  word       57844   Index - 1410
  '  word       57868   Index - 1411
  '  word       57891   Index - 1412
  '  word       57915   Index - 1413
  '  word       57938   Index - 1414
  '  word       57962   Index - 1415
  '  word       57985   Index - 1416
  '  word       58009   Index - 1417
  '  word       58032   Index - 1418
  '  word       58055   Index - 1419
  '  word       58079   Index - 1420
  '  word       58102   Index - 1421
  '  word       58125   Index - 1422
  '  word       58148   Index - 1423
  '  word       58171   Index - 1424
  '  word       58195   Index - 1425
  '  word       58218   Index - 1426
  '  word       58241   Index - 1427
  '  word       58264   Index - 1428
  '  word       58287   Index - 1429
  '  word       58310   Index - 1430
  '  word       58333   Index - 1431
  '  word       58356   Index - 1432
  '  word       58378   Index - 1433
  '  word       58401   Index - 1434
  '  word       58424   Index - 1435
  '  word       58447   Index - 1436
  '  word       58470   Index - 1437
  '  word       58492   Index - 1438
  '  word       58515   Index - 1439
  '  word       58537   Index - 1440
  '  word       58560   Index - 1441
  '  word       58583   Index - 1442
  '  word       58605   Index - 1443
  '  word       58628   Index - 1444
  '  word       58650   Index - 1445
  '  word       58672   Index - 1446
  '  word       58695   Index - 1447
  '  word       58717   Index - 1448
  '  word       58739   Index - 1449
  '  word       58762   Index - 1450
  '  word       58784   Index - 1451
  '  word       58806   Index - 1452
  '  word       58828   Index - 1453
  '  word       58850   Index - 1454
  '  word       58873   Index - 1455
  '  word       58895   Index - 1456
  '  word       58917   Index - 1457
  '  word       58939   Index - 1458
  '  word       58961   Index - 1459
  '  word       58983   Index - 1460
  '  word       59004   Index - 1461
  '  word       59026   Index - 1462
  '  word       59048   Index - 1463
  '  word       59070   Index - 1464
  '  word       59092   Index - 1465
  '  word       59113   Index - 1466
  '  word       59135   Index - 1467
  '  word       59157   Index - 1468
  '  word       59178   Index - 1469
  '  word       59200   Index - 1470
  '  word       59221   Index - 1471
  '  word       59243   Index - 1472
  '  word       59264   Index - 1473
  '  word       59286   Index - 1474
  '  word       59307   Index - 1475
  '  word       59329   Index - 1476
  '  word       59350   Index - 1477
  '  word       59371   Index - 1478
  '  word       59393   Index - 1479
  '  word       59414   Index - 1480
  '  word       59435   Index - 1481
  '  word       59456   Index - 1482
  '  word       59477   Index - 1483
  '  word       59498   Index - 1484
  '  word       59519   Index - 1485
  '  word       59540   Index - 1486
  '  word       59561   Index - 1487
  '  word       59582   Index - 1488
  '  word       59603   Index - 1489
  '  word       59624   Index - 1490
  '  word       59645   Index - 1491
  '  word       59666   Index - 1492
  '  word       59687   Index - 1493
  '  word       59707   Index - 1494
  '  word       59728   Index - 1495
  '  word       59749   Index - 1496
  '  word       59769   Index - 1497
  '  word       59790   Index - 1498
  '  word       59810   Index - 1499
  '  word       59831   Index - 1500
  '  word       59851   Index - 1501
  '  word       59872   Index - 1502
  '  word       59892   Index - 1503
  '  word       59913   Index - 1504
  '  word       59933   Index - 1505
  '  word       59953   Index - 1506
  '  word       59974   Index - 1507
  '  word       59994   Index - 1508
  '  word       60014   Index - 1509
  '  word       60034   Index - 1510
  '  word       60054   Index - 1511
  '  word       60075   Index - 1512
  '  word       60095   Index - 1513
  '  word       60115   Index - 1514
  '  word       60135   Index - 1515
  '  word       60155   Index - 1516
  '  word       60175   Index - 1517
  '  word       60194   Index - 1518
  '  word       60214   Index - 1519
  '  word       60234   Index - 1520
  '  word       60254   Index - 1521
  '  word       60274   Index - 1522
  '  word       60293   Index - 1523
  '  word       60313   Index - 1524
  '  word       60333   Index - 1525
  '  word       60352   Index - 1526
  '  word       60372   Index - 1527
  '  word       60391   Index - 1528
  '  word       60411   Index - 1529
  '  word       60430   Index - 1530
  '  word       60450   Index - 1531
  '  word       60469   Index - 1532
  '  word       60489   Index - 1533
  '  word       60508   Index - 1534
  '  word       60527   Index - 1535
  '  word       60546   Index - 1536
  '  word       60566   Index - 1537
  '  word       60585   Index - 1538
  '  word       60604   Index - 1539
  '  word       60623   Index - 1540
  '  word       60642   Index - 1541
  '  word       60661   Index - 1542
  '  word       60680   Index - 1543
  '  word       60699   Index - 1544
  '  word       60718   Index - 1545
  '  word       60737   Index - 1546
  '  word       60756   Index - 1547
  '  word       60775   Index - 1548
  '  word       60793   Index - 1549
  '  word       60812   Index - 1550
  '  word       60831   Index - 1551
  '  word       60850   Index - 1552
  '  word       60868   Index - 1553
  '  word       60887   Index - 1554
  '  word       60905   Index - 1555
  '  word       60924   Index - 1556
  '  word       60943   Index - 1557
  '  word       60961   Index - 1558
  '  word       60979   Index - 1559
  '  word       60998   Index - 1560
  '  word       61016   Index - 1561
  '  word       61034   Index - 1562
  '  word       61053   Index - 1563
  '  word       61071   Index - 1564
  '  word       61089   Index - 1565
  '  word       61107   Index - 1566
  '  word       61126   Index - 1567
  '  word       61144   Index - 1568
  '  word       61162   Index - 1569
  '  word       61180   Index - 1570
  '  word       61198   Index - 1571
  '  word       61216   Index - 1572
  '  word       61234   Index - 1573
  '  word       61252   Index - 1574
  '  word       61269   Index - 1575
  '  word       61287   Index - 1576
  '  word       61305   Index - 1577
  '  word       61323   Index - 1578
  '  word       61340   Index - 1579
  '  word       61358   Index - 1580
  '  word       61376   Index - 1581
  '  word       61393   Index - 1582
  '  word       61411   Index - 1583
  '  word       61429   Index - 1584
  '  word       61446   Index - 1585
  '  word       61463   Index - 1586
  '  word       61481   Index - 1587
  '  word       61498   Index - 1588
  '  word       61516   Index - 1589
  '  word       61533   Index - 1590
  '  word       61550   Index - 1591
  '  word       61567   Index - 1592
  '  word       61585   Index - 1593
  '  word       61602   Index - 1594
  '  word       61619   Index - 1595
  '  word       61636   Index - 1596
  '  word       61653   Index - 1597
  '  word       61670   Index - 1598
  '  word       61687   Index - 1599
  '  word       61704   Index - 1600
  '  word       61721   Index - 1601
  '  word       61738   Index - 1602
  '  word       61755   Index - 1603
  '  word       61772   Index - 1604
  '  word       61788   Index - 1605
  '  word       61805   Index - 1606
  '  word       61822   Index - 1607
  '  word       61838   Index - 1608
  '  word       61855   Index - 1609
  '  word       61872   Index - 1610
  '  word       61888   Index - 1611
  '  word       61905   Index - 1612
  '  word       61921   Index - 1613
  '  word       61938   Index - 1614
  '  word       61954   Index - 1615
  '  word       61970   Index - 1616
  '  word       61987   Index - 1617
  '  word       62003   Index - 1618
  '  word       62019   Index - 1619
  '  word       62035   Index - 1620
  '  word       62052   Index - 1621
  '  word       62068   Index - 1622
  '  word       62084   Index - 1623
  '  word       62100   Index - 1624
  '  word       62116   Index - 1625
  '  word       62132   Index - 1626
  '  word       62148   Index - 1627
  '  word       62164   Index - 1628
  '  word       62180   Index - 1629
  '  word       62196   Index - 1630
  '  word       62212   Index - 1631
  '  word       62227   Index - 1632
  '  word       62243   Index - 1633
  '  word       62259   Index - 1634
  '  word       62274   Index - 1635
  '  word       62290   Index - 1636
  '  word       62306   Index - 1637
  '  word       62321   Index - 1638
  '  word       62337   Index - 1639
  '  word       62352   Index - 1640
  '  word       62368   Index - 1641
  '  word       62383   Index - 1642
  '  word       62399   Index - 1643
  '  word       62414   Index - 1644
  '  word       62429   Index - 1645
  '  word       62444   Index - 1646
  '  word       62460   Index - 1647
  '  word       62475   Index - 1648
  '  word       62490   Index - 1649
  '  word       62505   Index - 1650
  '  word       62520   Index - 1651
  '  word       62535   Index - 1652
  '  word       62550   Index - 1653
  '  word       62565   Index - 1654
  '  word       62580   Index - 1655
  '  word       62595   Index - 1656
  '  word       62610   Index - 1657
  '  word       62625   Index - 1658
  '  word       62640   Index - 1659
  '  word       62654   Index - 1660
  '  word       62669   Index - 1661
  '  word       62684   Index - 1662
  '  word       62698   Index - 1663
  '  word       62713   Index - 1664
  '  word       62728   Index - 1665
  '  word       62742   Index - 1666
  '  word       62757   Index - 1667
  '  word       62771   Index - 1668
  '  word       62786   Index - 1669
  '  word       62800   Index - 1670
  '  word       62814   Index - 1671
  '  word       62829   Index - 1672
  '  word       62843   Index - 1673
  '  word       62857   Index - 1674
  '  word       62871   Index - 1675
  '  word       62886   Index - 1676
  '  word       62900   Index - 1677
  '  word       62914   Index - 1678
  '  word       62928   Index - 1679
  '  word       62942   Index - 1680
  '  word       62956   Index - 1681
  '  word       62970   Index - 1682
  '  word       62984   Index - 1683
  '  word       62998   Index - 1684
  '  word       63011   Index - 1685
  '  word       63025   Index - 1686
  '  word       63039   Index - 1687
  '  word       63053   Index - 1688
  '  word       63066   Index - 1689
  '  word       63080   Index - 1690
  '  word       63094   Index - 1691
  '  word       63107   Index - 1692
  '  word       63121   Index - 1693
  '  word       63134   Index - 1694
  '  word       63148   Index - 1695
  '  word       63161   Index - 1696
  '  word       63174   Index - 1697
  '  word       63188   Index - 1698
  '  word       63201   Index - 1699
  '  word       63214   Index - 1700
  '  word       63228   Index - 1701
  '  word       63241   Index - 1702
  '  word       63254   Index - 1703
  '  word       63267   Index - 1704
  '  word       63280   Index - 1705
  '  word       63293   Index - 1706
  '  word       63306   Index - 1707
  '  word       63319   Index - 1708
  '  word       63332   Index - 1709
  '  word       63345   Index - 1710
  '  word       63358   Index - 1711
  '  word       63371   Index - 1712
  '  word       63384   Index - 1713
  '  word       63396   Index - 1714
  '  word       63409   Index - 1715
  '  word       63422   Index - 1716
  '  word       63434   Index - 1717
  '  word       63447   Index - 1718
  '  word       63460   Index - 1719
  '  word       63472   Index - 1720
  '  word       63485   Index - 1721
  '  word       63497   Index - 1722
  '  word       63509   Index - 1723
  '  word       63522   Index - 1724
  '  word       63534   Index - 1725
  '  word       63546   Index - 1726
  '  word       63559   Index - 1727
  '  word       63571   Index - 1728
  '  word       63583   Index - 1729
  '  word       63595   Index - 1730
  '  word       63607   Index - 1731
  '  word       63620   Index - 1732
  '  word       63632   Index - 1733
  '  word       63644   Index - 1734
  '  word       63656   Index - 1735
  '  word       63668   Index - 1736
  '  word       63679   Index - 1737
  '  word       63691   Index - 1738
  '  word       63703   Index - 1739
  '  word       63715   Index - 1740
  '  word       63727   Index - 1741
  '  word       63738   Index - 1742
  '  word       63750   Index - 1743
  '  word       63762   Index - 1744
  '  word       63773   Index - 1745
  '  word       63785   Index - 1746
  '  word       63796   Index - 1747
  '  word       63808   Index - 1748
  '  word       63819   Index - 1749
  '  word       63831   Index - 1750
  '  word       63842   Index - 1751
  '  word       63853   Index - 1752
  '  word       63865   Index - 1753
  '  word       63876   Index - 1754
  '  word       63887   Index - 1755
  '  word       63898   Index - 1756
  '  word       63909   Index - 1757
  '  word       63921   Index - 1758
  '  word       63932   Index - 1759
  '  word       63943   Index - 1760
  '  word       63954   Index - 1761
  '  word       63965   Index - 1762
  '  word       63976   Index - 1763
  '  word       63986   Index - 1764
  '  word       63997   Index - 1765
  '  word       64008   Index - 1766
  '  word       64019   Index - 1767
  '  word       64030   Index - 1768
  '  word       64040   Index - 1769
  '  word       64051   Index - 1770
  '  word       64062   Index - 1771
  '  word       64072   Index - 1772
  '  word       64083   Index - 1773
  '  word       64093   Index - 1774
  '  word       64104   Index - 1775
  '  word       64114   Index - 1776
  '  word       64124   Index - 1777
  '  word       64135   Index - 1778
  '  word       64145   Index - 1779
  '  word       64155   Index - 1780
  '  word       64166   Index - 1781
  '  word       64176   Index - 1782
  '  word       64186   Index - 1783
  '  word       64196   Index - 1784
  '  word       64206   Index - 1785
  '  word       64216   Index - 1786
  '  word       64226   Index - 1787
  '  word       64236   Index - 1788
  '  word       64246   Index - 1789
  '  word       64256   Index - 1790
  '  word       64266   Index - 1791
  '  word       64276   Index - 1792
  '  word       64286   Index - 1793
  '  word       64295   Index - 1794
  '  word       64305   Index - 1795
  '  word       64315   Index - 1796
  '  word       64324   Index - 1797
  '  word       64334   Index - 1798
  '  word       64343   Index - 1799
  '  word       64353   Index - 1800
  '  word       64362   Index - 1801
  '  word       64372   Index - 1802
  '  word       64381   Index - 1803
  '  word       64391   Index - 1804
  '  word       64400   Index - 1805
  '  word       64409   Index - 1806
  '  word       64419   Index - 1807
  '  word       64428   Index - 1808
  '  word       64437   Index - 1809
  '  word       64446   Index - 1810
  '  word       64455   Index - 1811
  '  word       64464   Index - 1812
  '  word       64473   Index - 1813
  '  word       64482   Index - 1814
  '  word       64491   Index - 1815
  '  word       64500   Index - 1816
  '  word       64509   Index - 1817
  '  word       64518   Index - 1818
  '  word       64527   Index - 1819
  '  word       64535   Index - 1820
  '  word       64544   Index - 1821
  '  word       64553   Index - 1822
  '  word       64562   Index - 1823
  '  word       64570   Index - 1824
  '  word       64579   Index - 1825
  '  word       64587   Index - 1826
  '  word       64596   Index - 1827
  '  word       64604   Index - 1828
  '  word       64613   Index - 1829
  '  word       64621   Index - 1830
  '  word       64629   Index - 1831
  '  word       64638   Index - 1832
  '  word       64646   Index - 1833
  '  word       64654   Index - 1834
  '  word       64662   Index - 1835
  '  word       64671   Index - 1836
  '  word       64679   Index - 1837
  '  word       64687   Index - 1838
  '  word       64695   Index - 1839
  '  word       64703   Index - 1840
  '  word       64711   Index - 1841
  '  word       64719   Index - 1842
  '  word       64727   Index - 1843
  '  word       64734   Index - 1844
  '  word       64742   Index - 1845
  '  word       64750   Index - 1846
  '  word       64758   Index - 1847
  '  word       64765   Index - 1848
  '  word       64773   Index - 1849
  '  word       64781   Index - 1850
  '  word       64788   Index - 1851
  '  word       64796   Index - 1852
  '  word       64803   Index - 1853
  '  word       64811   Index - 1854
  '  word       64818   Index - 1855
  '  word       64826   Index - 1856
  '  word       64833   Index - 1857
  '  word       64840   Index - 1858
  '  word       64848   Index - 1859
  '  word       64855   Index - 1860
  '  word       64862   Index - 1861
  '  word       64869   Index - 1862
  '  word       64876   Index - 1863
  '  word       64883   Index - 1864
  '  word       64891   Index - 1865
  '  word       64898   Index - 1866
  '  word       64905   Index - 1867
  '  word       64911   Index - 1868
  '  word       64918   Index - 1869
  '  word       64925   Index - 1870
  '  word       64932   Index - 1871
  '  word       64939   Index - 1872
  '  word       64946   Index - 1873
  '  word       64952   Index - 1874
  '  word       64959   Index - 1875
  '  word       64966   Index - 1876
  '  word       64972   Index - 1877
  '  word       64979   Index - 1878
  '  word       64985   Index - 1879
  '  word       64992   Index - 1880
  '  word       64998   Index - 1881
  '  word       65005   Index - 1882
  '  word       65011   Index - 1883
  '  word       65017   Index - 1884
  '  word       65024   Index - 1885
  '  word       65030   Index - 1886
  '  word       65036   Index - 1887
  '  word       65042   Index - 1888
  '  word       65048   Index - 1889
  '  word       65054   Index - 1890
  '  word       65060   Index - 1891
  '  word       65066   Index - 1892
  '  word       65072   Index - 1893
  '  word       65078   Index - 1894
  '  word       65084   Index - 1895
  '  word       65090   Index - 1896
  '  word       65096   Index - 1897
  '  word       65102   Index - 1898
  '  word       65108   Index - 1899
  '  word       65113   Index - 1900
  '  word       65119   Index - 1901
  '  word       65125   Index - 1902
  '  word       65130   Index - 1903
  '  word       65136   Index - 1904
  '  word       65141   Index - 1905
  '  word       65147   Index - 1906
  '  word       65152   Index - 1907
  '  word       65158   Index - 1908
  '  word       65163   Index - 1909
  '  word       65168   Index - 1910
  '  word       65174   Index - 1911
  '  word       65179   Index - 1912
  '  word       65184   Index - 1913
  '  word       65189   Index - 1914
  '  word       65194   Index - 1915
  '  word       65199   Index - 1916
  '  word       65204   Index - 1917
  '  word       65210   Index - 1918
  '  word       65214   Index - 1919
  '  word       65219   Index - 1920
  '  word       65224   Index - 1921
  '  word       65229   Index - 1922
  '  word       65234   Index - 1923
  '  word       65239   Index - 1924
  '  word       65244   Index - 1925
  '  word       65248   Index - 1926
  '  word       65253   Index - 1927
  '  word       65258   Index - 1928
  '  word       65262   Index - 1929
  '  word       65267   Index - 1930
  '  word       65271   Index - 1931
  '  word       65276   Index - 1932
  '  word       65280   Index - 1933
  '  word       65285   Index - 1934
  '  word       65289   Index - 1935
  '  word       65293   Index - 1936
  '  word       65298   Index - 1937
  '  word       65302   Index - 1938
  '  word       65306   Index - 1939
  '  word       65310   Index - 1940
  '  word       65314   Index - 1941
  '  word       65319   Index - 1942
  '  word       65323   Index - 1943
  '  word       65327   Index - 1944
  '  word       65331   Index - 1945
  '  word       65335   Index - 1946
  '  word       65338   Index - 1947
  '  word       65342   Index - 1948
  '  word       65346   Index - 1949
  '  word       65350   Index - 1950
  '  word       65354   Index - 1951
  '  word       65357   Index - 1952
  '  word       65361   Index - 1953
  '  word       65365   Index - 1954
  '  word       65368   Index - 1955
  '  word       65372   Index - 1956
  '  word       65375   Index - 1957
  '  word       65379   Index - 1958
  '  word       65382   Index - 1959
  '  word       65386   Index - 1960
  '  word       65389   Index - 1961
  '  word       65392   Index - 1962
  '  word       65396   Index - 1963
  '  word       65399   Index - 1964
  '  word       65402   Index - 1965
  '  word       65405   Index - 1966
  '  word       65409   Index - 1967
  '  word       65412   Index - 1968
  '  word       65415   Index - 1969
  '  word       65418   Index - 1970
  '  word       65421   Index - 1971
  '  word       65424   Index - 1972
  '  word       65427   Index - 1973
  '  word       65429   Index - 1974
  '  word       65432   Index - 1975
  '  word       65435   Index - 1976
  '  word       65438   Index - 1977
  '  word       65441   Index - 1978
  '  word       65443   Index - 1979
  '  word       65446   Index - 1980
  '  word       65448   Index - 1981
  '  word       65451   Index - 1982
  '  word       65454   Index - 1983
  '  word       65456   Index - 1984
  '  word       65459   Index - 1985
  '  word       65461   Index - 1986
  '  word       65463   Index - 1987
  '  word       65466   Index - 1988
  '  word       65468   Index - 1989
  '  word       65470   Index - 1990
  '  word       65472   Index - 1991
  '  word       65475   Index - 1992
  '  word       65477   Index - 1993
  '  word       65479   Index - 1994
  '  word       65481   Index - 1995
  '  word       65483   Index - 1996
  '  word       65485   Index - 1997
  '  word       65487   Index - 1998
  '  word       65489   Index - 1999
  '  word       65491   Index - 2000
  '  word       65492   Index - 2001
  '  word       65494   Index - 2002
  '  word       65496   Index - 2003
  '  word       65498   Index - 2004
  '  word       65499   Index - 2005
  '  word       65501   Index - 2006
  '  word       65503   Index - 2007
  '  word       65504   Index - 2008
  '  word       65506   Index - 2009
  '  word       65507   Index - 2010
  '  word       65509   Index - 2011
  '  word       65510   Index - 2012
  '  word       65511   Index - 2013
  '  word       65513   Index - 2014
  '  word       65514   Index - 2015
  '  word       65515   Index - 2016
  '  word       65516   Index - 2017
  '  word       65518   Index - 2018
  '  word       65519   Index - 2019
  '  word       65520   Index - 2020
  '  word       65521   Index - 2021
  '  word       65522   Index - 2022
  '  word       65523   Index - 2023
  '  word       65524   Index - 2024
  '  word       65525   Index - 2025
  '  word       65526   Index - 2026
  '  word       65526   Index - 2027
  '  word       65527   Index - 2028
  '  word       65528   Index - 2029
  '  word       65529   Index - 2030
  '  word       65529   Index - 2031
  '  word       65530   Index - 2032
  '  word       65531   Index - 2033
  '  word       65531   Index - 2034
  '  word       65532   Index - 2035
  '  word       65532   Index - 2036
  '  word       65533   Index - 2037
  '  word       65533   Index - 2038
  '  word       65533   Index - 2039
  '  word       65534   Index - 2040
  '  word       65534   Index - 2041
  '  word       65534   Index - 2042
  '  word       65535   Index - 2043
  '  word       65535   Index - 2044
  '  word       65535   Index - 2045
  '  word       65535   Index - 2046
  '  word       65535   Index - 2047
  '  word       65535   Index - 2048

' /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{{

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                  TERMS OF USE: MIT License
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the
// Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}}