// Source               ::  MALP_AL5D_1Sheeld_V2

// Description          ::  Basic Vehicle and AL5D Robotic Arm control program using 1Sheeld and Smartphone/Tablet 
//                                ...Accelerometer controls throttle, steering, and forward/backwards
//                                ...Slider acts as ignition switch and throttle governor
//                                ...Gamepad controls arm servos
//                                ...Toggle Switch activates autonomous arm demo mode
//
//                          This program will also ...
//                                ...use an ultrasonic sensor, if installed and code activated, to prevent it from running into obstacles.
//
//                          MALP "Mobile Analytic Laboratory Probe"
//                                http://stargate.wikia.com/wiki/Mobile_Analytic_Laboratory_Probe
//

//
//                          Although this program is specifically written for an AL5D under 1Sheeld control,
//                          it can be easily changed for any robotic arm and controller.
//
//                          Performs best on my platform.  Make an appointment :)   ...
//                          If your arm servos are not configured exactly the same as mine, you might have to swap
//                          angles in various spots ( ie 180 instead of 0, etc ).  You'll know that this is necessary
//                          if this program causes some of your servos to run "opposite" to what you expect.


//                          Recommended components ( or equivalent ) 

//                          1)   Dagu 4WD Thumper
//                                     http://dagurobot.com/goods.php?id=88
//                          2)   Sabertooth Dual 25A 6V-24V Regenerative Motor Driver
//                                     http://www.robotshop.com/dimension-engineering-sabertooth-2x25.html
//                          3)   Lynxmotion AL5D 4 Degrees of Freedom (4DoF) Robotic Arm
//                                   http://www.robotshop.com/en/lynxmotion-al5d-4-degrees-freedom-robotic-arm-combo-kit.html 
//                          4)   Lynxmotion Wrist Rotate Upgrade (Heavy Duty)
//                                   http://www.robotshop.com/en/lynxmotion-heavy-duty-wrist-rotate-upgrade.html
//                          5)   Lynxmotion NotBoarduino
//                                    http://www.robotshop.com/en/lynxmotion-botboarduino-robot-controller.html
//                          6)   1Sheeld
//                                     http://www.robotshop.com/1sheeld-android-smartphone-multi-purpose-shield-arduino.html
//                          7)   6V, 10000mAh Rechargeable NiMh Battery
//                                     http://www.robotshop.com/6v-10000mah-rechargeable-nimh-battery.html
//                          8)   Ultrasonic Range Finder
//                                     http://www.robotshop.com/seeedstudio-ultrasonic-range-finder.html
//                          9)   Dagu - 2DoF Robot Arm (optional - used to demo arm to arm interaction) 
//                                     http://www.dagurobot.com/goods.php?id=72



// Original Program     ::  MALP control & program standards     - Joe M  / arduino (at) bellaliant.net   / May 2016
//                          

// Modified By          ::  Your Name Here / Contact / Date  



//  This program is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License 
//  as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  Lesser General Public License for more details.
//
//  For a copy of the GNU Lesser General Public License, 
//  write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA



// **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T ****
// ***********************************************************************************************************************************************************************
// Pinouts... IMPORTANT!!!! ... set the pinouts to your hardware configuration
//
// This program uses very few components.  Constants are included for your own expansion.
//
// ***********************************************************************************************************************************************************************


// Activate this line if you have a second arm installed
#define SecondArm_Installed


// ********** Digital Pins D0 thru D13 **********

// Using 0 and 1 disables upload!     

#define DO_NOT_USE_0         0
#define DO_NOT_USE_1         1

#define SHOULDER_PIN         2
#define ELBOW_PIN            3
#define WRIST_PIN            4
        
#define SPEAKER_PIN          5

#define BASE_PIN             6
#define ONBOARD_PB_A         7
#define ONBOARD_PB_B         8
#define ONBOARD_PB_C         9
#define WRISTROTATE_PIN     10 
#define GRIPPER_PIN         11

#ifdef SecondArm_Installed        // "ifdef's" are one of those obscure C++ features most beginning programmers don't know about.
                                  // It is a neat feature that allows "conditional compilation".  In the case of this program,
                                  // the various code segments using the second robotic arm will only be included in the compiled executable
                                  // if you defined SecondArm_Installed above ( ie #define SecondArm_Installed ).
                                  // For more on "conditional compilation", go to http://www.cplusplus.com/doc/tutorial/preprocessor/
#define BACKELBOW_PIN       12  
#define BACKGRIPPER_PIN     13  
#endif  


// ********** Analog Pins A0 thru A5 **********

// FYI... using analog pins for digital I/O is just the same as using digital ones.

// A0 is referred to as Pin 14     A1 is referred to as Pin 15
// A2 is referred to as Pin 16     A3 is referred to as Pin 17
// A4 is referred to as Pin 18     A5 is referred to as Pin 19

#define SABER_RX_PIN        14  // Serial receive pin... NOT USED (but still init'd)
#define SABER_TX_PIN        15  // Serial transmit pin.  This pin is used to control the motors 
//#define RANGEFINDER_PIN     16   // The ultrasonic rangefinder mounted on the front of the MALP (if installed)
                                   // Comment out this define if rangefinder not installed  
#define NOT_USED_A3         A3 
#define NOT_USED_A4         A4  
#define NOT_USED_A5         A5       



// *********************************************************************************************************************************************************************
// Required includes <--------------------------------------------------------------------------------------------------------------------------------------------------
// *********************************************************************************************************************************************************************

// 1Sheeld
// Not including these constants will compile the entire 1Sheeld library into your sketch,
// making the compiled size larger than required
#define CUSTOM_SETTINGS
#define INCLUDE_ACCELEROMETER_SENSOR_SHIELD
#define INCLUDE_SLIDER_SHIELD
#define INCLUDE_GAMEPAD_SHIELD 
#define INCLUDE_TOGGLE_BUTTON_SHIELD
#include <OneSheeld.h>


// Required for communicating with Sabertooth 
#include <SoftwareSerial.h>

// Required for arm servos
#include <Servo.h>

// Notes for playing music on onboard speaker
#include "pitch.h"


#ifdef RANGEFINDER_PIN
// Optional ultrasonic sensor library (if installed) 
#include <Ultrasonic.h>
#endif




// *********************************************************************************************************************************************************************
//  Global Variables.
//  Professional programmers cringe at the thought of globals, but they seem to be generally acceptable in the "wild west" Arduino world.
//  Prudent use of them simplifies program design since variable passing via parameters between functions are reduced.
// *********************************************************************************************************************************************************************

// An homage to every program I've written since my WATFIV days.
// Every program needs an "int i", right?
int i;


// Constants for Sabertooth 2x25 Motor Controller
// Set to 9600 through Sabertooth dip switches
#define SABER_BAUDRATE 9600


// Global variables for Motor Control

SoftwareSerial SaberSerial = SoftwareSerial( SABER_RX_PIN, SABER_TX_PIN );


static boolean actionRequired;

// Enumerated data types help make code more readable
// http://playground.arduino.cc/Code/Enum

enum ThrottleAndParkReverseDriveSelector { 
  
  REVERSE_4x4 , REVERSE_3x4 , REVERSE_2x4 , REVERSE_3x8 , REVERSE_1x4 ,
  STOPNOW_0x4 ,
  FORWARD_1x4 , FORWARD_3x8 , FORWARD_2x4 , FORWARD_3x4 , FORWARD_4x4
  
} throttle = STOPNOW_0x4 , prevThrottle = STOPNOW_0x4;

enum SteeringWheel { 
  
  LEFT , STRAIGHT , RIGHT
   
} steering = STRAIGHT , prevSteering = STRAIGHT;



// Constants for Right , Left Motor Control via Sabertooth Mode 3 Simplified Serial Mode.
// Full , 3/4 , Half, 1/4 Speeds , plus Stop , provided.
// If you want to add other speed levels, just follow the provided scheme.

#define FORWARD_4x4_R_MOTOR   255
#define FORWARD_3x4_R_MOTOR   240
#define FORWARD_2x4_R_MOTOR   224
#define FORWARD_3x8_R_MOTOR   216  
#define FORWARD_1x4_R_MOTOR   208
#define STOPNOW_0x4_R_MOTOR   192
#define REVERSE_1x4_R_MOTOR   175
#define REVERSE_3x8_R_MOTOR   167
#define REVERSE_2x4_R_MOTOR   159
#define REVERSE_3x4_R_MOTOR   143
#define REVERSE_4x4_R_MOTOR   128

#define FORWARD_4x4_L_MOTOR   127
#define FORWARD_3x4_L_MOTOR   112
#define FORWARD_2x4_L_MOTOR    96
#define FORWARD_3x8_L_MOTOR    88
#define FORWARD_1x4_L_MOTOR    80
#define STOPNOW_0x4_L_MOTOR    64
#define REVERSE_1x4_L_MOTOR    48
#define REVERSE_3x8_L_MOTOR    40
#define REVERSE_2x4_L_MOTOR    32
#define REVERSE_3x4_L_MOTOR    16
#define REVERSE_4x4_L_MOTOR     1

#define STOPALL_0x0_B_MOTOR    (byte)0x00   // Stop both motors



#ifdef RANGEFINDER_PIN     // "ifdef's" are one of those obscure C++ features most beginning programmers don't know about.
                           // It is a neat feature that allows "conditional compilation".  In the case of this program,
                           // the various code segments using the ultrasonic sensor will only be included in the compiled executable
                           // if you defined RANGEFINDER_PIN above ( ie #define RANGEFINDER_PIN 3 ).
                           // For more on "conditional compilation", go to http://www.cplusplus.com/doc/tutorial/preprocessor/

// Constants and variables for the ultrasonic sensor (if installed)
#define   DANGER_CLOSE    20  // Stop the MALP within this distance in cms from obstacle in front.


#endif


// Servos that make up the AL5D
Servo Base, Shoulder, Elbow, Wrist, WristRotate, Gripper;

// Store the current servo angles
int   currentBaseAngle, currentShoulderAngle, currentElbowAngle, 
      currentWristAngle, currentWristRotateAngle, currentGripperAngle;

      
#ifdef SecondArm_Installed
// Servos that make up the second (optional) 2DoF robot arm
Servo BackElbow, BackGripper;
int   currentBackElbowAngle, currentBackGripperAngle;
#endif


// Control the speed of the arm movements.  Increase angle, decrease delay to make arm react faster.
int   servoStepAngle = 2 , servoStepDelay = 40;

// Determine whether the arm is under user control, or autonomous demo mode.
bool   demoMode = true;



// **********************************************************************************************************************************************************************
// Vehicle Function Prototypes <-----------------------------------------------------------------------------------------------------------------------------------------                                                                                                                                             
// **********************************************************************************************************************************************************************

boolean   getCurrentThrottleSteeringSettings  ( void );
void      malpAction                          ( void );
void      motorTest                           ( void );

#ifdef RANGEFINDER_PIN
boolean   obstacleTooClose                    ( void );
#endif



// **********************************************************************************************************************************************************************
// Al5D Function Prototypes <--------------------------------------------------------------------------------------------------------------------------------------------                                                                                                                                             
// **********************************************************************************************************************************************************************

void AL5D                 ( void );
void gamepadControl       ( void );
void gamepadControl_BORG  ( void );
void gamepadControl_RULD  ( void );
void shaveAndAHaircut     ( void );

void rotateBase           ( const int& endAt );
void rotateShoulder       ( const int& endAt );
void rotateElbow          ( const int& endAt );
void rotateWrist          ( const int& endAt );
void rotateWristRotate    ( const int& endAt );
void rotateGripper        ( const int& endAt );
void rotateServo          ( const int& endAt , Servo& whichServo , int& currentAngle );

void deployArms           ( void );
void grabVerticalItem     ( void );
void dropItem             ( void );
void autonomousDemo       ( void );

#ifdef SecondArm_Installed
void grabItemFromBackArm  ( void );
void rotateBackElbow      ( const int& endAt );
void rotateBackGripper    ( const int& endAt );
#endif



// Functions with default parameters
// http://en.cppreference.com/w/cpp/language/default_arguments

void extendArm            ( const int& = 34 , const bool& = 0 );    // If no arguments provided, default Shoulder height is 34, default "withItem" is no
void retractArm           ( const int& = 150 );                     // If no argument provided, default Shoulder height is 150


   
// *********************************************************************************************************************************************************************
// Main program <-------------------------------------------------------------------------------------------------------------------------------------------------------                                                                                                                                                                                                                   
// *********************************************************************************************************************************************************************
 
void setup( ) {
    
  // Connect to your Smartphone/Tablet for pilot control
  OneSheeld.begin( );
  
  // Transmit pin for Sabertooth communication
  pinMode( SABER_TX_PIN, OUTPUT );
  
  // Connect to the Sabertooth for drive motor control
  SaberSerial.begin( SABER_BAUDRATE ); 

  // Wait for the Sabertooth to initialize
  delay( 2000 );
  
  // TEST TEST TEST TEST
  // Only activate this code to test your motors.
  // This function never terminates, so if you call it,
  // this sketch will do nothing else but spin the motors forward and back.
  // motorTest( );

    
  // Ensure MALP is "parked"
  throttle = STOPNOW_0x4;
  steering = STRAIGHT;
  malpAction( );


  #ifdef RANGEFINDER_PIN 
  // If Ultrasonic sensor installed...     
  pinMode( RANGEFINDER_PIN, INPUT );       
  #endif

  // Init the BotBoarduino onboard push buttons A,B,C
  pinMode ( ONBOARD_PB_A , INPUT );
  pinMode ( ONBOARD_PB_B , INPUT );
  pinMode ( ONBOARD_PB_C , INPUT );
    
  // Set robot arms to working position
  deployArms( );


}





void loop( ) {

  actionRequired = getCurrentThrottleSteeringSettings ( );
  
  #ifdef RANGEFINDER_PIN
  // Override pilot command if obstacle detected.
  // If no obstacle, keep pilot command (if any).
  actionRequired = ( obstacleTooClose ( ) ) ? true : actionRequired;
  #endif

  if ( actionRequired ) malpAction( ); 

  // Allow control of the robotic arm if the MALP is stopped and ignition is off.
  if ( Slider.getValue() < 20 ) AL5D( );
  
  
}




// *********************************************************************************************************************************************************************
// Vehicle Functions <--------------------------------------------------------------------------------------------------------------------------------------------------
// *********************************************************************************************************************************************************************


boolean getCurrentThrottleSteeringSettings  ( void ) {

  // Incoming parameters :   Nothing.
    
  // Global variables    :   Uses    -   
  //                         Changes -  prevThrottle , throttle
  
  // Responsibilities    :   Reads Smartphone/Tablet accelerometer via 1Sheeld,
  //                         and maps Z and X readings to throttle and steering commands
  //                         to be executed in malpAction(). 

  //                         If a throttle or steering command is the same as last reading,
  //                         this function reports "no new command detected".

  //                         1Sheeld "Slider" used to determine ignition status and govern throttle
  
  // Returns             :   true if new pilot command detected, false otherwise

  
  // Make local variables static to save the CPU from having to
  // create them every time this function is called
  static boolean  commandDetected = false; 
  static int newThrottleInput;
  static int newSteeringInput;
  
  // Save the previous throttle/steering values before reading the new ones
  prevThrottle = throttle;
  prevSteering = steering;
 
  // Accelerometer X Axis controls steering 
  //    ... left / right
  newSteeringInput = ( (int) AccelerometerSensor.getX() );

  if ( newSteeringInput >= 3 )        steering = LEFT;
  else if ( newSteeringInput <= -3 )  steering = RIGHT; 
  else                                steering = STRAIGHT;

    
  // Accelerometer Z Axis controls throttle 
  //    ... the more you tilt the accelerometer forward, the faster the MALP will go forward
  //    ... the more you tilt the accelerometer back, the faster the MALP will go in reverse
  newThrottleInput = ( (int) AccelerometerSensor.getZ() ) / 2 ;  // Increase divisor to decrease accelerometer sensitivity

  
  // The Slider is used to restrict the throttle to a single low speed 
  if( Slider.getValue() < 128 ) {
      if ( newThrottleInput > 0 )        newThrottleInput =  99;
      else if ( newThrottleInput < 0 )   newThrottleInput = -99;
  }
  
  
  switch ( newThrottleInput ) {  
                                  
    case -99:  throttle = REVERSE_3x8;   break;     // Throttle is restricted
    
    case  -3:  throttle = REVERSE_3x4;   break;

    case  -2:  throttle = REVERSE_2x4;   break;
    
    case  -1:  throttle = REVERSE_1x4;   break;
    
    case   0:  throttle = STOPNOW_0x4;   break;

    case   1:  throttle = FORWARD_1x4;   break;

    case   2:  throttle = FORWARD_2x4;   break;

    case   3:  throttle = FORWARD_3x4;   break;

    case  99:  throttle = FORWARD_3x8;   break;   // Throttle is restricted

    // Anything else, full speed...
    default :  throttle = ( newThrottleInput > 0 ) ? FORWARD_4x4 : REVERSE_4x4;
               break;

  }

  
  // If "ignition" is not turned on (via Slider), override any pilot commands and stop MALP
  if( Slider.getValue() < 20 ) throttle = STOPNOW_0x4;

  commandDetected = (  ( throttle != prevThrottle ) || ( steering != prevSteering ) ) ? true : false ;
  
  return commandDetected;

}


void malpAction ( void ) {

  // Incoming parameters :   Nothing.
    
  // Global variables    :   Uses    -  throttle, steering
  //                         Changes -  
  
  // Responsibilities    :   Alters motor speed/direction depending on previously
  //                         calculated throttle and steering settings
  
  // Returns             :   Nothing
  
   
  // Communicate with Sabertooth using Mode 3 Simplified Serial Mode (ssm)... 
  // ... a single byte of data represents the desired speed/direction for a motor
  static byte ssm_L_Motor = 0 , ssm_R_Motor = 0;
  
  // Apply current throttle and steering settings
  switch ( throttle ) {

    case REVERSE_4x4:
    
        ssm_L_Motor = REVERSE_4x4_L_MOTOR;
        ssm_R_Motor = REVERSE_4x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_4x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_4x4_R_MOTOR;  break;
        }
        
        break;
     
    case REVERSE_3x4:
    
        ssm_L_Motor = REVERSE_3x4_L_MOTOR;
        ssm_R_Motor = REVERSE_3x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_3x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_3x4_R_MOTOR;  break;
        }

        break;       

    case REVERSE_2x4:
    
        ssm_L_Motor = REVERSE_2x4_L_MOTOR;
        ssm_R_Motor = REVERSE_2x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_2x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_2x4_R_MOTOR;  break;
        }
        
        break;    

    case REVERSE_3x8:
    
        ssm_L_Motor = REVERSE_3x8_L_MOTOR;
        ssm_R_Motor = REVERSE_3x8_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_3x8_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_3x8_R_MOTOR;  break;
        }
        
        break;

    case REVERSE_1x4:
    
        ssm_L_Motor = REVERSE_1x4_L_MOTOR;
        ssm_R_Motor = REVERSE_1x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_1x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_1x4_R_MOTOR;  break;
        }
        
        break;
 
    case STOPNOW_0x4:
              
        ssm_L_Motor = STOPNOW_0x4_L_MOTOR;
        ssm_R_Motor = STOPNOW_0x4_R_MOTOR;
        break;

    case FORWARD_1x4:
    
        ssm_L_Motor = FORWARD_1x4_L_MOTOR;
        ssm_R_Motor = FORWARD_1x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = REVERSE_1x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = REVERSE_1x4_R_MOTOR;  break;
        }
        
        break;

    case FORWARD_3x8:
    
        ssm_L_Motor = FORWARD_3x8_L_MOTOR;
        ssm_R_Motor = FORWARD_3x8_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = REVERSE_3x8_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = REVERSE_3x8_R_MOTOR;  break;
        }
        
        break;

    case FORWARD_2x4:
    
        ssm_L_Motor = FORWARD_2x4_L_MOTOR;
        ssm_R_Motor = FORWARD_2x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = REVERSE_2x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = REVERSE_2x4_R_MOTOR;  break;
        }
        
        break;

    case FORWARD_3x4:
    
        ssm_L_Motor = FORWARD_3x4_L_MOTOR;
        ssm_R_Motor = FORWARD_3x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_3x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_3x4_R_MOTOR;  break;
        }
        
        break;

    case FORWARD_4x4:
    
        ssm_L_Motor = FORWARD_4x4_L_MOTOR;
        ssm_R_Motor = FORWARD_4x4_R_MOTOR;

        switch ( steering ) {
          case LEFT:  ssm_L_Motor = FORWARD_4x4_L_MOTOR;  break;
          case RIGHT: ssm_R_Motor = FORWARD_4x4_R_MOTOR;  break;
        }
        
        break;

    default:
              
        break;
  }
 
  SaberSerial.write( ssm_R_Motor );
  SaberSerial.write( ssm_L_Motor ); 
  
}

#ifdef RANGEFINDER_PIN
boolean obstacleTooClose( void ) {
  
  // Incoming parameters :   Nothing.
    
  // Global variables    :   Uses    -   
  //                         Changes -  throttle
  
  // Responsibilities    :   Uses Ultrasonic sensor to detect obstacle in front.
  //                         Will set throttle to STOPNOW_0x4 if within DANGER_CLOSE range
  //                         (Optionally) Will set throttle to half speed if within 2.0 x DANGER_CLOSE range
  //                                      Will set throttle to quarter speed if within 1.5 x DANGER_CLOSE range
  
  // Returns             :   true if "danger close", false otherwise.
  


  // If the MALP isn't moving forward, there's no need to do anything here
  if ( throttle <= STOPNOW_0x4 ) return false;

  // Ultrasonic sensor object 
  static Ultrasonic ultrasonicFront(RANGEFINDER_PIN);

  // This will contain the calculated distance to obstacle
  static int frontRangeInCentimeters;
  
  ultrasonicFront.DistanceMeasure();                                                 // Get the current signal time
  frontRangeInCentimeters =   ultrasonicFront.microsecondsToCentimeters();           // Convert the time to centimeters
  
  // Possible obstacle... but because the ultrasonic is so unreliable, take a second look
  if ( frontRangeInCentimeters <= ( DANGER_CLOSE * 2 ) ) {
      
    ultrasonicFront.DistanceMeasure();                                                // Get the current signal time
    frontRangeInCentimeters += ultrasonicFront.microsecondsToCentimeters();           // Convert the time to centimeters
        
    frontRangeInCentimeters /= 2;   // Average the 2 readings
    
  }


  // Too close, stop!
  if ( frontRangeInCentimeters <= DANGER_CLOSE ) {
    
    throttle = STOPNOW_0x4;       
    return true;
        
  }
  
  // Getting really close to potential threat, slow down 
  if ( frontRangeInCentimeters <= ( DANGER_CLOSE * 1.5 ) ) {
    
    if ( throttle > FORWARD_1x4 ) { 
      throttle = FORWARD_1x4;      
      return true;
    }   
    
  }

  // Getting close to potential threat, slow down
  if ( frontRangeInCentimeters <= ( DANGER_CLOSE * 2.0 ) ) {
    
    if ( throttle > FORWARD_2x4 ) { 
      throttle = FORWARD_2x4;
      return true;
    }   
    
  }
  
  return false;
 
 }
#endif


void motorTest ( void ) {
  
  // Incoming parameters :   Nothing.
    
  // Global variables    :   Uses    -   
  //                         Changes -  
  
  // Responsibilities    :   Use this to test your motors.
  //                         If your motors do not perform to expectations, it may indicate...
  //                              1) Your battery/motor/controller configuration does not support that speed
  //                                 (especially the lower speeds, motors stall out due to lack of current)
  //                              2) Batteries are too weak... ensure you have a full charge
  
  // Returns             :   Nothing.  This function never terminates.

    int desiredTime = 2000;
    
    // You could disable the loop to make this a "one time" test
    while ( true ) {
      
      SaberSerial.write( FORWARD_1x4_R_MOTOR );  
      SaberSerial.write( FORWARD_1x4_L_MOTOR );
      delay ( desiredTime ) ;
      
      SaberSerial.write( FORWARD_2x4_R_MOTOR );  
      SaberSerial.write( FORWARD_2x4_L_MOTOR );
      delay ( desiredTime ) ;
      
      SaberSerial.write( FORWARD_3x4_R_MOTOR );  
      SaberSerial.write( FORWARD_3x4_L_MOTOR );
      delay ( desiredTime ) ;
            
      SaberSerial.write( FORWARD_4x4_L_MOTOR );  
      SaberSerial.write( FORWARD_4x4_R_MOTOR );
      delay ( desiredTime ) ;
      
      SaberSerial.write( STOPALL_0x0_B_MOTOR );
      delay ( desiredTime ) ;
      
      SaberSerial.write( REVERSE_1x4_R_MOTOR );   
      SaberSerial.write( REVERSE_1x4_L_MOTOR );
      delay ( desiredTime ) ;

      SaberSerial.write( REVERSE_2x4_R_MOTOR );   
      SaberSerial.write( REVERSE_2x4_L_MOTOR );
      delay ( desiredTime ) ;

      SaberSerial.write( REVERSE_3x4_R_MOTOR );   
      SaberSerial.write( REVERSE_3x4_L_MOTOR );
      delay ( desiredTime ) ;
           
      SaberSerial.write( REVERSE_4x4_R_MOTOR );   
      SaberSerial.write( REVERSE_4x4_L_MOTOR );
      delay ( desiredTime ) ;  
      
      SaberSerial.write( STOPALL_0x0_B_MOTOR );
      delay ( desiredTime ) ;
      
    }
  
}








// *********************************************************************************************************************************************************************
// AL5D Functions <-----------------------------------------------------------------------------------------------------------------------------------------------------                                                                                                                                                                                                              
// *********************************************************************************************************************************************************************

void AL5D( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -   
    //                         Changes -  demoMode
    
    // Responsibilities    :   Control robotic arm.
    
    // Returns             :   Nothing.

    // Activate if you want to get mode from 1Sheeld Toggle Button.
    demoMode = ToggleButton.getStatus( );

    // Activate if you want to set demoMode using BotBoarduino onboard push buttons A and B.
    // For the BotBoarduino pushbuttons, LOW means "pressed".
    // if ( digitalRead ( ONBOARD_PB_A ) == LOW ) demoMode = true;    
    // if ( digitalRead ( ONBOARD_PB_B ) == LOW ) demoMode = false; 
    
    if ( demoMode )  autonomousDemo( ); 
    else             gamepadControl( );

}


// Incoming parameters :   constant reference to an int denoting servo angle to move to.  
// Global variables    :   Uses    -   "Servo", current"Servo"Angle
//                         Changes -  
// Responsibilities    :   Passes servo variables on to generic servo control function.
// Returns             :   Nothing.

void rotateBase         ( const int& endAt ) {  rotateServo( endAt , Base ,        currentBaseAngle );        }
void rotateShoulder     ( const int& endAt ) {  rotateServo( endAt , Shoulder ,    currentShoulderAngle );    }
void rotateElbow        ( const int& endAt ) {  rotateServo( endAt , Elbow ,       currentElbowAngle );       }
void rotateWrist        ( const int& endAt ) {  rotateServo( endAt , Wrist ,       currentWristAngle );       }
void rotateWristRotate  ( const int& endAt ) {  rotateServo( endAt , WristRotate , currentWristRotateAngle ); }
void rotateGripper      ( const int& endAt ) {  rotateServo( endAt , Gripper ,     currentGripperAngle );     }

#ifdef SecondArm_Installed
void rotateBackElbow    ( const int& endAt ) {  rotateServo( endAt , BackElbow ,   currentBackElbowAngle );   }
void rotateBackGripper  ( const int& endAt ) {  rotateServo( endAt , BackGripper , currentBackGripperAngle ); }
#endif

void rotateServo ( const int& endAt , Servo& whichServo , int& currentAngle ) {  

    // Incoming parameters :   constant reference to an int denoting servo angle to move to,
    //                         reference to a Servo instance,
    //                         reference to the current angle of the Servo
      
    // Global variables    :   Uses    -   servoStepAngle, servoStepSpeed
    //                         Changes -   "current...Angle" of Servo passed by reference
    
    // Responsibilities    :   Moves desired Servo to requested angle.  Changes "current...Angle" of that Servo.
    //                         Why this function?  Moving a Servo from one angle to another can be a rather fast and brutal action,
    //                         not very conducive to finely controlling the action(s) of the arm.
    //                         This function slows the movements down to a smooth and managable speed.
    //                         Want faster or slower?   Just manipulate the values of globals servoStepAngle and servoStepSpeed.
    
    // Returns             :   Nothing.
    
    if ( currentAngle == endAt ) return;

    if ( currentAngle < 0 ) { 
      tone(5, 999, 250);  // Stop that!
      whichServo.write(currentAngle = 2);    delay(servoStepDelay);
      return;
    }

    if ( currentAngle > 180 ) { 
      tone(5, 999, 250);  // Stop that!
      whichServo.write(currentAngle = 178);    delay(servoStepDelay);
      return;
    }

    
    if ( endAt > currentAngle ) {
        for ( int i = currentAngle; i <= endAt; i += servoStepAngle ) { 
            
            whichServo.write(currentAngle = i);    delay(servoStepDelay); 
            
        }
    }
    else {
        for ( int i = currentAngle; i >= endAt; i -= servoStepAngle ) { 
                        
            whichServo.write(currentAngle = i);    delay(servoStepDelay); 
              
        }
    }
   
}


void deployArms ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -   
    //                         Changes -  currentBaseAngle, currentShoulderAngle, currentElbowAngle, 
    //                                    currentWristAngle, currentWristRotateAngle, currentGripperAngle,
    //                                    currentBackElbowAngle, currentBackGripperAngle
    
    // Responsibilities    :   Set arm(s) and global current...Angle's to initial working position. 
    //                         You can fine tune this to other behaviour if you wish.
    
    // Returns             :   Nothing.


    #ifdef SecondArm_Installed
    // Lower the (optional) back arm and grab an item.
                   BackGripper.attach  (BACKGRIPPER_PIN);   BackGripper.write (currentBackGripperAngle  = 180);
    delay(1500);   BackElbow.attach    (BACKELBOW_PIN);     BackElbow.write   (currentBackElbowAngle    =  88);  
    delay(1500);   BackElbow.detach    (  );      
    delay(3000);   BackGripper.write   (currentBackGripperAngle  =  00);
   
    //BackGripper.detach  (  );
   #endif

   
    delay( 500);   Shoulder.attach     (SHOULDER_PIN);      Shoulder.write    (currentShoulderAngle     =  90); 
    delay( 500);   Elbow.attach        (ELBOW_PIN);         Elbow.write       (currentElbowAngle        = 136); 
    
    delay( 500);   Wrist.attach        (WRIST_PIN);         Wrist.write       (currentWristAngle        =  70);
    delay( 500);   WristRotate.attach  (WRISTROTATE_PIN);   WristRotate.write (currentWristRotateAngle  = 160);
    delay( 500);   Gripper.attach      (GRIPPER_PIN);       Gripper.write     (currentGripperAngle      = 180);                                                        
    
    delay( 500);   Base.attach         (BASE_PIN);          Base.write        (currentBaseAngle         =  90);
                                                            
    delay(1000);   retractArm( ); 
 
}


void gamepadControl ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -   
    //                         Changes -  
    
    // Responsibilities    :   Control arm servos using 1Sheeld Gamepad Buttons...
    //
    //                                 Orange, Blue, Red, Green buttons are used in conjunction with Up, Left, Right, Down buttons.
    //                                 Up, Left, Right, Down buttons can be used independently.
    
    // Returns             :   Nothing.
    
    if (  GamePad.isBluePressed( )    ||  GamePad.isOrangePressed( ) || 
          GamePad.isRedPressed( )     ||  GamePad.isGreenPressed( ) ) {
                        
            gamepadControl_BORG( );
        
    }
    else if (  GamePad.isUpPressed( )       ||  GamePad.isDownPressed( ) || 
               GamePad.isLeftPressed( )     ||  GamePad.isRightPressed( ) ) {
      
            gamepadControl_RULD( );
        
    }
  
}

void gamepadControl_BORG ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -  currentBaseAngle, currentShoulderAngle, currentElbowAngle, 
    //                                    currentWristAngle, currentWristRotateAngle, currentGripperAngle,
    //                                    servoStepAngle 
    //                         Changes -  
    
    // Responsibilities    :   Control primary arm servos using the following 1Sheeld Gamepad Button Combinations...
    //
    //                          Green   ... Up/Down - Extend Arm/Retract Arm, Left/Right - Base 0 through 180 degrees
    //                          Blue    ... Up/Down - Shoulder 0 through 180 degrees, Left/Right - Gripper Open/Close
    //                          Red     ... Up/Down - Elbow 0 through 180 degrees, Left/Right nothing
    //                          Orange  ... Up/Down - Wrist 0 through 180 degrees, Left/Right - WristRotate 0 through 180 degrees
    
    // Returns             :   Nothing.

    if ( GamePad.isGreenPressed( ) ) {

            if ( GamePad.isUpPressed( ) ) {
                  // tone(5, 111, 250);
                  extendArm( );          
            }
    
            else if ( GamePad.isDownPressed( ) ) {
                  // tone(5, 222, 250);   // Activate tones for debugging purposes if you wish
                  retractArm( );
            }
            
            else if ( GamePad.isLeftPressed( ) ) {
                  // tone(5, 333, 250);
                  rotateBase( currentBaseAngle - servoStepAngle );          
            }
    
            else if ( GamePad.isRightPressed( ) ) {
                  // tone(5, 444, 250);
                  rotateBase( currentBaseAngle + servoStepAngle );
            }
      
    }
    else if ( GamePad.isBluePressed( ) ) {

            if ( GamePad.isUpPressed( ) ) {
                  // tone(5, 111, 250);
                  rotateShoulder( currentShoulderAngle - servoStepAngle );          
            }
    
            else if ( GamePad.isDownPressed( ) ) {
                  // tone(5, 222, 250);
                  rotateShoulder( currentShoulderAngle + servoStepAngle );
            }
                                
            else if ( GamePad.isLeftPressed( ) ) {
                  // tone(5, 333, 250);
                  rotateGripper( currentGripperAngle - servoStepAngle );          
            }
    
            else if ( GamePad.isRightPressed( ) ) {
                  // tone(5, 444, 250);
                  rotateGripper( currentGripperAngle + servoStepAngle );
            }
      
    }

    else if ( GamePad.isRedPressed( ) ) {

            if ( GamePad.isUpPressed( ) ) {
                  // tone(5, 111, 250);
                  rotateElbow( currentElbowAngle + servoStepAngle );          
            }
    
            else if ( GamePad.isDownPressed( ) ) {
                  // tone(5, 222, 250);
                  rotateElbow( currentElbowAngle - servoStepAngle );
            }
                        
            else if ( GamePad.isLeftPressed( ) ) {
                  // tone(5, 333, 250);
                  // Add functionality here          
            }
    
            else if ( GamePad.isRightPressed( ) ) {
                  // tone(5, 444, 250);
                  // Add functionality here  
            }
      
    }

    else if ( GamePad.isOrangePressed( ) ) {

            if ( GamePad.isUpPressed( ) ) {
                  // tone(5, 111, 250);
                  rotateWrist( currentWristAngle - servoStepAngle );          
            }
    
            else if ( GamePad.isDownPressed( ) ) {
                  // tone(5, 222, 250);
                  rotateWrist( currentWristAngle + servoStepAngle );
            }
                    
            else if ( GamePad.isLeftPressed( ) ) {
                  // tone(5, 333, 250);
                  rotateWristRotate( currentWristRotateAngle - servoStepAngle );          
            }
    
            else if ( GamePad.isRightPressed( ) ) {
                  // tone(5, 444, 250);
                  rotateWristRotate( currentWristRotateAngle + servoStepAngle );
            }
        
    }

}

void gamepadControl_RULD ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -  currentBackElbowAngle, currentBackGripperAngle,
    //                                    servoStepAngle 
    //                         Changes -  
    
    // Responsibilities    :   Control 2DoF Arm (optional secondary arm).
    //                         If you don't have a second arm, you can easily redeploy these buttons
    //                         for more actions on your primary arm.
    //
    //                          Up    ...   Raise Back Arm
    //                          Down  ...   Lower Back Arm
    //                          Left  ...   Open Back Gripper
    //                          Right ...   Close Back Gripper

    
    // Returns             :   Nothing.

      if ( GamePad.isUpPressed( ) ) {           
           // tone(5, 111, 250);  // Activate tones for debugging purposes if you wish
           #ifdef SecondArm_Installed
           BackElbow.attach (BACKELBOW_PIN);
           rotateBackElbow  ( currentBackElbowAngle - servoStepAngle ); 
           BackElbow.detach ( );
           #endif         
      }
    
      else if ( GamePad.isDownPressed( ) ) {
           // tone(5, 222, 250);
           #ifdef SecondArm_Installed
           BackElbow.attach (BACKELBOW_PIN);
           rotateBackElbow  ( currentBackElbowAngle + servoStepAngle );
           BackElbow.detach ( );
           #endif
      }
                                
     else if ( GamePad.isLeftPressed( ) ) {
           // tone(5, 333, 250);
           #ifdef SecondArm_Installed
           BackGripper.attach (BACKGRIPPER_PIN);
           rotateBackGripper  ( currentBackGripperAngle + servoStepAngle );  
           //BackGripper.detach ( );  
           #endif      
     }
    
     else if ( GamePad.isRightPressed( ) ) {
           // tone(5, 444, 250);
           #ifdef SecondArm_Installed
           BackGripper.attach (BACKGRIPPER_PIN);
           rotateBackGripper  ( currentBackGripperAngle - servoStepAngle );
           //BackGripper.detach ( );
           #endif
     }

}


void shaveAndAHaircut ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -   
    //                         Changes -  
    
    // Responsibilities    :   Plays the tune. 
    
    // Returns             :   Nothing.

    // Individual notes
    int melody_ShaveHaircut[] = {
      NOTE_C4, NOTE_G3, NOTE_G3, NOTE_A3, NOTE_G3, 0, NOTE_B3, NOTE_C4
    };
    // Note durations: 4 = quarter note, 8 = eighth note, etc.
    int noteDurations_ShaveHaircut[] = {
      4, 8, 8, 4, 4, 4, 4, 4
    };
    int noteDuration;
  
    for (int thisNote = 0; thisNote < 8; thisNote++) {
  
      // To calculate the note duration, take one second divided by the note type.
      // e.g. quarter note = 1000 / 4, eighth note = 1000/8, etc.
      noteDuration = 1000 / noteDurations_ShaveHaircut[thisNote];
      
      tone(SPEAKER_PIN, melody_ShaveHaircut[thisNote], noteDuration);
  
      // To distinguish the notes, set a minimum time between them.
      // The note's duration + 30% seems to work well      
      delay(noteDuration * 1.30);
      
      // Stop the tone playing
      noTone(SPEAKER_PIN);
      
    }

}





void retractArm ( const int& shoulderStop ) {

    // Incoming parameters :   constant reference to an int denoting ending angle of Shoulder servo.
      
    // Global variables    :   Uses    -  servoStepAngle, servoStepDelay 
    //                         Changes -  currentShoulderAngle, currentElbowAngle, currentWristAngle
    
    // Responsibilities    :   Retracts the arm in a predetermined manner.  
    //                         A function of convenience, takes the place of the user having to perform
    //                         a number of actions.   Modify to suit your needs. 
    
    // Returns             :   Nothing.
    
    int braking = 0;
    
    for ( int i =  currentShoulderAngle; i <=  shoulderStop; i += servoStepAngle ) { 
        
        Shoulder.write(currentShoulderAngle = i);
        
        if ( (i+86) <= 180 ) Elbow.write(currentElbowAngle = i+86);
        
        // Keep the wrist as horizontal as possible
        if      ( i <= 90  ) Wrist.write(currentWristAngle = 140);
        else if ( i >= 150 ) Wrist.write(currentWristAngle =  75);             
        else                 Wrist.write(currentWristAngle = ( 140 - ( i - 90 )) ); 
        
        // As you rise towards 90 degrees, weight is working for you, keeping things slow.
        // Past 90 degrees, weight works against you as it increases the speed of the arm,
        // resulting in an momentum that builds the farther you go.
        // The following formula calculates an extra time in servo delay, resulting in a
        // braking action.
        braking = ( i > 90 ) ?  ( ( i - 90 ) / 10 ) * 15  : 0;
        delay(servoStepDelay+braking); 
        
    }

  
    delay(1000);
  
}

void extendArm ( const int& shoulderStop , const bool& withItem ) {

    // Incoming parameters :   constant reference to an int denoting ending angle of Shoulder servo.
    //                         constant reference to a bool denoting if the arm is holding an item ( 0 - no , 1 - yes ).
      
    // Global variables    :   Uses    -  servoStepAngle, servoStepDelay 
    //                         Changes -  currentShoulderAngle, currentElbowAngle, currentWristAngle
    
    // Responsibilities    :   Extend the arm in a predetermined manner.  
    //                         A function of convenience, takes the place of the user having to perform
    //                         a number of actions.   Modify to suit your needs.
    
    // Returns             :   Nothing.
   

    for ( int i =  currentShoulderAngle; i >=  shoulderStop; i -= servoStepAngle ) { 
        
        if ( (i+86) <= 180 ) Elbow.write(currentElbowAngle = i+86);  
        
        Shoulder.write(currentShoulderAngle = i);     
        
        if ( withItem ) {
            // Keep the wrist as horizontal as possible
            if      ( i <= 90  ) Wrist.write(currentWristAngle = 140);
            else if ( i >= 150 ) Wrist.write(currentWristAngle =  75);             
            else                 Wrist.write(currentWristAngle = ( 140 - ( i - 90 )) );
        }
        else {
            // Move the wrist to angle up onto an item that is to be grasped
                                 Wrist.write(currentWristAngle = 160-i);
        }
        
        delay(servoStepDelay); 
    }


    
    delay(1000);
  
}



void grabVerticalItem ( void ) {
 
    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -   
    //                         Changes -  
    
    // Responsibilities    :   Grabs a vertical item (something "standing up" rather than "laying down") with the Gripper  
    //                         A function of convenience, takes the place of the user having to perform
    //                         a number of actions.   Modify to suit your needs.
    
    // Returns             :   Nothing.
      
    delay( 100);   rotateWristRotate  ( 70);  // Rotate to grab a vertical item ( ie | )
    delay( 100);   rotateGripper      ( 00);  // Open 
    delay(2000);   rotateGripper      (180);  // Close
    //delay(2000);   rotateWristRotate  (160);  // Rotate item to horizontal ( ie --- )
    //delay( 100);   rotateWrist        (180);  // Raise
    //delay(1000);
  
}

void dropItem ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -  
    //                         Changes -  
    
    // Responsibilities    :   Releases an item from the Gripper.  
    //                         A function of convenience, takes the place of the user having to perform
    //                         a number of actions.   Modify to suit your needs.  
    
    // Returns             :   Nothing.
    
   
    delay( 500);   rotateWrist    ( 40);   // Lower  
    delay( 500);   rotateGripper  ( 00);   // Open 
    delay( 500);   rotateShoulder ( 44);   // Raise arm a little
    delay( 500);   rotateGripper  (180);   // Close
    delay( 500);   rotateWrist    (180);   // Raise 
    delay( 500);
  
}





void autonomousDemo ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -   
    //                         Changes -  currentBaseAngle
    
    // Responsibilities    :   Run the arm through a series of actions for demonstration purposes.
    
    // Returns             :   Nothing.
    
    
    // Announce start of demo
    shaveAndAHaircut  (   );

    
    // Grab an item from the front
    rotateBase        ( 90);    // Move arm to front
    rotateWristRotate ( 70);    // Rotate to grab a vertical item ( ie | )
    rotateGripper     ( 00);    // Open before moving arm into position                
    extendArm         ( 14);    // Modify to suit your height requirement   
    rotateWrist       (130);    // Tweak wrist angle if required  
    rotateGripper     (180);    // Close on item     
    retractArm        (110);    // Move back to this Shoulder angle
  
    delay(1000);
    
    // Place it to the right
    rotateBase        (180);    // Move arm full right
    rotateWristRotate ( 70);    // Rotate to place vertical item ( ie | )           
    extendArm         (14,1);   // Modify to suit your height requirement
    //rotateWrist       (???);    // Tweak wrist angle if required    
    rotateGripper     ( 00);    // Open to release item     
    retractArm        (   );    // Park arm


    #ifdef SecondArm_Installed   
    grabItemFromBackArm (   );  // Interact with back arm (optional)    
    retractArm          (   );  // Return to standard starting position
    #endif 
    
    
    // Grab an item from the right
    rotateBase        (180);    // Move arm full right
    rotateWristRotate ( 70);    // Rotate to grab a vertical item ( ie | )
    rotateGripper     ( 00);    // Open before moving arm into position                
    extendArm         ( 14);    // Modify to suit your height requirement 
    //rotateWrist       (???);    // Tweak wrist angle if required
    rotateGripper     (180);    // Close on item      
    retractArm        (110);    // Move back to this Shoulder angle

    delay(1000);
    
    // Place it in front
    rotateBase        ( 90);    // Move arm to front
    rotateWristRotate ( 70);    // Rotate to place vertical item ( ie | )           
    extendArm         (14,1);   // Modify to suit your height requirement
    rotateWrist       (130);    // Tweak wrist angle if required    
    rotateGripper     ( 00);    // Open to release item      
    retractArm        (   );    // Park arm
    
    
}




#ifdef SecondArm_Installed
void grabItemFromBackArm ( void ) {

    // Incoming parameters :   Nothing.
      
    // Global variables    :   Uses    -  Wrist, WristRotate, Gripper, 
    //                                    BackElbow, BackGripper 
    //                         Changes -  currentWristAngle, currentWristRotateAngle, currentGripperAngle, 
    //                                    currentBackElbowAngle, currentBackGripperAngle
    
    // Responsibilities    :   The primary arm will reach back, grab an item from the secondary arm,
    //                         raise it in the air, and then return it to the secondary arm.
    //                         Admittedly not very useful, just a demo of arm capabilities. 
    //                         A function of convenience, takes the place of the user having to perform
    //                         a number of actions.   Modify to suit your needs.
    //
    //                         Why all the attach's and detach's?   With all the servos operating, I was
    //                         getting feedback through some of the circuits, causing servo flutter and 
    //                         undesirable effects.  Shutting down unnecessary servos solved the problem.  
    //                         You may or may not have the same issues.
    
    // Returns             :   Nothing.

    // Announce beginning of sequence.
    for ( i = 0; i < 4; i++ ) { tone(5, 444, 250); delay(500); tone(5, 888, 250); delay(500); }
    
    // Grab something if BotBoarduino BotBoarduino onboard push button C is pressed 
    if ( digitalRead ( ONBOARD_PB_C ) == LOW ) {

                       BackGripper.attach  (BACKGRIPPER_PIN);
       delay( 500);    rotateBackGripper   (180);                                      // Open to accept item
                       while ( digitalRead ( ONBOARD_PB_C ) == LOW ) delay(500);       // Wait until button released
                       rotateBackGripper   ( 00);                                      // Close to grab item
      
    }    

    
    // Raise back arm.
    BackElbow.attach    (BACKELBOW_PIN);
    rotateBackGripper   ( 00);      // Make sure gripper is closed    
    rotateBackElbow     ( 16); 
    BackElbow.detach    (   );        
    
    // Orient wrist and gripper for the task.           
    rotateWrist         (120);    
    rotateWristRotate   ( 70);
    rotateGripper       ( 00);  // Open in preparation  
    
    // Move front arm backwards to contact item.
    rotateBase          ( 90);
    rotateShoulder      ( 90); 
    rotateElbow         ( 00);    
   
    // Exchange item from back to front.
    // My gripper servos are configured differently...
    // Gripper closes at 180, but BackGripper opens at 180.
    // Adjust your angles to suit your setup.
    delay( 500);   BackGripper.attach  (BACKGRIPPER_PIN);    
    delay( 500);   rotateGripper       (180);       // Close on item
    delay( 500);   rotateBackGripper   (180);       // Open to release item
    delay( 500);   BackGripper.detach  (   );


    // Raise in the air, then return item.    
    delay( 500);    rotateElbow        ( 90); 
    delay( 500);    rotateShoulder     ( 88);
    delay(2000);    rotateElbow        ( 00);

     
    // Exchange item from front to back.
    // My gripper servos are configured differently...
    // Gripper opens at 00, but BackGripper closes at 00.
    // Adjust your angles to suit your setup.
                    BackGripper.attach  (BACKGRIPPER_PIN);    
    delay( 500);    rotateBackGripper   ( 00);      // Close to grip item   
    delay( 500);    rotateGripper       ( 00);      // Open to release item

    
    // At this point, there a lot of power being drawn because so many servos are operating.
    // Shut down unnecessary servos to lessen the load on your circuits. 
    // Leave the Shoulder attached because it's holding up the arm.
    delay( 500);    Wrist.detach        ( );     
                    WristRotate.detach  ( );
                    Gripper.detach      ( );
    
    // Lower back arm
    BackElbow.attach    (BACKELBOW_PIN);
    rotateBackElbow     ( 88);      // Lower back arm
    BackElbow.detach    (   );

    
    // Raise front arm.   
    delay( 500);    rotateElbow        (180);    
    delay( 500);    rotateShoulder     (150);


    // Reattach to front arm servos.
    Wrist.attach        (WRIST_PIN);
    WristRotate.attach  (WRISTROTATE_PIN);
    Gripper.attach      (GRIPPER_PIN);
    
    delay(500);
 
}
#endif
