////////////////////////////////////////////////////////////////////////////
// Name: hardware.c
// Author: B. Gian James <gian@BattleDroids.net>
// Description: Utility functions for external hardware, etc.
//
// $Id: hardware.c,v 1.1.1.1 2009/08/17 08:10:59 gian Exp $
////////////////////////////////////////////////////////////////////////////
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include "common.h"
#include "hardware.h"

static volatile uint8	timer;
static volatile uint16	timerTime;
static volatile uint16	uptime;
static volatile uint8	usartTimer;
static uint8			ldr;

void	ClearRxTimer()
{	
	usartTimer = 0;
}

uint8	TimeSinceLastRx()
{
	return usartTimer;
}
uint16	TimeSinceGameStart()
{
	return uptime;
}

uint8	TimeOneSec()
{
	return ( timer >= InterruptsPerSec );
}

uint16	TimeSinceTimer()
{
	return timerTime;
}

void InitTimer()
{
	timer = 0;
	timerTime = 0;
	uptime = 0;
	usartTimer = 0;
	EnableCTCTimer();		// Enable compare match timer
	EnableCTCInterrupt();	// fire interrupt on match
	EnableCTCInterruptB();	// interrupt for USART timer
	SetCompareMatch();		// Set match value
	SetUARTCompareMatch();	// ditto
	
	BSET(SREG,7);			// Global interrupt flag
}

void InitRandom(uint8 l)
{
	ldr = l;
	if (ldr)
	{
		// 20MHz / Prescaler 128 = 156 kHz
		BSET(ADCSRA,ADPS2);
		BSET(ADCSRA,ADPS1);
		BSET(ADCSRA,ADPS0);
		BSET(ADMUX,REFS0);	// ADC reference voltage
		BSET(ADMUX,ADLAR);	// Right-adjust 10-bit to 8-bit value (0 - 255)
		BCLR(ADCSRA,PRADC);	// disable power reduction ADC bit
		BSET(ADCSRA,ADEN);	// Enable ADC
	}
	BSET(TCCR1B,CS10);
}

uint8 GetRandom()
{
	uint16	randval = TCNT1;
	TCNT1 = 0;
	if (ldr)
	{
		BSET(ADCSRA,ADSC);	// Start ADC measurements (autocleared by hw)
		// Takes 13 clock cycles
		_delay_ms(1);
		uint8 analog = ADCH;
		
		randval = (randval % analog);
		BCLR(ADCSRA,ADSC);
	}

	return (randval);
}

ISR(TIMER0_COMPA_vect)
{
	++timer;
	if (timer == InterruptsPerSec) {
		timer = 0;
		++timerTime;
		++uptime;
	}
}

ISR(TIMER0_COMPB_vect)
{
	// One tick is 1/100 sec
	++usartTimer;
}