////////////////////////////////////////////////////////////////////////////
// Name: usart.c
// Author: B. Gian James <gian@BattleDroids.net>
// Description: USART functions for ATmega162
//
// $Id: usart.c,v 1.1.1.1 2009/08/17 08:10:59 gian Exp $
////////////////////////////////////////////////////////////////////////////
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include "common.h"
#include "hardware.h"
#include "usart.h"

#define EnableRxInterrupt() 	BSET(UCSR0B,RXCIE0)
#define DisableRxInterrupt()	BCLR(UCSR0B,RXCIE0)

static volatile char 	rxBuff[MAX_RX_SZ];
static volatile uint8	rxHead;
static volatile uint8	rxTail;

void ReadLine(char * buff)
{
	while ( (TimeSinceLastRx() <= UARTRxDelay) ) ;	// half a second
	
	StopTimer();
	ClearRxTimer();
	
	uint8 x = 0;
	while(rxHead != rxTail)
	{
		buff[x] = rxBuff[rxHead];
		if ((++rxHead) >= MAX_RX_SZ)
			rxHead = 0;
		++x;
	}
}

char ReadChar()
{
	while ( !(UCSR0A & (1<<RXC0)) ) ;
	return UDR0;
}

void WriteInt(const int i)
{
	char buff[5];
	WriteRAMString(itoa(i,buff,10));
}
void WriteRAMString(const char * str)
{
	while (*str != 0x00)
		WriteChar(*(str++));
}
void WriteString(const char * str)
{
	while (pgm_read_byte(str) != 0x00)
		WriteChar(pgm_read_byte(str++));
}

void WriteChar(unsigned char c)
{
	loop_until_bit_is_set(UCSR0A,UDRE0);
	UDR0 = c;
}

void InitUSART()
{

	rxHead = rxTail = 0;
	
#include <util/setbaud.h>
	UBRR0H = UBRRH_VALUE;
	UBRR0L = UBRRL_VALUE;
#if USE_2X
	BSET(UCSR0A,U2X0);
#else
	BCLR(UCSR0A,U2X0);
#endif
	
	BSET(UCSR0B,RXEN0);
	BSET(UCSR0B,TXEN0);
	
	EnableRxInterrupt();
}

ISR(USART_RX_vect)
{
	StopTimer();
	ClearRxTimer();
	
	rxBuff[rxTail] = ReadChar();
	if ((++rxTail) >= MAX_RX_SZ)
		rxTail = 0;
	
	StartTimer();
}
