// Special thanks to:

// http://www.psychicorigami.com/2012/12/29/making-a-musical-robot-santa-ornament-using-an-attiny-85/
// http://www.instructables.com/id/Fully-functional-Duplo-traffic-light/step2/Programming-the-microcontroller/
//  http://hlt.media.mit.edu/?p=1706
//  http://www.arduino.cc/en/Tutorial/PlayMelody
//  http://code.google.com/p/arduino-tiny/

#include <avr/power.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <avr/pgmspace.h>
#include "notes.h"

// If using a standalone ATiny85, adjust these variables:
#ifdef __AVR_ATtiny85__
#define SWITCH_PIN 0 // ATtiny pin 5 -- the switch MUST be on this pin
#define LED_PIN 2 // ATtiny pin 7
#define LED_PINA 4 // ATtiny pin 3
#define SPEAKER_PIN 1 // ATtiny pin 6 -- the speaker MUST be on this pin, or else crappy sound will be produced
#else
// If using an Arduino or ATmega32 on a breadboard, adjust these variables:
#define SWITCH_PIN 2 // Button
#define LED_PIN 3 // LED 1
#define LED_PINA 5 // LED 2
#define SPEAKER_PIN 13 // Speaker
#endif

#define BPM 45 // Speed of beats -- lower number, slower beats
#define NOTES_LEN 24

//Airship Victory Music by  Koji Kondo
prog_uint16_t notes[2*NOTES_LEN] PROGMEM = {
  NOTE_E6, HALF_BEAT,
  NOTE_REST, QUARTER_BEATA,
  NOTE_E6, QUARTER_BEAT,
  NOTE_F6, QUARTER_BEAT,
  NOTE_E6, QUARTER_BEAT,
  NOTE_D6, QUARTER_BEAT,
  NOTE_C6, QUARTER_BEAT,
  NOTE_G5, QUARTER_BEAT,
  NOTE_C6, QUARTER_BEAT,
  NOTE_D6, QUARTER_BEAT,
  NOTE_REST, QUARTER_BEAT,
  NOTE_G6, QUARTER_BEAT,
  NOTE_G6, HALF_BEAT,
  NOTE_REST, QUARTER_BEATA,
  NOTE_G6, QUARTER_BEATA,
  NOTE_D6, QUARTER_BEAT,
  NOTE_G6, QUARTER_BEAT,
  NOTE_G6S, QUARTER_BEAT,
  NOTE_D6S, QUARTER_BEAT,
  NOTE_G6S, QUARTER_BEAT,
  NOTE_Bb6, QUARTER_BEAT,
  NOTE_F6, QUARTER_BEAT,
  NOTE_Bb6, QUARTER_BEAT,
  NOTE_C7, BEAT,
};

void playSong() {
  long beatDurationMillis = (60l*1000l)/(BPM*BEAT);
  for ( int i = 0; i < NOTES_LEN; i++ ) {
    int note = i << 1;
    long freq = pgm_read_word_near(notes + note);
    long duration = pgm_read_word_near(notes + note + 1);
#ifdef DEBUG
    Serial.print("freq: ");
    Serial.println(freq);
    Serial.print(", duration: ");
    Serial.println(duration);
#endif
    if ( freq == NOTE_REST ) {
      noTone(SPEAKER_PIN);
      delay(duration*beatDurationMillis);
    }
    else {
      tone(SPEAKER_PIN, freq);
      digitalWrite(LED_PIN, HIGH);
      digitalWrite(LED_PINA, HIGH);
      delay(5*duration*beatDurationMillis/6);
      noTone(SPEAKER_PIN);
      digitalWrite(LED_PIN, LOW);
      digitalWrite(LED_PINA, LOW);
      delay(duration*beatDurationMillis/6);
    }
  }
  noTone(SPEAKER_PIN);
}

void setup() {
  pinMode(SWITCH_PIN, INPUT);
  digitalWrite(SWITCH_PIN, HIGH);
  
  pinMode(LED_PIN, OUTPUT);
  pinMode(LED_PINA, OUTPUT);
  pinMode(SPEAKER_PIN, OUTPUT);

  ADCSRA &= ~(1<<ADEN); //turn off ADC
  ACSR |= _BV(ACD);     //disable the analog comparator
  
#ifndef DEBUG
  Serial.begin(9600);
#endif
}

void gotoSleep() {
#ifdef __AVR_ATtiny85__
  GIMSK |= 1<<PCIE;  //Enable Pin Change Interrupt
  PCMSK |= 1<<PCINT0; //Watch for Pin Change on Pin5 (PB0)
#else
  attachInterrupt(0, interrup, FALLING);
#endif
  
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);
  sleep_enable();
  sleep_mode();
      
  // waking up from sleep mode.
  sleep_disable();

#ifdef __AVR_ATtiny85__
  GIMSK &= ~(1<<PCIE); //Disable the interrupt so it doesn't keep flagging
  PCMSK &= ~(1<<PCINT0);
#else
  detachInterrupt(0);
#endif
}

void loop() {
  gotoSleep();
      
  // Wait for the button to be released
  while (digitalRead(SWITCH_PIN) == LOW) {  }
  
  playSong();
  delay(10);
  digitalWrite(LED_PIN, LOW);
  digitalWrite(LED_PINA, LOW);
}

void interrup() {}

// Interrupt for PIN0 falling edge
ISR(PCINT0_vect) {
  
}
