// Source             ::     RoverV2_FollowWall_V2

// Description        ::     Drive straight ahead, hugging the wall on the left. 
//                           Go around corners.  
//                           Pivot right when faced with a wall in front.
//                           https://www.youtube.com/watch?v=D7FLrwdpCOM 


//                        Recommended components ( or equivalent ) 

//                        1)  http://www.robotshop.com/dfrobotshop-rover-v2---arduino-compatible-tracked-robot-autonomous-dev-kit.html
//
//                        OR
//
//                        1)  Any Arduino powered vehicle platform
//                        2)  Dagu Mini Pan and Tilt Kit
//                        3)  SeeedStudio 3 Pin Ultrasonic Range Finder
//                        4)  Dagu Compound Infrared Sensor
//                        5)  TM1638 Display Module
//                        6)  DFRobot 7.4V Lipo 2200mAh Battery

// Original Program   ::     Joe M  / arduino (at) bellaliant.net / March 2016

//                           This program works best on my Rover.  Make an appointment, lol.
//                           To help you "fine tune" it to your platform, I've labelled key 
//                           statements with "4U2DO" ( for you to do ).  Just do a search for
//                           occurances of that string in this sketch.

// Modified By        ::     Your Name



//  This program is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License 
//  as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  Lesser General Public License for more details.
//
//  For a copy of the GNU Lesser General Public License, 
//  write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA



// *********************************************************************************************************************************************************************
// Optional components and general definitions
// *********************************************************************************************************************************************************************


String  whoAmI = "LUNO   4";  // Name your Rover, mine is Lunokhod 4


// How fast you want the DFRobotShop Rover V2 to go... the stronger your battery pack, the lower your number should be.
// 90 seems to be a good setting for http://www.robotshop.com/dfrobot-7-4v-lipo-2200mah-battery.html 

// 4U2DO ... Because so much depends on battery and motor size, you should adjust this speed to suit your specific vehicle.

#define BASELINESPEED     90

// These values are the desired upper and lower ranges of IR reflected light read by the compound IR sensor.
// Low values  ( ie  20,  21,  22 ) denote little light is being reflected, so the object is far away , 
// High values ( ie  58,  59,  60 ) denote more light is being reflected, so the object is close. 
// Fine tune these values to suit your situation.

// These settings are "light saturation"!    
// Do not confuse them with units of distance!  ( ie 20 does not mean 20 centimeters! )
// LOW means FAR!  HIGH means CLOSE!
// Normally, the Dagu Compound IR Sensor returns values in a range from 20's to 600's.   We don't need such precision for this simple job.
// So, in the function that reads the Dagu, the value is divided by 10 to result in smaller numbers... ie 200/10 == 20 , 207/10 == 20, 213/10 == 21, etc.

// 4U2DO ...  More than anything, these settings will affect the performance of your vehicle.
//            Light reflectivity will vary from surface to surface, so even when you get this 
//            adjusted to your needs, the vehicle will work completely differently when you test
//            it against a different surface ( ie a wall with flat paint vs a wall with gloss paint )
//            You'll need to carefully consider these settings... a bit of experimentation will absolutely
//            be required before you get the readings you need for the surface you're attempting to hug.

#define IR_HOWFAR             25   // The maximum distance from the wall you want to be.
#define IR_HOWCLOSE           30   // The minimum distance from the wall you want to be.
#define IR_STOP_PIVOT_HERE    50   // For "right pivot" logic, keep pivoting as long as IR reading is more than this.


// Used by the motor controllers
#define AHEAD         LOW
#define BACK          HIGH

// Used by calls to rangeFinders()
#define panIR         0    // Pan Servo mounted compound infrared sensor
#define frontUS       1    // Front mounted ultrasonic sensor


#define    TM1638INSTALLED                // Only activate this line if you have a TM1638 LED/Key module installed https://code.google.com/p/tm1638-library/
                                          // The TM1638 is an excellent addition to the Rover as it provides a nice onboard I/O package.
                                          
//#define    XBeeINSTALLED                  // Communicate with a monitoring station.   
                                          // Only activate this line if you are using XBee, OR ...
                                          // ... if you have your Rover connected to the PC with a USB cable.
                             
                                          





// **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T ****
// ***********************************************************************************************************************************************************************
// Pinouts... IMPORTANT!!!! ... set the pinouts to your hardware configuration
//
// NEVER USE Digital pins 5,6,7,8 ... they are permanently connected to the onboard motor controller on all versions of the DFRobotShop Rover V2.
//                                    If you are using a different vehicle, ensure your motor controller is configured correctly.  
//
// Digital pins 0 and 1 can only be used if you ARE NOT using Serial communications ( ie Serial.begin() ).
// Also note that having components connected to D0 and D1 will interfere with uploading your Sketch.  
//
// If you plan on someday expanding your Rover to use a second Arduino board ( to add more components, sensors, etc ), do not use (analog pins) A4 (SDA) and A5 (SCL)
// http://www.instructables.com/id/I2C-between-Arduinos/?ALLSTEPS 
// 
// ***********************************************************************************************************************************************************************

// ********** Digital Pins D0 thru D13 **********

// Using 0 and 1 disables upload!     

#define DO_NOT_USE_0        0
#define DO_NOT_USE_1        1

#define PANPIN              2        // Which pin the pan servo is connected to.  For example, here it is attached to (digital pin) D2.
                                     // Assuming you are using the I/O Expansion Shield V7 ( http://www.dfrobot.com/wiki/index.php/IO_Expansion_Shield_for_Arduino_V7_SKU:DFR0265 ) ,
                                     // configure your three sensor leads so that ORANGE is Signal and BROWN is ground.  The power RED is in the middle.

//#define TILTPIN             3        // Only activate this line if you want to use the tilt servo.
                                     // The tilt servo is not needed for the original functionality of this program,
                                     // so you might want to free up pin 3 for other purpses.
          
#define NOT_USED_4          4

#define MOTORRIGHTSPEED     5        // Nothing connected to these pins , but the DFRobotShop Rover V2 onboard motor controller uses these circuits.
#define MOTORLEFTSPEED      6        // If you're using a different vehicle platform, configure your motor controllers accordingly.
#define MOTORRIGHTDIREC     7
#define MOTORLEFTDIREC      8

#define NOT_USED_9          9

// Pins D10, D11, D12 available UNLESS TM1638 installed...
#ifdef TM1638INSTALLED    // "ifdef's" are one of those obscure C++ features many beginning programmers don't know about.
                          // It is a neat feature that allows "conditional compilation".  In the case of this program,
                          // the various code segments using the TM1638 LED/Key module will only be included in the compiled executable
                          // if you defined TM1838INSTALLED above ( ie #define TM1638INSTALLED ).
                          // For more on "conditional compilation", go to http://www.cplusplus.com/doc/tutorial/preprocessor/

  #define TM1638STBPIN     10        // TM1638 Strobe D10.
  #define TM1638DTOPIN     11        // TM1638 Data D11.
  #define TM1638CLKPIN     12        // TM1638 Clock D12.
  
#endif
  
  
#define IRLEDSPIN          13        // IR Compound Eye LEDs D13.




// ********** Analog Pins A0 thru A5 **********

// FYI... using analog pins for digital I/O is just the same as using digital ones.

// A0 is referred to as Pin 14     A1 is referred to as Pin 15
// A2 is referred to as Pin 16     A3 is referred to as Pin 17
// A4 is referred to as Pin 18     A5 is referred to as Pin 19

// A2 thru A5 (plus digital pin D13) are used in this program for the Dagu Compound Infrared Sensor.
// To help you with the connections, I provide this "color-coded" explanation.   
// In this explanation, I am using a ribbon cable consisting of the following colors...
// Yellow , Green , Blue , Violet , Gray , White , Black

// Holding the sensor array with the back facing you and the pins facing up, the above color sequence should apply.
// Yellow being VCC and Black being GND with Green , Blue , Violet , Gray , White between.

// When mounted on the pan/tilt mechanism with the pins facing up, this is the color sequence you should see when looking
// at the unit from the *front*...

// Black (Ground) , White (IR LEDs) , Grey (Right) , Violet (Bottom) , Blue (Left) , Green (Top) , Yellow (Power).

// On the I/O Expansion Shield V7 , the following connections apply...

// A2 (Signal) - Grey (Right), A2 (Power) - Yellow (Power) , A2 (Ground) - Black (Ground)
// A3 (Signal) - Blue (Left)
// A4 (Signal) - Green (Top) 
// A5 (Signal) - Violet (Bottom) 
// D13 - White

// The DFRobotShop Rover V2 uses these circuits for the wheel encoders.  
// Not used by this program, you can redeploy them for other uses...
// ...I used A1 for my forward ultrasonic sensor.
//#define RIGHTENCODERPIN         A0          // Right motor encoder  A0.
//#define LEFTENCODERPIN          A1          // Left motor encoder   A1.


#define RANGEFINDERPIN          15           // (A1) The ultrasonic rangefinder mounted always facing forward.


#define IRRIGHTPIN              A2          // (Grey)   IR Compound Eye Right (analog pin)    A2.
#define IRLEFTPIN               A3          // (Blue)   IR Compound Eye Left (analog pin)     A3.

// The top/bottom IRs are not needed for the original seek/destroy functionality of this program,
// so you might want to free up pins A4 and A5 for other purpses.
//#define IRTOPPIN                A4          // (Green)  IR Compound Eye Top (analog pin)      A4.
//#define IRBOTTOMPIN             A5          // (Violet) IR Compound Eye Bottom (analog pin)   A5.


// A4 == SDA, A5 == SCL if second Arduino board attached



                     
                  
                             


// *********************************************************************************************************************************************************************
// Required includes
// *********************************************************************************************************************************************************************

#include <stdio.h>
#include <Servo.h> 
#include <Ultrasonic.h>


#ifdef TM1638INSTALLED 
// *********************************************************************************************************************************************************************
//  TM1638 Board
// *********************************************************************************************************************************************************************
//
// https://code.google.com/p/tm1638-library/
//
// Based on a TM1638 chip, the board uses a serial communications interface to enable three-wire control of the LEDs and reading of the pushbuttons.
//

#include <TM1638.h>

// *********************************************************************************************************************************************************************
//  End of TM1638 definitions
// *********************************************************************************************************************************************************************
#endif




/*********************************************************************************************************************************************************************/
//  Global Variables.
//  Professional programmers cringe at the thought of globals, but they seem to be generally acceptable in the "wild west" Arduino world.
//  Prudent use of them simplifies program design since variable passing via parameters between functions are reduced.
/*********************************************************************************************************************************************************************/

// An homage to every program I've written since my WATFIV days.
// Every program needs an "int i", right?
int i;

// How fast you want the Rover to go
byte speedRight = BASELINESPEED, speedLeft = BASELINESPEED;

#ifdef TM1638INSTALLED 
// Used for TM1638
// define module pins... data DT0 11, clock CLK 12, strobe STB 10
TM1638 tm1638(TM1638DTOPIN, TM1638CLKPIN, TM1638STBPIN);
char outputString[128];
String  tm1638Msg; 
int tm1638LEDs, tm1638Buttons;
#endif


// Create servo object(s)
Servo panServo;

#ifdef TILTPIN
Servo tiltServo; 
#endif


// Create ultrasonic sensor object(s)
Ultrasonic ultrasonicFront(RANGEFINDERPIN); 


// You have access to the current sensor readings at any time without incurring the CPU overhead
// of calling rangeFinders().  These will contain the readings from the most recent call to rangeFinders().
int frontRangeInCentimeters;
int irValue, irLeft, irRight;
int irValue_baseline, irLeft_baseline, irRight_baseline;


// Used throughout the program when you want to time events
unsigned long timeIsNow;


// Enumerated types to record "what you are doing"... a simple way for functions to communicate with each other

enum roverAction { 
  stopped, goingForward, goingBackward, veeringLeftForward, veeringRightForward, 
  veeringLeftBackward, veeringRightBackward, turningLeft , turningRight 
} whatAreYouDoing;



 
/*********************************************************************************************************************************************************************/
// Function Prototypes
/*********************************************************************************************************************************************************************/

void     followLeftWall    ( void );
void     wallAhead         ( void );
void     pivotRight        ( void );
void     stopMotors        ( void );
int      compoundIRSensor  ( void );
boolean  timeIsUp          ( unsigned long& then , unsigned int desiredInterval);



#ifdef TM1638INSTALLED
void     setDistance       ( void );
void     (* resetFunc)     ( void ) = 0;  // Declare reset function @ address 0.
                                          // Calling this function is the same as pressing reset on the Arduino board.
#endif

// Functions with default parameters
void    goForward         ( int percentL = 100 , int percentR = 100 );
void    goBackward        ( int percentL = 100 , int percentR = 100 );
int     rangeFinders      ( int returnWhich = 0 , int here = -99);



/*********************************************************************************************************************************************************************/
// Main program
/*********************************************************************************************************************************************************************/

 
 
void setup() { 
  
  // Define pinmodes
  #ifdef PANPIN              
            pinMode(PANPIN,OUTPUT);              
  #endif
  #ifdef TILTPIN             
            pinMode(TILTPIN,OUTPUT);             
  #endif
  #ifdef RANGEFINDERPIN      
            pinMode(RANGEFINDERPIN,INPUT);       
  #endif
  #ifdef IRLEDSPIN      
            pinMode(IRLEDSPIN,OUTPUT);       
  #endif
  #ifdef IRTOPPIN      
            pinMode(IRTOPPIN,INPUT);       
  #endif
  #ifdef IRBOTTOMPIN      
            pinMode(IRBOTTOMPIN,INPUT);       
  #endif
  #ifdef IRLEFTPIN      
            pinMode(IRLEFTPIN,INPUT);       
  #endif
  #ifdef IRRIGHTPIN      
            pinMode(IRRIGHTPIN,INPUT);       
  #endif
  
  
  
  stopMotors();  
   
  // 4U2DO ... Change writes to center *your* servos.
  // Delays are in place to allow servos time to move, and time for you to see settings.
  // Modify delay time to suit your own platform.
  
  // Important!
  // All the code in this program is written so that if you are sitting with the DFRobotShop Rover V2 in front of you,
  // rear closest to you, front furthest away from you, pan 180 is to the left, pan 0 is to the right.
  //
  // If you orient your servo the opposite way, all the "left/right" decision making will be reversed,
  // and you'll have to flip a lot of code.  Better to solve the problem here!
  
  panServo.attach(PANPIN);
  
          #ifdef TM1638INSTALLED
          sprintf(outputString, "     %03d", 0);   
          tm1638.setDisplayToString(outputString,0,0);  
          #endif  
  panServo.write(  0);            
  compoundIRSensor();  // Get baseline IR readings (should be nothing in sight).
  delay(2000);
  
          #ifdef TM1638INSTALLED
          sprintf(outputString, "   %02d   ", 90);    
          tm1638.setDisplayToString(outputString,0,0);  
          #endif  
  panServo.write(90);            delay(2000);
  
          #ifdef TM1638INSTALLED
          sprintf(outputString, "%03d     ", 180);  
          tm1638.setDisplayToString(outputString,0,0);  
          #endif  
  panServo.write(180);           delay(2000);
  
  panServo.detach(); 

  #ifdef TILTPIN
  tiltServo.attach(TILTPIN);
  tiltServo.write(  0);         delay(2000); 
  tiltServo.write( 90);         delay(1000);  
  tiltServo.detach(); 
  #endif

    
  // while(1);   // Activate this line if you want to put the Rover to sleep so you can fine tune your servos.  
 
  
  
  #ifdef XBeeINSTALLED
  // Communicate with monitoring station
  Serial.begin  (9600);
  Serial.print  (whoAmI);
  Serial.println(" activated. Launch sequence commencing.");  
  #endif
  
  
  
  #ifdef TM1638INSTALLED
  setDistance( );      
  // Countdown to launch.
  tm1638.setDisplayToString(whoAmI,0,0);
  tm1638LEDs = 256;
  for (int i=0; i<9; i++) {
    tm1638.setLEDs(tm1638LEDs-1); delay(500);
    tm1638LEDs /= 2;
  }   
  #endif
   
} 


 
 
void loop() { 
 
  followLeftWall();
 
  #ifdef TM1638INSTALLED
  // You can use one of your TM1638 buttons as a "reset".
  if ( tm1638.getButtons( ) == 128 ) resetFunc();
  #endif

 
}



/*********************************************************************************************************************************************************************/
// Functions <--------------------------------------------------------------------------------------------------------------------------------------------------------
/*********************************************************************************************************************************************************************/




void followLeftWall( void ) {
  
  // Incoming parameters : Nothing
  
  // Responsibilities    : Stay close to the left wall.
  //                       This function modifies motor speeds based on distance from wall,
  //                       but does not affect motor direction.
  
  // Returns             : Nothing.
  
    
  // If the front ultrasonic indicates there's a wall ahead, take corrective action.
  // 4U2DO ... If required, fine tune distance (7) for your platform.
  if ( rangeFinders(frontUS) <= 7 ) wallAhead();  
   
  // Use the pan mounted IR to determine what's the current situation with the left wall.
  rangeFinders(panIR, 180); 

  // Control motors to maintain correct distance from wall.
  if ( irValue >= IR_HOWCLOSE+4 ) {                                    
       // Way too close, VEER right
        
        goForward ( 100 , 60 );
      
        #ifdef TM1638INSTALLED
        tm1638.setLEDs( 15 );
        #endif
        
  }
  else if ( irValue > IR_HOWCLOSE ) {                                  
            // A little close, veer right
        
        goForward ( 100 , 75 );
        
        #ifdef TM1638INSTALLED
        tm1638.setLEDs( 12 ); 
        #endif
        
  }
  else if ( irValue >= IR_HOWFAR  && irValue <= IR_HOWCLOSE ) {        
            // Good, go straight
                    
        goForward ( 100 , 100 );
        
        #ifdef TM1638INSTALLED
        tm1638.setLEDs( 60 );
        #endif
        
  }
  else if ( irValue <= IR_HOWFAR-8 ) {                                 
            // Probably a corner?
        
        delay(400);
        goForward ( 0 , 100 );
        
        #ifdef TM1638INSTALLED
        tm1638.setLEDs( 255 );
        #endif
       
  }  
  else if ( irValue <= IR_HOWFAR-4 ) {                                 
            // Way too far, VEER left
        
        goForward ( 60 , 100 );
        
        #ifdef TM1638INSTALLED
        tm1638.setLEDs( 240 );
        #endif
       
  }
  else {                                                               
        // A little too far, veer left
        
        goForward ( 75 , 100 );
        
        #ifdef TM1638INSTALLED
        tm1638.setLEDs( 48 );
        #endif
        
  }
  
   
  
  return;
  
}



void wallAhead( void ) {
  
  // Incoming parameters : Nothing
  
  // Responsibilities    : Look ahead for obstacle (wall).
  //                       If obstacle detected, it will use the IR sensor to approach to a predetermined distance and execute a right turn.
  //                       It uses the IR sensor because the ultrasonic is less reliable at close range.
  //                       You came into this function with the Rover going forward,
  //                       so you leave this function with the Rover going forward.
  
  // Returns             : Nothing.
  
  
  // Let's stop before doing anything
  stopMotors();

  // Pivot the vehicle right so that the wall is on your left
  pivotRight();


  // Back to the job of hugging the wall
  goForward ( 80 , 80 ); 
  delay(500);
   
  return;      

}


void pivotRight( void ) {  
    
  // Incoming parameters : Nothing
  
  // Responsibilities    : Pivot right and align yourself to resume going forward.
  
  // Returns             : Nothing.

  // This function is the most challenging task in the program.
  
  // You're faced with a wall.  Solution... pivot right 90 degrees to continue.
  // Well, ideally 90 degrees, but it's highly unlikely that you're facing the wall directly head on.
  // Chances are you're facing at a slight angle.  So how do you pivot the right amount?
  
  // If you had wheel encoders, or better still an on-board compass, you could accurately measure the angle.
  // But this project is designed to work with just the specified components.  So the problem has to be solved
  // with just the sensors, which are not 100% reliable.
  
  // A proposed solution might be to "time" how long your vehicle takes to do a 90 degree pivot, and use that.
  // However, as your batteries get weaker, your speed will decay, increasing the time required to perform the
  // desired pivot.   I found a timed solution to not be accurate enough.
  
  // I experimented with a number of increasingly complicated algorithms whose code was rather convoluted.  
  // Not really suitable for a beginner project such as this. 
  
  // I could solve the problem with more sensors, but that would be cheating.  I wanted to solve it with the existing configuration.
  
  // The solution I present here is simple enough to understand, but accurate enough to work most of the time,
  // at least for me.     https://www.youtube.com/watch?v=Io_K32Qe_1Y  
  
  // Want to challenge your programming skills?  See if you can improve on this function :)
 

  int highestIR = 0 , panAngle;
  
  // 4U2DO ...  This little algorithm errs on the side of caution.
  //            You can play with it to speed things up if you want.
  //            Just remember, accidents increase with speed :)
  
  do {  
    
    // Pivot right, slower than normal speed.  You don't have to do everything full speed you know :) 
    // 4U2DO ... Change pivot speed to suit your platform 
    digitalWrite(MOTORLEFTDIREC,AHEAD);          analogWrite (MOTORLEFTSPEED, speedLeft *.65);   // Forward Left Motor  
    digitalWrite(MOTORRIGHTDIREC,BACK);          analogWrite (MOTORRIGHTSPEED,speedRight*.65);   // Reverse Right Motor 
    delay(750);  // 4U2DO ... Fine tune this delay for your platform
    stopMotors();

    // Move up a smidge. 
    goForward( 60 , 60 );
    delay(400);  // 4U2DO ... Fine tune this delay for your platform
    stopMotors();

    highestIR = 0;

    // Find the highest IR reading in sight
    // You can't rely on just a single reading at a single angle, 
    // so take multiple readings at different angles.
    // Change panAngle to take fewer readings.
    for ( panAngle = 90 ; panAngle <= 180 ; panAngle += 15 ) {
          
      rangeFinders(panIR,panAngle);
      highestIR = ( irValue > highestIR ) ? irValue : highestIR;
      
    }

   

  } while ( highestIR > IR_STOP_PIVOT_HERE );
  // If the highest IR reading you can obtain is higher than desired,
  // you must still be angled towards the wall.  Try again.
 
  //while(1);        // Activate this line if you want to fine tune your "right turn" logic.

}


void goForward( int percentL , int percentR ) {
  
  // Incoming parameters : int Percentage of Left Motor Speed, int Percentage of Left Motor Speed.
  
  // Responsibilities    : Go forward with motors set to provided percentages applied against baseline speed.
  //                       Default parameters of ( 100, 100 ) provided in prototype.
  //                       This function leaves both motors in "forward".
  
  // Returns             : Nothing.  
  
  // Failsafe
  if ( percentL < 0 )     percentL = 0;
  if ( percentR < 0 )     percentR = 0;
  if ( percentL > 100 )   percentL = 100;
  if ( percentR > 100 )   percentR = 100;
  
  if ( percentL == percentR )  whatAreYouDoing = goingForward;  
  else                         whatAreYouDoing = ( percentL < percentR ) ? veeringLeftForward : veeringRightForward;
  
  if ( percentL == 0 && percentR != 0 ) whatAreYouDoing = turningLeft;
  if ( percentR == 0 && percentL != 0 ) whatAreYouDoing = turningRight;

  digitalWrite(MOTORLEFTDIREC,AHEAD);        // Forward Left Motor 
  analogWrite (MOTORLEFTSPEED, ( speedLeft * percentL ) / 100);

  digitalWrite(MOTORRIGHTDIREC,AHEAD);        // Forward Right Motor
  analogWrite (MOTORRIGHTSPEED, ( speedRight * percentR ) / 100);  
  
}


void goBackward( int percentL , int percentR ) {
  
  // Incoming parameters : int Percentage of Left Motor Speed, int Percentage of Left Motor Speed.
  
  // Responsibilities    : Go backward with motors set to provided percentages applied against baseline speed.
  //                       Default parameters of ( 100, 100 ) provided in prototype.
  //                       This function leaves both motors in "forward".
  
  // Returns             : Nothing.  
  
  // Failsafe
  if ( percentL < 0 )     percentL = 0;
  if ( percentR < 0 )     percentR = 0;
  if ( percentL > 100 )   percentL = 100;
  if ( percentR > 100 )   percentR = 100;
  
  if ( percentL == percentR )  whatAreYouDoing = goingBackward;  
  else                         whatAreYouDoing = ( percentL < percentR ) ? veeringLeftBackward : veeringRightBackward;
  
  digitalWrite(MOTORLEFTDIREC,BACK);        // Backward Left Motor 
  analogWrite (MOTORLEFTSPEED,( speedLeft * percentL ) / 100);   
  
  digitalWrite(MOTORRIGHTDIREC,BACK);        // Backward Right Motor
  analogWrite (MOTORRIGHTSPEED,( speedRight * percentR ) / 100); 
  
}



void stopMotors( void ) {
    
  // Incoming parameters : Nothing.
  
  // Responsibilities    : Full stop & delay.   Reads sensors for current location.
  //                       This function does not modify motor direction.
  
  // Returns             : Nothing. 
  
  whatAreYouDoing = stopped;

  analogWrite (MOTORLEFTSPEED,0);          // Left motor 
  analogWrite (MOTORRIGHTSPEED,0);          // Right motor 
  
  rangeFinders();

}



int rangeFinders( int returnWhich , int here ) {
  
  // Incoming parameters : int 0 - Compound IR sensor (on pan servo) , 1 - Front ultrasonic
  
  // Responsibilities    : Locate and measure distance (in cm) to object (wall).
  //                       Resolve possible false readings.
  //                       Display readings on TM1638 if installed.
  
  // Returns             : rangeInCentimeters for ultrasonic sensor, or IR value.
  
  static int currentServoAngle = 999;
  int  frontRangeInCentimetersPrevious = frontRangeInCentimeters; 
  
  // Does the servo need to move?
  if ( here != -99 && currentServoAngle != here ) {
    
      panServo.attach(PANPIN);    panServo.write(here);    delay(500);   panServo.detach();
      currentServoAngle = here; 
    
  }
  
  
  ultrasonicFront.DistanceMeasure();                                              // get the current signal time
  frontRangeInCentimeters =   ultrasonicFront.microsecondsToCentimeters();        // convert the time to centimeters
   
  // Throw out bad sensor readings... the ultrasonics are glitchy at close range.
  if ( frontRangeInCentimeters == 0 ) frontRangeInCentimeters =   frontRangeInCentimetersPrevious;
  
  // We're not interested in anything past 99
  if ( frontRangeInCentimeters   > 99 )    frontRangeInCentimeters = 99;
  
  irValue = compoundIRSensor();

  
  #ifdef XBeeINSTALLED
  // Display current readings
  sprintf(outputString, "%03d   %02d",  irValue, frontRangeInCentimeters );
  Serial.println(outputString);  
  #endif
    
  #ifdef TM1638INSTALLED
  // Display current readings
  sprintf(outputString, "%03d   %02d",  irValue, frontRangeInCentimeters );
  tm1638.setDisplayToString(outputString,0,0);  
  #endif
  

  
  switch ( returnWhich ) {
    
    case 0 :   return irValue; 
    case 1 :   return frontRangeInCentimeters;

     
    default :  return irValue;
  }

}


int compoundIRSensor( void ) {

  // Incoming parameters :  Nothing.
 
  // Responsibilities    :  Uses IR Sensor to detect target.
  
  // Returns             :  true if target sighted, false otherwise.
  
  
  // This compound IR sensor works by emitting IR light onto an object
  // and then detecting the reflected IR. The IR LEDs are controlled by a
  // digital output (IRLEDSPIN) so that ambient light as well as reflected light can be
  // measured. 
  
  // The level of reflected light read by each of the 4 phototransistors allows the compound sensor to detect 
  // where the object is.  For example, if the left phototransister reads 100 and the right one reads 200,
  // that must mean there is something on the right reflecting more IR light back.
  
  // The Dagu has Top, Bottom, Left, and Right phototransistors.  This program only uses the left and right.
  

  static boolean getBaseline = true;
    
  digitalWrite(IRLEDSPIN,HIGH);                                // Turn on IR LEDs to read Total light (ambient + reflected IR)
  delayMicroseconds(500);                                      // Allow time for phototransistors to respond. (may not be needed)

  irLeft   = analogRead(IRLEFTPIN);                            // Total  = Ambient + LED IR reflected from object
  irRight  = analogRead(IRRIGHTPIN);
  
  // Top/Bottom not needed by this program.  Code included for experimentation if you wish.
  //irTop    = analogRead(IRTOPPIN);                              
  //irBottom = analogRead(IRBOTTOMPIN);  

  digitalWrite(IRLEDSPIN,LOW);                                 // Turn off IR LEDs to read Ambient light (ie from indoor lighting and natural light)
  delayMicroseconds(500);                                      // Allow time for phototransistors to respond. (may not be needed)
   
  irLeft      =    irLeft   -   analogRead(IRLEFTPIN);         // Reflected IR = Total (read previously) - Ambient (read now)
  irRight     =    irRight  -   analogRead(IRRIGHTPIN);            
  
  //irTop       =    irTop    -   analogRead(IRTOPPIN);        // Top/bottom not used by this program.               
  //irBottom    =    irBottom -   analogRead(IRBOTTOMPIN);
  //irValue = ( irLeft + irRight + irTop + irBottom ) / 4;  

  irValue = ( irLeft + irRight ) / 2;                          // Distance of object is average of reflected IR
                                                               // Low values ( ie 34,35,36 denote far away , high values ie 634,635,636 denote close )
                                                               
  irValue /= 10;                                               // Divide by 10 to turn values like 110,115,118 all into 11.
                                                               // For this program, we don't need the precision of a three digit number.
                                                               // Of course, I could have combined this statement with the previous, but I've kept them
                                                               // seperate so you can fully understand the math involved.
                                                               // This would do both... irValue = ( irLeft + irRight ) / 20;
                                                              
         
  // Establishing baseline light level not needed by this program.
  // Code provided for your experimentation if you wish.
  /*
  if ( getBaseline ) {  // Get the baseline level denoting "nothing in sight"
                        // This code is only executed once.
      getBaseline        = false;        
      irValue_baseline   = irValue; 
      irLeft_baseline    = irLeft;
      irRight_baseline   = irRight;
      //irTop_baseline     = irTop;
      //irBottom_baseline  = irBottom;
  } 
  */  
                                                               
   
  return  ( irValue );                        
 
}



boolean timeIsUp( unsigned long& then , unsigned int desiredInterval) {
  
  // Incoming parameters : unsigned long (by ref) counter (in milliseconds) , unsigned int (by copy) desired cycle.
 
  // Responsibilities    : When the counter hits the cycle threshold (based on runtime), it will set the counter to current runtime, 
  //                       and inform whoever called this that "time is up".
  
  // Returns             : true if cycle threshold achieved, false otherwise
  
  unsigned long now = millis();
  boolean desiredIntervalAchieved = false;
  
  if (now - then >= desiredInterval)  {
    then = now;
    desiredIntervalAchieved = true;
  }

  return desiredIntervalAchieved;

}





#ifdef TM1638INSTALLED

void setDistance ( void ) {
  
  // Incoming parameters : Nothing.
 
  // Responsibilities    : Uses TM1638 to display when the Rover has been placed in the correct start position.
  //                       Flashing LED indicates to click on "Start button" to go.
  
  // Returns             : Nothing.
  
 // 4U2DO ...   More than anything, the predetermined IR settings in the code will affect the performance of your vehicle.
 //             Light reflectivity will vary from surface to surface, so even when you get this adjusted to your needs, 
 //             the vehicle will work completely differently when you test it against a different surface.
 //             It would be a simple task to modify this function to allow you to use the buttons on the TM1638 to record 
 //             the readings according to wherever you placed the vehicle, and base reflective readings of the surface it is facing.

  do {
    
     rangeFinders(panIR, 170);
     
     if ( irValue >= IR_HOWFAR && irValue <= IR_HOWCLOSE ) {
    
         // Blink the first LED to let the pilot know everything is ready.
         // Click the first button to start.
         tm1638.setLEDs(60); delay(250);
         tm1638.setLEDs(61); delay(250);
         
         tm1638Buttons = tm1638.getButtons( ); 
         
     }
     
     else {

       // Provide a visual reference to determine how close or far you should be.
       tm1638.setLEDs(( irValue < IR_HOWCLOSE ) ? 240 : 15);  delay(250);
       tm1638.setLEDs(0);                                     delay(250);
       
     }
       
  } while ( tm1638Buttons != 1 );
  
  delay(2000);

}

#endif


