// Source          ::     RoverV2_SeekDestroy_V4

// Description     ::     Using the servo mounted Ultrasonic & Dagu Compound Infrared Sensors, 
//                        the Rover scans for threats within its exclusion zone. 
//
//                        Once a threat is detected, it approaches for the kill with its "laser cannon".
//
//                        The ultrasonic is used to detect targets at long range,
//                        while the infrared is used to control the final approach.
//
//      
//                        This version communicates readings back to an optional monitoring station via XBee or USB Cable


//                        Recommended components ( or equivalent ) 

//                        1)  http://www.robotshop.com/en/dfrobotshop-rover-v2---arduino-compatible-tracked-robot-autonomous-dev-kit.html
//
//                        OR
//
//                        1)  Any Arduino powered vehicle platform
//                        2)  Dagu Mini Pan and Tilt Kit
//                        3)  SeeedStudio 3 Pin Ultrasonic Range Finder
//                        4)  Dagu Compound Infrared Sensor
//                        5)  TM1638 Display Module
//                        6)  Red Dot Laser Module
//                        7)  DFRobot 7.4V Lipo 2200mAh Battery

// Original Program   ::     Joe M  / arduino (at) bellaliant.net / December 2015

// Modified By        ::     Your Name



//  This program is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License 
//  as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  Lesser General Public License for more details.
//
//  For a copy of the GNU Lesser General Public License, 
//  write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA



// *********************************************************************************************************************************************************************
// Optional components and general definitions
// *********************************************************************************************************************************************************************


String  whoAmI = "Cerebus";  // Name of your Rover, mine is Cerebus


// How fast you want the DFRobotShop Rover V2 to go... the stronger your battery pack, the lower your number should be.
// 80 seems to be a good setting for http://www.robotshop.com/ca/en/dfrobot-7-4v-lipo-2200mah-battery.html 
#define BASELINESPEED     80

// React to anything that is detected wthin this zone.
#define EXCLUSION_ZONE    50  // Number in centimeters.  Change as desired.


// Associated with the Dagu Compound Infrared Sensor, this setting determines the reading
// at which to fire the laser.
// The bigger the number, the closer the target will be when you fire.
#define ENGAGE_TARGET     350

// Associated with the Dagu Compound Infrared Sensor, this setting determines the maximum reading
// before putting the Rover into "retreat" mode because it is too close to the target.
// The bigger the number, the closer the target will get to the Sentinel before it decides to retreat.
#define DANGER_CLOSE      450

// Used by the motor controllers
#define AHEAD             LOW
#define BACK              HIGH


#define    TM1638INSTALLED                // Only activate this line if you have a TM1638 LED/Key module installed https://code.google.com/p/tm1638-library/
                                          // The TM1638 is an excellent addition to the Rover as it provides a nice onboard I/O package.
                                          
//#define    XBeeINSTALLED                  // Communicate with a monitoring station.   
                                          // Only activate this line if you are using XBee, OR ...
                                          // ... if you have your Rover connected to the PC with a USB cable.
                                          
                                          





// **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T ****
// ***********************************************************************************************************************************************************************
// Pinouts... IMPORTANT!!!! ... set the pinouts to your hardware configuration
//
// NEVER USE Digital pins 5,6,7,8 ... they are permanently connected to the onboard motor controller on all versions of the DFRobotShop Rover V2.
//
// Digital pins 0 and 1 can only be used if you ARE NOT using Serial communications ( ie Serial.begin() ).
// Also note that having components connected to D0 and D1 will interfere with uploading your Sketch.  
//
// If you plan on someday expanding your Rover to use a second Arduino board ( to add more components, sensors, etc ), do not use (analog pins) A4 (SDA) and A5 (SCL)
// http://www.instructables.com/id/I2C-between-Arduinos/?ALLSTEPS 
// 
// ***********************************************************************************************************************************************************************

// ********** Digital Pins D0 thru D13 **********

// Using 0 and 1 disables upload!     

#define DO_NOT_USE_0        0
#define DO_NOT_USE_1        1

#define PANPIN              2        // Which pin the pan servo is connected to.  For example, here it is attached to (digital pin) D2.
                                     // Assuming you are using the I/O Expansion Shield V7 ( http://www.dfrobot.com/wiki/index.php/IO_Expansion_Shield_for_Arduino_V7_SKU:DFR0265 ) ,
                                     // configure your three sensor leads so that ORANGE is Signal and BROWN is ground.  The power RED is in the middle.

//#define TILTPIN             3        // Only activate this line if you want to use the tilt servo.
                                     // The tilt servo is not needed for the original seek/destroy functionality of this program,
                                     // so you might want to free up pin 3 for other purpses.

          
#define RANGEFINDERPIN      4        // The ultrasonic rangefinder on the pan servo.

#define MOTORRIGHTSPEED     5        // Nothing connected to these pins , but the DFRobotShop Rover V2 onboard motor controller uses these circuits.
#define MOTORLEFTSPEED      6        // If you're using a different vehicle platform, configure your motor controllers accordingly.
#define MOTORRIGHTDIREC     7
#define MOTORLEFTDIREC      8

#define LASERPIN            9       // Only activate this line if you've installed a "laser cannon".
                                    // You can buy a "laser module" designed for Arduino, or make one using a laser pointer...
                                    // Remove batteries, tape switch "on"
                                    // Connect the "spring" inside the body to ground, and connect the "body" to pin 9
                                    // If this doesn't work, change connections.  Pretty easy.

// Pins D10, D11, D12 available UNLESS TM1638 installed...
#ifdef TM1638INSTALLED    // "ifdef's" are one of those obscure C++ features many beginning programmers don't know about.
                          // It is a neat feature that allows "conditional compilation".  In the case of this program,
                          // the various code segments using the TM1638 LED/Key module will only be included in the compiled executable
                          // if you defined TM1838INSTALLED above ( ie #define TM1638INSTALLED ).
                          // For more on "conditional compilation", go to http://www.cplusplus.com/doc/tutorial/preprocessor/

  #define TM1638STBPIN     10        // TM1638 Strobe D10.
  #define TM1638DTOPIN     11        // TM1638 Data D11.
  #define TM1638CLKPIN     12        // TM1638 Clock D12.
  
#endif
  
  
#define IRLEDSPIN          13        // IR Compound Eye LEDs D13.




// ********** Analog Pins A0 thru A5 **********

// FYI... using analog pins for digital I/O is just the same as using digital ones.

// A0 is referred to as Pin 14     A1 is referred to as Pin 15
// A2 is referred to as Pin 16     A3 is referred to as Pin 17
// A4 is referred to as Pin 18     A5 is referred to as Pin 19

// A2 thru A5 (plus digital pin D13) are used in this program for the Dagu Compound Infrared Sensor.
// To help you with the connections, I provide this "color-coded" explanation.   
// In this explanation, I am using a ribbon cable consisting of the following colors...
// Yellow , Green , Blue , Violet , Gray , White , Black

// Holding the sensor array with the back facing you and the pins facing up, the above color sequence should apply.
// Yellow being VCC and Black being GND with Green , Blue , Violet , Gray , White between.

// When mounted on the pan/tilt mechanism with the pins facing up, this is the color sequence you should see when looking
// at the unit from the *front*...

// Black (Ground) , White (IR LEDs) , Grey (Right) , Violet (Bottom) , Blue (Left) , Green (Top) , Yellow (Power).

// On the I/O Expansion Shield V7 , the following connections apply...

// A2 (Signal) - Grey (Right), A2 (Power) - Yellow (Power) , A2 (Ground) - Black (Ground)
// A3 (Signal) - Blue (Left)
// A4 (Signal) - Green (Top) 
// A5 (Signal) - Violet (Bottom) 
// D13 - White

// The DFRobotShop Rover V2 uses these circuits for the wheel encoders.  
// Not used by this program, you can redeploy them for other uses.
//#define RIGHTENCODERPIN         A0          // Right motor encoder  A0.
//#define LEFTENCODERPIN          A1          // Left motor encoder   A1.


#define IRRIGHTPIN              A2          // (Grey)   IR Compound Eye Right (analog pin)    A2.
#define IRLEFTPIN               A3          // (Blue)   IR Compound Eye Left (analog pin)     A3.

// The top/bottom IRs are not needed for the original seek/destroy functionality of this program,
// so you might want to free up pins A4 and A5 for other purpses.
//#define IRTOPPIN                A4          // (Green)  IR Compound Eye Top (analog pin)      A4.
//#define IRBOTTOMPIN             A5          // (Violet) IR Compound Eye Bottom (analog pin)   A5.


// A4 == SDA, A5 == SCL if second Arduino board attached



                     
                  
                             


// *********************************************************************************************************************************************************************
// Required includes
// *********************************************************************************************************************************************************************

#include <stdio.h>
#include <Servo.h> 
#include <Ultrasonic.h>


#ifdef TM1638INSTALLED 
// *********************************************************************************************************************************************************************
//  TM1638 Board
// *********************************************************************************************************************************************************************
//
// https://code.google.com/p/tm1638-library/
//
// Based on a TM1638 chip, the board uses a serial communications interface to enable three-wire control of the LEDs and reading of the pushbuttons.
//

#include <TM1638.h>

// *********************************************************************************************************************************************************************
//  End of TM1638 definitions
// *********************************************************************************************************************************************************************
#endif



// *********************************************************************************************************************************************************************
//  Global Variables.
//  Professional programmers cringe at the thought of globals, but they seem to be generally acceptable in the "wild west" Arduino world.
//  Prudent use of them simplifies program design since variable passing via parameters between functions are reduced.
// *********************************************************************************************************************************************************************

// An homage to every program I've written since my WATFIV days.
// Every program needs an "int i", right?
int i;

byte speedRight = BASELINESPEED, speedLeft = BASELINESPEED;

// Create two servo objects
Servo panServo;

#ifdef TILTPIN
Servo tiltServo;  
#endif

// Variables associated with panServo
int panAngle = 90, panAngle_previous = -99, increment = 3; 


// Targeting variables
boolean targetSightedByCompoundIR = false, targetSightedByUS = false;

#ifdef LASERPIN
boolean laserCharged = false; 
#endif

// Ultrasonic sensor object 
Ultrasonic ultrasonicPan(RANGEFINDERPIN);

// This will contain the reading from the most recent call to ultrasonicSensor().
int panRangeInCentimeters;


// These will contain the readings from the most recent call to compoundIRSensor().
int irValue, irLeft, irRight, irTop, irBottom; 
int irValue_baseline, irLeft_baseline, irRight_baseline, irTop_baseline, irBottom_baseline;


// Used throughout the program when you want to time events
unsigned long targetSightedByCompoundIRAtTime=millis(), targetSightedByUSAtTime=millis();



// Enumerated types for objects make code easier to read
enum panStatus { 
  isDetached, isAttached
} pan;

enum roverAction {
  doNothing, relocate, pivotLeft, pivotRight, approachTarget , retreatTarget , shoot
} previous_roverShould = doNothing , roverShould = doNothing;


#ifdef TM1638INSTALLED 
// Used for TM1638
// define module pins... data DT0 11, clock CLK 12, strobe STB 10
TM1638 tm1638(TM1638DTOPIN, TM1638CLKPIN, TM1638STBPIN);
char outputString[128];
String  tm1638Msg; 
int tm1638LEDs, tm1638Buttons;
#endif




 
// **********************************************************************************************************************************************************************
// Function Prototypes                                                                                                                                                
// **********************************************************************************************************************************************************************

void       servoAction           ( void );
void       roverAction           ( void );
void       relocateRover         ( void );
boolean    compoundIRSensor      ( void );
boolean    ultrasonicSensor      ( void );
boolean    timeIsUp              ( unsigned long& then , unsigned int desiredInterval );

#ifdef TM1638INSTALLED
void     (* resetFunc)     ( void ) = 0;  // Declare reset function @ address 0.
                                          // Calling this function is the same as pressing reset on the Arduino board.
#endif

// Functions with default parameters
// http://en.cppreference.com/w/cpp/language/default_arguments

void       changePanServoAngle   ( int angle = 90 , int delayms = 50 , boolean detachAfter = true );

#ifdef LASERPIN
boolean    fireLaser             ( int times = 1 );
#endif



// *********************************************************************************************************************************************************************
// Main program                                                                                                                                                                                                                   
// *********************************************************************************************************************************************************************
 
void setup() { 
  
  // Define active pinmodes
  #ifdef PANPIN              
            pinMode(PANPIN,OUTPUT);              
  #endif
  #ifdef TILTPIN             
            pinMode(TILTPIN,OUTPUT);             
  #endif
  #ifdef RANGEFINDERPIN      
            pinMode(RANGEFINDERPIN,INPUT);       
  #endif
  #ifdef LASERPIN      
            pinMode(LASERPIN,OUTPUT);       
  #endif
  #ifdef IRLEDSPIN      
            pinMode(IRLEDSPIN,OUTPUT);       
  #endif
  #ifdef IRLEFTPIN      
            pinMode(IRLEFTPIN,INPUT);       
  #endif
  #ifdef IRRIGHTPIN      
            pinMode(IRRIGHTPIN,INPUT);       
  #endif
  
    
  // Change writes to center *your* servos.
  // Delays are in place to allow ssrvos time to move, and time for you to see settings.
  // Modify delay time to suit your own platform.
  
  // Important!
  // All the code in this program is written so that if you are sitting with the DFRobotShop Rover V2 in front of you,
  // rear closest to you, front furthest away from you, pan 180 is to the left, pan 0 is to the right.
  //
  // If you orient your servo the opposite way, all the "left/right" decision making will be reversed,
  // and you'll have to flip a lot of code.  Better to solve the problem here!
  
  panServo.attach(PANPIN);
          #ifdef TM1638INSTALLED
          sprintf(outputString, "     %03d", 0);   
          tm1638.setDisplayToString(outputString,0,0);  
          #endif  
  panServo.write(  0);           delay(2000); 
          #ifdef TM1638INSTALLED
          sprintf(outputString, "%03d     ", 180);    
          tm1638.setDisplayToString(outputString,0,0);  
          #endif  
  panServo.write(180);           delay(2000);
          #ifdef TM1638INSTALLED
          sprintf(outputString, "   %02d   ", 90);  
          tm1638.setDisplayToString(outputString,0,0);  
          #endif  
  panServo.write( 90);           delay(2000);  
  panServo.detach();    
  pan = isDetached;  
  
    
  #ifdef TILTPIN
  tiltServo.attach(TILTPIN);
  tiltServo.write(  0);         delay(2000); 
  tiltServo.write( 90);         delay(1000);  
  tiltServo.detach(); 
  #endif  

  // while(1);   // Activate this line if you want to put the Rover to sleep so you can fine tune your servos.
  
  compoundIRSensor();  // Get baseline IR readings with no target sighted.
  
  #ifdef XBeeINSTALLED
  // Communicate with monitoring station
  Serial.begin  (9600);
  Serial.print  (whoAmI);
  Serial.println(" activated. Launch sequence commencing.");  
  Serial.println("Testing laser.");  
  #endif
  
  #ifdef LASERPIN
          #ifdef TM1638INSTALLED  
          tm1638.setDisplayToString(" LASER",0,0);
          delay(2000);  
          #endif
  fireLaser ( 4 );
  laserCharged = true;
  #endif  
  
  #ifdef TM1638INSTALLED
  // Countdown to launch.
  tm1638.setDisplayToString(whoAmI,0,0);
  tm1638LEDs = 256;
  for (i=0; i<9; i++) {
    tm1638.setLEDs(tm1638LEDs-1); delay(500);
    tm1638LEDs /= 2;
  }   
  #endif
  
  
  #ifdef XBeeINSTALLED
  // Communicate with monitoring station
  Serial.print  ("Warning! Sentinel is operational.  Exclusion zone is ");  
  Serial.print  (EXCLUSION_ZONE); 
  Serial.println("."); 
  #endif
  
} 

 
 
void loop() {
  
  servoAction();
  roverAction(); 
  
  #ifdef TM1638INSTALLED
  // You can use one of your TM1638 buttons as a "reset".
  if ( tm1638.getButtons( ) == 128 ) resetFunc();
  #endif
    
}



// *********************************************************************************************************************************************************************
// Functions <------------------------------------------------------------------------------------------------------------------------------------------------------- 
// *********************************************************************************************************************************************************************



void servoAction ( void ) {

  // Incoming parameters :  Nothing
 
  // Responsibilities    :  Control panServo, attempt to locate target
  
  // Returns             :  Nothing
  
   
  if ( targetSightedByCompoundIR = compoundIRSensor() ) {  // Look for target with IR, take action based on result
    
    // Target detected at short range
    #ifdef XBeeINSTALLED
    Serial.print  ("Imminent target at angle ");
    Serial.print  (panAngle);
    Serial.print  (".  IR reading ");
    Serial.print  (irValue);
    Serial.println(".");
    #endif

    // Align servo head for best view and to match upcoming pivot action
    if ( ( irRight - irLeft ) > 8 ) { panAngle += 2; } // Pan more right... you can change the 8 and 2 if you like
    else 
    if ( ( irLeft - irRight ) > 8 ) { panAngle -= 2; } // Pan more left
    
    if ( panAngle_previous != panAngle ) changePanServoAngle ( panAngle );

  }
  else if ( targetSightedByUS = ultrasonicSensor() ) {  // Look for target with US, take action based on result
  
    // Possible target detected at long range
    #ifdef XBeeINSTALLED
    Serial.print  ("Possible target at angle ");
    Serial.print  (panAngle);
    Serial.print  (".  Range reading ");
    Serial.print  (panRangeInCentimeters);
    Serial.println(".");
    #endif
    
    // Align servo head to match upcoming pivot action
    if ( panAngle < 80 ) { panAngle += 3; } // Pan more left 
    else
    if ( panAngle > 100 ) { panAngle -= 3; } // Pan more right
    
    if ( panAngle_previous != panAngle ) changePanServoAngle ( panAngle );
  
  }
  else {
    
    // Apparently, nothing in range.   Scan for target.
    if ( panAngle > 175 ) increment = -3; 
    else if ( panAngle <   5 ) increment = 3;  
    panAngle += increment; 
    
    changePanServoAngle( panAngle, 8 , false );   // Small angle change requires shorter delay ( parameter 2 == 8 ms ), pan goes quicker if you don't detach ( parameter 3 )
    
  }
 
  panAngle_previous = panAngle;
  
  
  return;
  
  
}

void roverAction ( void ) {
  
  // Incoming parameters :  Nothing
 
  // Responsibilities    :  Based on current target info, what should the Rover do?
  //                        I would normally place most of the code found in this function into their own functions,
  //                        but I know that beginning programmers sometimes get lost in a plethora of functions calling functions,
  //                        so I've left the logic as is.  You more advanced programmers will know how to improve this function.
  
  // Returns             :  Nothing
  
  // What was the Rover doing previously?
  previous_roverShould = roverShould;
  

  if ( targetSightedByCompoundIR ) {     // Target sighted by short range infrared... decide what to do...
    
          targetSightedByCompoundIRAtTime = millis();                       // Note time        
                                                                
          #ifdef TM1638INSTALLED
          // Display current readings
          sprintf(outputString, "%03d  %03d", irLeft, irRight);   
          tm1638.setDisplayToString(outputString,0,0);  
          #endif
    
          if ( irValue >= DANGER_CLOSE ) {
    
            // Too close
            roverShould = retreatTarget;
    
          }  
          else if ( irValue >= ENGAGE_TARGET ) {
    
            #ifdef LASERPIN
            // Take a shot if laser charged
            roverShould = ( laserCharged ) ? shoot : doNothing; 
            #else
            roverShould = doNothing;
            #endif
           
    
          }
          else if ( panAngle < 80 || panAngle > 100 ) {
    
            // Align Rover to face target
            roverShould = ( panAngle > 100 ) ? pivotLeft : pivotRight;    
      
          }
          else if ( panAngle >= 80 && panAngle <= 100 ) {
    
            // Drive forward
            roverShould = approachTarget;
    
          }
  
  }
  else  if ( targetSightedByUS ) { // Target sighted by long range ultrasonic... decide what to do...
  
             targetSightedByUSAtTime=millis();    // Note time
             
             #ifdef TM1638INSTALLED
             // Display current reading
             sprintf(outputString, "   %02d   ", panRangeInCentimeters);   
             tm1638.setDisplayToString(outputString,0,0);  
             #endif
             
             if ( panAngle < 80 || panAngle > 100  ) {
    
                  // Align Rover to face target
                  roverShould = ( panAngle > 100 ) ? pivotLeft : pivotRight;    
      
             }
             else if ( panAngle >= 80 && panAngle <= 100 ) {
    
                  // Drive forward
                  roverShould = approachTarget;
    
             }
    
  }
  else {     // Target not sighted ... decide what to do... 

       #ifdef TM1638INSTALLED
       tm1638.setDisplayToString("        ",0,0);  
       #endif  
    
      // Has it been too long without sighting anything?
      // I chose 30 seconds...
      roverShould = ( timeIsUp ( targetSightedByCompoundIRAtTime , 30000 ) ) ? relocate : doNothing;          
        
  }


  if ( roverShould != previous_roverShould )    // Compare what the Rover "was doing" to what the Rover "should be doing now".
                                                // Has a different action been decided?
  switch ( roverShould ) {                      // Invoke chosen action
   
    case relocate:
    
        #ifdef XBeeINSTALLED
        Serial.println("Relocating. ");
        #endif
        
        relocateRover();
        targetSightedByUSAtTime = targetSightedByCompoundIRAtTime = millis();   // Reset target acquisition clocks
        break; 
    
    case pivotLeft:  
        
        #ifdef XBeeINSTALLED
        Serial.println("Turning left to possible target. ");
        #endif
        
        digitalWrite(MOTORLEFTDIREC,BACK);          analogWrite (MOTORLEFTSPEED,speedLeft);      // Reverse Left Motor  
        digitalWrite(MOTORRIGHTDIREC,AHEAD);        analogWrite (MOTORRIGHTSPEED,speedRight);    // Forward Right Motor
        break;   
  
     case pivotRight:
   
        #ifdef XBeeINSTALLED
        Serial.println("Turning right to possible target. ");
        #endif
        
        digitalWrite(MOTORLEFTDIREC,AHEAD);         analogWrite (MOTORLEFTSPEED,speedLeft);      // Forward Left Motor  
        digitalWrite(MOTORRIGHTDIREC,BACK);         analogWrite (MOTORRIGHTSPEED,speedRight);    // Reverse Right Motor
        break; 
   
     case approachTarget:
   
        #ifdef XBeeINSTALLED
        Serial.println("Approaching target. ");
        #endif
        
        digitalWrite(MOTORLEFTDIREC,AHEAD);         analogWrite (MOTORLEFTSPEED,speedLeft);      // Forward Left Motor  
        digitalWrite(MOTORRIGHTDIREC,AHEAD);        analogWrite (MOTORRIGHTSPEED,speedRight);    // Forward Right Motor
        
        #ifdef LASERPIN        
          #ifdef XBeeINSTALLED
          // If laser was discharged, it's now recharged
          if ( laserCharged == false ) Serial.println("Laser hot. ");
          #endif
        
          // Laser hot
          laserCharged = true;    // Activate this line if you want the laser cannon operational
        
  
          #ifdef TM1638INSTALLED
          tm1638.setLEDs( 129 );  // Indicate laser charged
          #endif
        #endif
        
        break;
        
    case retreatTarget:
   
        #ifdef XBeeINSTALLED
        Serial.println("Proximity alert! Retreating. ");
        #endif
        
        digitalWrite(MOTORLEFTDIREC,BACK);         analogWrite (MOTORLEFTSPEED,speedLeft);      // Reverse Left Motor  
        digitalWrite(MOTORRIGHTDIREC,BACK);        analogWrite (MOTORRIGHTSPEED,speedRight);    // Reverse Right Motor
        break;
        
     case shoot:
   
        #ifdef XBeeINSTALLED
        Serial.println("Target engaged! ");
        #endif
        
        analogWrite (MOTORLEFTSPEED,0);           // Stop Left motor 
        analogWrite (MOTORRIGHTSPEED,0);          // Stop Right motor
        
        #ifdef LASERPIN
        laserCharged = fireLaser ( );
        #endif
        
        break;
       
     case doNothing:
     default:
        
        analogWrite (MOTORLEFTSPEED,0);           // Stop Left motor 
        analogWrite (MOTORRIGHTSPEED,0);          // Stop Right motor 
        break;    
    
  }
 
  
}

void changePanServoAngle ( int angle , int delayMs , boolean detachAfter ) {
  
  // Incoming parameters :  int Desired Angle, int Required Delay in ms , boolean "Should you detach?" ( false - no , true - yes )
 
  // Responsibilities    :  Attach servo (if required), swing to provided angle , detach servo (if desired).
  
  // Returns             :  Nothing.
  
   if ( pan == isDetached ) { 
      panServo.attach(PANPIN); 
      pan = isAttached;
    } 
   
   panServo.write(angle);  
   delay(delayMs);   
   
   if ( detachAfter ) {
     panServo.detach(); 
     pan = isDetached;
   }
   
   return;
   
}

boolean ultrasonicSensor( void ) {
  
  // Incoming parameters :   Nothing.
  
  // Responsibilities    :   Uses Ultrasonic sensor to detect target.
  
  // Returns             :   true if target sighted, false otherwise.
  
  ultrasonicPan.DistanceMeasure();                                               // Get the current signal time
  panRangeInCentimeters =   ultrasonicPan.microsecondsToCentimeters();           // Convert the time to centimeters
  
  // Possible sighting... but because the ultrasonic is so unreliable, take a second look
  if ( panRangeInCentimeters >= 20 && panRangeInCentimeters <= EXCLUSION_ZONE ) {
      
    // Ensure servo chatter is not affecting reading 
    if ( pan == isAttached ) { 
        panServo.detach(); 
        pan = isDetached;
    } 
    
    ultrasonicPan.DistanceMeasure();                                              // Get the current signal time
    panRangeInCentimeters += ultrasonicPan.microsecondsToCentimeters();           // Convert the time to centimeters
        
    panRangeInCentimeters /= 2;   // Average the 2 readings
    
  }
  
  // We're not interested in anything below 20 (probable false reading... should have already been detected by the IR) or above 99
  if ( panRangeInCentimeters < 20 || panRangeInCentimeters > 99 )      panRangeInCentimeters  = 99;

  return ( panRangeInCentimeters <= EXCLUSION_ZONE ) ? true : false;

}


boolean compoundIRSensor( void ) {

  // Incoming parameters :  Nothing.
 
  // Responsibilities    :  Uses IR Sensor to detect target.
  
  // Returns             :  true if target sighted, false otherwise.
  
  
  // This compound IR sensor works by emitting IR light onto an object
  // and then detecting the reflected IR. The IR LEDs are controlled by a
  // digital output (IRLEDSPIN) so that ambient light as well as reflected light can be
  // measured. 
  
  // The level of reflected light read by each of the 4 phototransistors allows the compound sensor to detect 
  // where the object is.  For example, if the left phototransister reads 100 and the right one reads 200,
  // that must mean there is something on the right reflecting more IR light back.
  
  // The Dagu has Top, Bottom, Left, and Right phototransistors.  This program only uses the left and right.
 
  static boolean getBaseline = true;
    
  digitalWrite(IRLEDSPIN,HIGH);                                // Turn on IR LEDs to read Total light (ambient + reflected IR)
  delayMicroseconds(500);                                      // Allow time for phototransistors to respond. (may not be needed)

  irLeft   = analogRead(IRLEFTPIN);                            // Total  = Ambient + LED IR reflected from object
  irRight  = analogRead(IRRIGHTPIN);
  
  // Top/Bottom not needed by this program.  Code included for experimentation if you wish.
  //irTop    = analogRead(IRTOPPIN);                              
  //irBottom = analogRead(IRBOTTOMPIN);  

  digitalWrite(IRLEDSPIN,LOW);                                 // Turn off IR LEDs to read Ambient light (ie from indoor lighting and natural light)
  delayMicroseconds(500);                                      // Allow time for phototransistors to respond. (may not be needed)
   
  irLeft      =    irLeft   -   analogRead(IRLEFTPIN);         // Reflected IR = Total (read previously) - Ambient (read now)
  irRight     =    irRight  -   analogRead(IRRIGHTPIN);            
  
  //irTop       =    irTop    -   analogRead(IRTOPPIN);                
  //irBottom    =    irBottom -   analogRead(IRBOTTOMPIN);
  //irValue = ( irLeft + irRight + irTop + irBottom ) / 4;  

  irValue = ( irLeft + irRight ) / 2;                          // Distance of object is average of reflected IR
                                                               // Low values ( ie 34,35,36 denote far away , high values ie 634,635,636 denote close )
                                                              
         
  if ( getBaseline ) {  // Get the baseline level denoting "no target detected"
                        // This code is only executed once.
      getBaseline        = false;        
      irValue_baseline   = irValue; 
      irLeft_baseline    = irLeft;
      irRight_baseline   = irRight;
      //irTop_baseline     = irTop;
      //irBottom_baseline  = irBottom;
  }    
                                                               

  return   ( ( irValue - irValue_baseline ) > 25 ) ? true : false;  // I chose 25 as my sensitivity level over baseline, , change if you wish
  
}


#ifdef LASERPIN
boolean fireLaser( int times ) {  
    
  // Incoming parameters : int Number of shots
  
  // Responsibilities    : Fire laser while illuminating LEDs (if any), then extinguish LEDs. 
  
  // Returns             : false to indicate laser discharged
  
  if ( times > 4 ) times = 4;   // No particular reason for 4 as an upper limit, change if you wish
  
  for ( i=0; i < times; i++ ) {
    
      digitalWrite( LASERPIN , HIGH );   
      

      #ifdef TM1638INSTALLED
      tm1638.setDisplayToString("  FIRE  ",0,0);
      tm1638.setLEDs( 24 );   delay(150);
      tm1638.setLEDs( 36 );   delay(150);
      tm1638.setLEDs( 66 );   delay(150);
      tm1638.setLEDs( 129 );  delay(150);
      #else
      delay(400);
      #endif
      
      digitalWrite( LASERPIN , LOW );          
          #ifdef TM1638INSTALLED
          tm1638.setDisplayToString("  ----  ",0,0);
          #endif  
      delay(400);

    
  }
  
  #ifdef TM1638INSTALLED
  tm1638.setLEDs( 0 );  // Laser discharged
  #endif
  
  #ifdef XBeeINSTALLED
  Serial.println("Laser is discharged. ");
  #endif
  
  return false;
  
}
#endif


boolean timeIsUp( unsigned long& then , unsigned int desiredInterval) {
  
  // Incoming parameters : unsigned long (by ref) counter (in milliseconds) , unsigned int (by copy) desired cycle.
 
  // Responsibilities    : When the counter hits the cycle threshold (based on runtime), it will set the counter to current runtime, 
  //                       and inform whoever called this that "time is up".
  
  // Returns             : true if cycle threshold achieved, false otherwise
  
  unsigned long now = millis();
  boolean desiredIntervalAchieved = false;
  
  if (now - then >= desiredInterval)  {
    then = now;
    desiredIntervalAchieved = true;
  }

  return desiredIntervalAchieved;

}



void relocateRover ( void ) {   
  
  // Incoming parameters : Nothing
 
  // Responsibilities    : Function stub... see explanation below
  
  // Returns             : Nothing
  
  
 // If you've been looking too long without seeing anything, you should consider searching elsewhere.
 // It could be as simple as pivoting around to look behind you, or as complicated as driving somewhere else.
 
 // I'll leave this up to you.  It will give you an opportunity to add your own code to this program.
 
 // What *I'm* going to do for fun here is pivot left to change where I'm looking.
  
  
  digitalWrite(MOTORLEFTDIREC,BACK);          analogWrite (MOTORLEFTSPEED,speedLeft);     // Reverse Left Motor  
  digitalWrite(MOTORRIGHTDIREC,AHEAD);        analogWrite (MOTORRIGHTSPEED,speedRight);   // Forward Right Motor
  delay(1500);
  analogWrite (MOTORLEFTSPEED,0);                                                         // Stop Left motor 
  analogWrite (MOTORRIGHTSPEED,0);                                                        // Stop Right motor 


}



