/*
   Created by:  Jay Weeks
   On date:     13-Sep-2016
   Last update: 3-Nov-2016

   Purpose:
    A PID speed controller, designed to work with Digilent
    gear-motors/encoders and the Pmod HB5.

   Current:
    PWM state machine
      Accepts PWM value
        -255 to 255
        Negative values will reverse the motor.
      Uses H-bridge safe functions to set motor PWM and direction.
    Encoder ISR
      Calculates time between interrupts using timer difference
      function
      Store in a volatile ISR output variable
    Loop
      Calls PWM test state machine to vary PWM output signal.
      Calls PWM state machine to output PWM to motors.
      Prints the time between interrupts to serial.
        Also prints the measured and set direction, to ensure
        agreement between direction booleans.
    PID error state machine
      Checks for update from interrupt
        If no update, returns false.
      If updated, converts period into speed using speed conversion
      function.
      (Resets period variable.)
      NOTE: If the period is too long, this function will default the
        speed to 0.
        It will also defualt the speed to zero if it has been waiting
        too long for an update.
      Converts target value into speed using another speed conversion
      function.
      Uses target_speed to find error.
      Returns true.
    PID state machine
      Runs after there has been an update from PID_error
      Updates history with the current error value
      Calculates P
      Calculates I
      Calculates D
      Uses P + I + D to set output
    Target control state machine
      Works EXACTLY the same as PWM control, only a change in
      variable names.

   Next:
    We're done!

   Notes:
    This sketch is used in tandem to a Pmod HB5.
    pin7/RB14       ->  Direction
    pin6/RB13/OC5   ->  Enable
    pin3/RB9/INT1   <-  SA
    pin9/RA0        <-  SB
    All grounds tied together
    3v3             <-  V

    The micros() function will loop back to zero after about 70
    minutes.
    The millis() function will loop after about 50 days.
    Both of these functions return an unsigned long, which has a
    maximal value of 4294967295, or 0xFFFFFFFF.
    The time_diff() function will account for this looping, so
    no matter what, you can always get a good time difference.

    Our direction pin happens to allign with our assigned
    direction. In other words, when we tell our motor to spin in
    the 0 direction, our encoder direction pin will show a 0,
    and vice-versa.
    It's important to note that this will not always be the case,
    and if it is not, then notation must be change in code so
    the input and the output agree. Otherwise the PID will be
    confused and useless, and that makes PIDs sad. :(

    General convention for directions and signs:
      False/0 : + direction
      True/1  : - direction
*/

/****************************************************************

 ****************************************************************/

/********************************

 ********************************/

/****************

 ****************/



/****************************************************************
   Libraries
 ****************************************************************/
#include "Utility.h"

/****************************************************************
   Function Definitions and Variables
 ****************************************************************/

/********************************
   State Machines
 ********************************/

/****************
   target_ctrl
 ****************/
void target_ctrl();
/*
  Functions exactly the same as PWM_ctrl, but adjusts the target,
  and not the PWM.
*/
unsigned long target_ctrl_pot_time = 0;
unsigned long target_ctrl_btn1_time = 0;
unsigned long target_ctrl_btn2_time = 0;
// Stores the last time our pot, btn1, and btn2 updated.
bool target_ctrl_btn1_prev = false;
bool target_ctrl_btn2_prev = false;
// Stores the previous state of button 1 and 2.
int target_ctrl_target_inc = 0;
// Stores our target increment.

/****************
   PID_error_update
 ****************/
bool PID_error_update();
/*
   Checks to see if there's been an update from the interrupt.
   If there has, updates the history and outputs true.
   Otherwise, outputs false.
*/
unsigned int PID_error_time = 0;
  // This will keep track of the last time we ran an update.
int ISR_speed = 0;
int target_value = 0;
  // Target ranges from -255 to 255
  // This will need to be converted to a speed before it's useful.
int PID_error = 0;

/****************
   PID
 ****************/
int PID();
/*
    Updates our history array.
    Updates our PID 
*/
float I = 0;
int PID_output = 0;
int PID_prev_error = 0;

/****************
   PWM_set
 ****************/
void PWM_set(int PWM_input);
unsigned int PWM_state = 0;
// PWM set state list
// 0: Nothing special
// 1: Direction switch: Just set PWM to zero. Waiting to settle
//                      before switing direction pin.
// 2: Direction switch: Just switched direction pin. Waiting to
//                      settle before setting new PWM.
bool PWM_dir = true;
unsigned long PWM_time = 0;
// Used to store the current direction being output to the motor.
// positive RPM => 1 or true direction
// negative RPM => 0 or false direction


/********************************
   Utility functions
 ********************************/
unsigned long time_diff(unsigned long prev, unsigned long curr);
int period_to_speed(unsigned long period);
int value_to_speed(int value);
  // convert -255 to 255 target value to a speed


/********************************
   Encoder ISR
 ********************************/
void encoderISR();
volatile unsigned long ISR_time = 0;
volatile bool ISR_dir = false;
volatile unsigned long ISR_diff = 0;

/****************************************************************
   Setup and Loop
 ****************************************************************/
void setup()
{
  /********************************
     Serial
   ********************************/
  #if DEBUGGING
    Serial.begin(9600);
  #endif
  
  /********************************
     IO pins
   ********************************/
  // Input
  pinMode(SA, INPUT);
  pinMode(SB, INPUT);
  // Output
  pinMode(DIR, OUTPUT);
  pinMode(EN, OUTPUT);
  
  /********************************
     Attaching encoder interrupt
   ********************************/
  attachInterrupt(1, encoderISR, ENCODER_DIRECTION);
  
  /********************************
     Initializing our motor
   ********************************/
  analogWrite(EN, 0);
  delay(10); // Let our state settle
  digitalWrite(DIR, PWM_dir);
  delay(10); // Let our state settle
}

void loop()
{
  target_ctrl();
  PID(PID_error_update());
  PWM_set(PID_output);
  
  
  #if DEBUGGING
  #if ISR_DEBUGGING
    if (ISR_diff)
    {
      Serial.print("ISR_diff\t");
      Serial.print(ISR_diff);
      Serial.print("\tISR_dir\t");
      Serial.print(ISR_dir);
      Serial.print("\n");
    }
  #endif  // End loop debugging
  #endif  // End debugging
}


/****************************************************************
   State Machines
 ****************************************************************/

/********************************
   target_ctrl
 ********************************/
void target_ctrl()
{
  #if DEBUGGING
  #if TARGET_CTRL_DEBUGGING
    bool updateSerial = false;
  #endif
  #endif
  bool target_ctrl_btn1_curr = 0, target_ctrl_btn2_curr = 0;
  unsigned long target_ctrl_pot_diff = time_diff(target_ctrl_pot_time, millis());
  unsigned long target_ctrl_btn1_diff = time_diff(target_ctrl_btn1_time, millis());
  unsigned long target_ctrl_btn2_diff = time_diff(target_ctrl_btn2_time, millis());
  int potRead = 0;

  // Set our target increment
  // Has it been long enough since our last change?
  if (target_ctrl_pot_diff >= 50)
  { // It has? Cool.
    // First get our Pot value
    int potRead = analogRead(POT);
    // Convert our POT reading to a discreet PWM increment
    if (potRead <= 341) potRead = 1;
    else if (potRead <= 682) potRead = 10;
    else potRead = 100;
    // See if that's different from what we've already got
    if (target_ctrl_target_inc != potRead)
    { // If it is
      // Set our new target increment
      target_ctrl_target_inc = potRead;
      #if DEBUGGING
      #if TARGET_CTRL_DEBUGGING
            // Let serial know it needs to update
            updateSerial = true;
      #endif
      #endif
      // Store our current Pot time
      target_ctrl_pot_time = millis();
    }
  }

  // Increment up
  // Check for a button 2 change
  // Has it been long enough since last time?
  if (target_ctrl_btn2_diff >= 50)
  { // It has? Cool.
    // Get our current Btn2 state
    target_ctrl_btn2_curr = digitalRead(BTN2);
    // Has it changed since last time?
    if (target_ctrl_btn2_curr && !target_ctrl_btn2_prev)
    { // If it has, increment our PWM up
      target_value += target_ctrl_target_inc;
      #if DEBUGGING
      #if TARGET_CTRL_DEBUGGING
            // Let serial know it needs to update
            updateSerial = true;
      #endif
      #endif
      // Store our current Btn2 time
      target_ctrl_btn2_time = millis();
    }
    // Store our current Btn2 state
    target_ctrl_btn2_prev = target_ctrl_btn2_curr;
  }

  // Increment down
  // Check for a button 1 change
  // Has it been long enough since last time?
  if (target_ctrl_btn1_diff >= 50)
  { // It has? Cool.
    // Get our current Btn1 state
    target_ctrl_btn1_curr = digitalRead(BTN3);
    // Has it changed since last time?
    if (target_ctrl_btn1_curr && !target_ctrl_btn1_prev)
    { // If it has, increment our PWM down
      target_value -= target_ctrl_target_inc;
      #if DEBUGGING
      #if TARGET_CTRL_DEBUGGING
            // Let serial know it needs to update
            updateSerial = true;
      #endif
      #endif
      // Store our current Btn1 time
      target_ctrl_btn1_time = millis();
    }
    // Store our current Btn1 state
    target_ctrl_btn1_prev = target_ctrl_btn1_curr;
  }

  // Check for boundaries
  if (target_value < -255) target_value = -255;
  else if (target_value > 255) target_value = 255;
  
  #if DEBUGGING
  #if TARGET_CTRL_DEBUGGING
    // Check for any changes to promt a Serial update
    if (updateSerial)
    {
      // Output our increment
      Serial.print("target_ctrl_target_inc\t");
      Serial.print(target_ctrl_target_inc);
      Serial.print("\ttarget_value\t");
      // Output our current PWM
      Serial.print(target_value);
  
      // End the line
      Serial.print("\n");
    }
  #endif
  #endif
}


/********************************
   PID_error_update()
 ********************************/
bool PID_error_update()
{
  int target_speed;
  double temp;
  bool ret;
  
  ret = false;
  
  // How long has it been since the last update?
  unsigned long PID_error_diff = time_diff(PID_error_time, millis());
  
  // Has there been an interrupt?
  if (ISR_diff)
  { // If there has
    // Have we been waiting a long time since our last update?
    if (PID_error_diff > PERIOD_MAX)
    { // If we have..
      // Default our speed to zero.
      ISR_speed = 0;
    }
    else
    { // Otherwise
      // Get our actual encoder/motor speed.
      ISR_speed = period_to_speed(ISR_diff);
    }

    #if DEBUGGING
    #if PID_ERROR_DEBUGGING
        Serial.print("ISR_diff\t");
        Serial.print(ISR_diff);
        Serial.print("\tISR_speed\t");
        Serial.print(ISR_speed);
    #endif
    #endif
    
    ISR_diff = 0;
    
    // Calculate our error
    target_speed = value_to_speed(target_value);
    PID_error = target_speed - ISR_speed;
    
    #if DEBUGGING
    #if PID_ERROR_DEBUGGING
        Serial.print("\ttarget_speed\t");
        Serial.print(target_speed);
        Serial.print("\tPID_error\t");
        Serial.print(PID_error);
    #endif
    #endif
    
    // Update our PID_error_time
    PID_error_time = millis();

    #if DEBUGGING
    #if PID_ERROR_DEBUGGING
        Serial.print("\tPID_error_time");
        Serial.print(PID_error_time);
    #endif
    #endif
    
    ret = true;
  }
  // Have we been waiting a long time since the last update?
  else if ((PID_error_diff > PERIOD_MAX))
  { // If we have
    // We assume a measured speed of zero.
    PID_error = -target_speed;

    #if DEBUGGING
    #if PID_ERROR_DEBUGGING
        Serial.print("\tISR_speed");
        Serial.print(ISR_speed);
    #endif
    #endif

    // Calculate our error
    target_speed = value_to_speed(target_value);
    PID_error = target_speed - ISR_speed;

    #if DEBUGGING
    #if PID_ERROR_DEBUGGING
        Serial.print("\ttarget_speed");
        Serial.print(target_speed);
        Serial.print("\t");
        Serial.print(PID_error);
    #endif
    #endif
    
    // Update our PID_error_time
    PID_error_time = millis();
    
    ret = true;
  }
  
  #if DEBUGGING
  #if PID_ERROR_DEBUGGING
      if (ret) Serial.print("\n");
  #endif
  #endif
  
  return ret;
}


/********************************
   PID()
 ********************************/
int PID(bool update)
{
  unsigned int PID_diff;
  float P, D;
  float total;
  int temp1;
  
  // Have we gotten an update on our error?
  if (update)
  { // If it has
    
    #if DEBUGGING
    #if PID_DEBUGGING
      Serial.print("time (ms)\t");
      Serial.print(micros());
      Serial.print("\ttarget speed\t");
      Serial.print(value_to_speed(target_value));
      Serial.print("\tISR_speed\t");
      Serial.print(ISR_speed);
      Serial.print("\tPID_error\t");
      Serial.print(PID_error);
      Serial.print("\tPID_output\t");
      Serial.print(PID_output);
    #endif
    #endif

    // Calculate P value
    P = PID_error * K_P;
    
    #if DEBUGGING
    #if PID_DEBUGGING
      Serial.print("\tP\t");
      Serial.print(P);
    #endif
    #endif
    
    // Calculate I value
    I += PID_error * K_I;
    
    #if DEBUGGING
    #if PID_DEBUGGING
      Serial.print("\tI\t");
      Serial.print(I);
    #endif
    #endif
    
    temp1 = PID_error - PID_prev_error;
    D = temp1 * K_D;
    
    #if DEBUGGING
    #if PID_DEBUGGING
      Serial.print("\tD\t");
      Serial.print(D);
    #endif
    #endif
    
    // Calculate our total increment
    total = P + I + D;
    
    #if DEBUGGING
    #if PID_DEBUGGING
      Serial.print("\ttotal\t");
      Serial.print(total);
    #endif
    #endif
    
    // Set our PWM output
    PID_output = total;
    
    #if DEBUGGING
    #if PID_DEBUGGING
      Serial.print("\tPID_output\t");
      Serial.print(PID_output);
      Serial.print("\n");
    #endif
    #endif
    
    // Store our PID_error
    PID_prev_error = PID_error;
  }

  return 0;
}

/********************************
   PWM_set
 ********************************/
void PWM_set(int PWM_new)
{
  unsigned long PWM_time_diff = 0;
  
  // Enter state machine!
  switch (PWM_state)
  {
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Normal Operation
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    case 0: // Normal operation
      // Check to see if it's okay to set the new PWM
      if ((PWM_new >= 0) == PWM_dir)
      { // If we don't need to reverse things, we can set the new
        // PWM
        // But first we need to get it's absolute value
        if (PWM_new < 0) PWM_new = -PWM_new;
        // NOW we can set the new PWM
        analogWrite(EN, PWM_new);
        
        #if DEBUGGING
        #if PWM_SET_DEBUGGING
          Serial.write("PWM_set() has set a new motor speed.\n");
        #endif
        #endif
        
      } else
      { // If we need to reverse things, we can't set the new PWM
        // Bring our PWM output down to zero
        analogWrite(EN, 0);
        // Set our state to step 1 of a direction switch
        PWM_state = 1;
        // set our timer to waiting
        PWM_time = millis();
        // Remember, for PWM_set, we use milliseconds
        
        #if DEBUGGING
        #if PWM_SET_DEBUGGING
          Serial.write("PWM_set() has stopped the motor and is waiting to reverse.\n");
        #endif
        #endif
        
      }
      break;
  
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Waiting for PWM/EN to settle
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    case 1:
      // How long has it been since we set the PWM to zero?
      PWM_time_diff = time_diff(PWM_time, millis());
      // Remember, for PWM_set, we use milliseconds
  
      // Has it been long enough?
      if (PWM_time_diff >= PWM_SETTLE_TIME)
      { // If it has been long enough...
        // Reverse the direction
        PWM_dir = !PWM_dir;
        // And set the new direction to the direction pin.
        digitalWrite(DIR, PWM_dir);
  
        // Now switch to the next state
        PWM_state = 2;
        // And set our new timer
        PWM_time = millis();
        // Remember, for PWM_set, we use milliseconds
        
        #if DEBUGGING
        #if PWM_SET_DEBUGGING
          Serial.write("PWM_set() has reversed and is waiting to start again.\n");
        #endif
        #endif
        
      }
      break;
  
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Waiting for DIR to settle
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    case 2:
      // How long has it been since we switched direction?
      PWM_time_diff = time_diff(PWM_time, millis());
      // Remember, for PWM_set, we use milliseconds
  
      // Has it been long enough?
      if (PWM_time_diff >= PWM_SETTLE_TIME)
      { // If it has been long enough...
        // Switch to the next state so we can finally set our new
        // PWM
        PWM_state = 0;
      }
      break;
  }
}

/****************************************************************
   Utility Functions
 ****************************************************************/
unsigned long time_diff(unsigned long prev, unsigned long curr)
{
  long diff = curr - prev;
  if (diff < 0) diff = diff + 0xFFFFFFFF;
  return diff;
}

int period_to_speed(unsigned long period)
{
  int time_output;
  
  // False / 0  : - direction
  // True / 1   : + direction
  
  // Calculate tics per second
  time_output = 1000000 / (period);

  // Assign our value polarity based on direction
  if (!ISR_dir) time_output = -time_output;

  return time_output;
}

int value_to_speed(int value)
{
  // For this speed conversion, we will assume
  //  0: stopped
  //  1: lowest possible speed
  //  255:  Highest possible speed

  float temp1;
  int speed_output;
  
  #if DEBUGGING
  #if VALUE_TO_SPEED_DEBUGGING
    Serial.print("value\t");
    Serial.print(value);
  #endif
  #endif
  
  // Split between positive and negative signs.
  if (value > 0)
  { // If our value is positive...
    // Use positive constants to convert to speed
    temp1 = value / 255.0;
    
    #if DEBUGGING
    #if VALUE_TO_SPEED_DEBUGGING
      Serial.print("\ttemp1\t");
      Serial.print(temp1);
    #endif
    #endif
    
    speed_output = temp1 * SPEED_RANGE;
    
    #if DEBUGGING
    #if VALUE_TO_SPEED_DEBUGGING
      Serial.print("\tspeed_output\t");
      Serial.print(speed_output);
    #endif
    #endif
    
    speed_output += SPEED_MIN;
    
    #if DEBUGGING
    #if VALUE_TO_SPEED_DEBUGGING
      Serial.print("\tspeed_output\t");
      Serial.print(speed_output);
      Serial.print("\n");
    #endif
    #endif
  }
  else if (value < 0)
  { // If our value is negative...
    // Use negative constants to convert to speed
    temp1 = value / 255.0;
    
    #if DEBUGGING
    #if VALUE_TO_SPEED_DEBUGGING
      Serial.print("\ttemp1\t");
      Serial.print(temp1);
    #endif
    #endif
    
    speed_output = temp1 * SPEED_RANGE;
    
    #if DEBUGGING
    #if VALUE_TO_SPEED_DEBUGGING
      Serial.print("\tspeed_output\t");
      Serial.print(speed_output);
    #endif
    #endif
    
    speed_output -= SPEED_MIN;
    
    #if DEBUGGING
    #if VALUE_TO_SPEED_DEBUGGING
      Serial.print("\tspeed_output\t");
      Serial.print(speed_output);
      Serial.print("\n");
    #endif
    #endif
  }
  else  return 0; // If our value is zero, no need for conversion.
  
  return speed_output;
}

/****************************************************************
   Encoder ISR
 ****************************************************************/
void encoderISR()
{
  // Get our time difference
  ISR_diff = time_diff(ISR_time, micros());
  // Get our direction
  ISR_dir = digitalRead(SB);
  // Store our current time
  ISR_time = micros();
}



// I really hate typing at the very bottom of the page, don't you?
