// Source          ::     VulcanTurret_WiFi_V1 ( June 2016 )

// Description     ::     Turret control program using WiFi. Allows the operator to control the turret in the following ways: 
//                            aiming the turret from left to right,  aiming the turret up and down, 
//                            automatic panning, and firing the turret (provided the gun is loaded with an ammunition belt).
//
//
//                        WiFi access would typically be done through a local router,
//                        but by using Dynamic DNS ie http://www.noip.com/remote-access you could control the turret
//                        from anywhere in the world.
//
//                        Available commands ...
//
//                                   ( a - z ) - Controls tilt between 65 and 115 degrees, in 2 degree steps
//                                   ( A - Z ) - Controls pan between 52 and 152 degrees, in 4 degree steps
//                                   ( 0 )     - Fires a single round
//                                   ( 9 )     - Fires three round burst
//                                   ( + )     - Starts automatic panning
//                                   ( - )     - Stops automatic panning

//                        Vulcan Sentry Gun construction instructions...
//                        http://www.instructables.com/id/WiFi-Internet-Android-Controlled-Nerf-Vulcan-Sentr/



// Original Program   ::     Main control & program standards      - Joe M                        / arduino (at) bellaliant.net   / March 2015
//                           WiFi Connectivity                     - Rezk Bouras aka (Prudent++)  / Student, NBCC Saint John      / March 2015
//                          

// Modified By        ::     Conversion to turret functionality    - Joshua Parker                / Student, NBCC Saint John      / March 2016
//                           Code optimization                     - Joe M                        / arduino (at) bellaliant.net   / May   2016    
//                    ::     Your Name Here / Contact / Date  



//  This program is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License 
//  as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  Lesser General Public License for more details.
//
//  For a copy of the GNU Lesser General Public License, 
//  write to the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA


// **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T **** **** I M P O R T A N T ****
// ***********************************************************************************************************************************************************************
// Pinouts... IMPORTANT!!!! ... set the pinouts to your hardware configuration
//
// Digital pins 0 and 1 can only be used if you ARE NOT using Serial communications ( ie Serial.begin() ).
// Also note that having components connected to D0 and D1 will interfere with uploading your Sketch.  
//
// 
// Pinouts on WiFi Shield explained here ... http://arduino.cc/en/Main/ArduinoWiFiShield
//
// ***********************************************************************************************************************************************************************

// ********** Digital Pins D0 thru D13 **********

// Using 0 and 1 disables upload!     

#define DO_NOT_USE_0        0
#define DO_NOT_USE_1        1

#define NOT_USED_2          2
#define NOT_USED_3          3
#define DO_NOT_USE_4        4         // SS for SD Card
#define SPEAKER             5         // Onboard speaker on BotBoarduino

#define NOT_USED_6          6         
#define WIFI_HANDSHAKE      7         // Pin 7 will be used for the WiFi Shield handshake
#define NOT_USED_8          8         
#define NOT_USED_9          9  
       
#define DO_NOT_USE_10       10        // SS for WiFi
#define DO_NOT_USE_11       11        // MOSI
#define DO_NOT_USE_12       12        // MISO
#define DO_NOT_USE_13       13        // SCK


// ********** Analog Pins A0 thru A5 **********

// FYI... using analog pins for digital I/O is just the same as using digital ones.

// A0 is referred to as Pin 14     A1 is referred to as Pin 15
// A2 is referred to as Pin 16     A3 is referred to as Pin 17
// A4 is referred to as Pin 18     A5 is referred to as Pin 19

#define NOT_USED_A0         A0 
#define NOT_USED_A1         A1
#define NOT_USED_A2         A2 

#define SERVO_TRIGGER       17         // Trigger motor 
#define SERVO_PAN           18         // Pan servomotor  
#define SERVO_TILT          19         // Tilt servomotor



// *********************************************************************************************************************************************************************
// Required includes
// *********************************************************************************************************************************************************************


#include <SPI.h>
#include <WiFi.h>
#include <Servo.h>


// *********************************************************************************************************************************************************************
//  Global Variables.
//  Professional programmers cringe at the thought of globals, but they seem to be generally acceptable in the "wild west" Arduino world.
//  Prudent use of them simplifies program design since variable passing via parameters between functions are reduced.
// *********************************************************************************************************************************************************************

// An homage to every program I've written since my WATFIV days.
// Every program needs an "int i", right?
int i;

char operatorCommand = '5'; 

char ssid[]       = "VulcanTurret";   // Change to your network SSID (name)  
                             
char pass[]       = "pass1234";       // Change to your network password (use for WPA, or use as key for WEP)

int tiltAngle     = 91;               // default positions for turret aim
int tiltAngleMIN  = 65;               // default minimum for tilt is 65 degrees
int tiltAngleMAX  = 115;              // default maximum for tilt is 115 degrees
int tiltAngleNEW  = 91;               // this will be used to change the tilt

int panAngle      = 104;              // default positions for turret aim
int panAngleMIN   = 52;               // default minimum for pan is 45 degrees
int panAngleMAX   = 152;              // default minimum for pan is 135 degrees
int panAngleNEW   = 104;              // this will be used to change the pan 

boolean autoScan      = false;        // boolean to control access to auto pan - true indicates auto pan enabled
char    scanDirection = 'l';          // scanDirection used by auto-scan feature in android app, initialized to l for left 


int keyIndex = 0;                     // Your network key Index number (needed only for WEP)

int status = WL_IDLE_STATUS;

WiFiServer server(1994);
WiFiClient commChannel;
  
// General defines

#define NOT  !



//#define SERIALCOMM            // Activate this line if you want Serial Communications for debugging purposes.
                              // For example, prior to achieving full functionality, you might want to connect the turret
                              // to a computer using the USB cable.  Then, you can use the Arduino Serial Monitor to view
                              // information messages being generated by the turret.
                              // Once you have full functionality, you can recompile this program with this line commented out
                              // because once you are operating using WiFi only, you have no need of the Serial Communications.
                              // You won't harm anything leaving them activated, but why waste CPU cycles executing useless commands? 

//#define WIFICOMM              // Activate this line if you want the gun communicating messages back to the operator via the WiFi channel.

  
// **********************************************************************************************************************************************************************
// Servo Objects
// **********************************************************************************************************************************************************************

  Servo tilt;                 // servo object for tilt control (up/down)
  Servo pan;                  // servo object for pan control  (left/right)

// **********************************************************************************************************************************************************************
// Function Prototypes                                                                                                                                                
// **********************************************************************************************************************************************************************

void turretAction       ( void );   // Decide on which action to perform.
void changeGunPosition  ( void );   // Move gun up/down , left/right.
void fireGun            ( void );   // Fire single round.
void autoPan            ( void );   // Automatically pans turret from left to right, and vice versa.

#ifdef SERIALCOMM
void WiFiStatus         ( void );   // This function available only if Serial Communications are requested.
#endif

   
   
// *********************************************************************************************************************************************************************
// Main program                                                                                                                                                                                                                   
// *********************************************************************************************************************************************************************
  
void setup( ) {
  
  #ifdef SERIALCOMM
  //Initialize serial communications (ie via USB cable).
  Serial.begin(9600);
  #endif
     
  // Prepare pin for handshaking.
  pinMode(WIFI_HANDSHAKE,INPUT);
  
  // Check for the presence of the shield
  while ( WiFi.status() == WL_NO_SHIELD ) {  
    #ifdef SERIALCOMM  
    Serial.println("WiFi shield not present");
    #endif
    delay(1000);    
  }
  
  // Attempt to connect to WiFi network
  while ( status != WL_CONNECTED ) {
    
    #ifdef SERIALCOMM
    Serial.print("Attempting to connect to SSID: ");
    Serial.println(ssid);
    #endif
    
    // Connect to WPA/WPA2 network. Change this line if using Open or WEP network
    status = WiFi.begin(ssid, pass);

    // Wait 10 seconds for connection
    delay(10000);
  }

  // Start the server
  server.begin();
  
  #ifdef SERIALCOMM
  // You're connected now, report status
  WiFiStatus();
  #endif
  
  do {
   commChannel = server.available();
   delay(1000); 
  } while ( NOT commChannel.connected() );
  
  
  commChannel.flush();
  

  tilt.attach (SERVO_TILT);    
  pan.attach  (SERVO_PAN);     
  
  #ifdef SERIALCOMM
  Serial.println("Vulcan Turret online.");
  Serial.println("Enter any character between a-z to control tilt and A-Z to control pan.");
  Serial.println("To enable auto-pan scanning, press +, and to disable auto-pan scanning press - .");
  Serial.println("To fire a round, press 0 (Zero).");
  #endif
  
  #ifdef WIFICOMM
  commChannel.println("Vulcan Turret online.");
  commChannel.println("Enter any character between a-z to control tilt and A-Z to control pan.");
  commChannel.println("To enable auto-pan scanning, press +, and to disable auto-pan scanning press - .");
  commChannel.println("To fire a round, press 0 (Zero).");
  delay(1000);
  #endif

}

void loop( ) {

    
    // If a command is in the buffer...
    if ( commChannel.available ( ) > 0 ) {
             
      // ..read it...    
      operatorCommand = commChannel.read ( );    
     
      // ...and decide what action to perform.
      turretAction ( );  
      
    }
 
    
}



// *********************************************************************************************************************************************************************
// Functions <------------------------------------------------------------------------------------------------------------------------------------------------------- 
// *********************************************************************************************************************************************************************



void turretAction ( void ) {

  // Incoming parameters : Nothing.
  // Global variables    : Uses    -   operatorCommand, tiltAngle, panAngle
  //                       Changes -   tiltAngleNEW, panAngleNEW, autoScan
  // Responsibilities    : Decide what action to perform based on operatorCommand
  // Returns             : Nothing.
      
      
      tiltAngleNEW = tiltAngle;
      panAngleNEW  = panAngle;
      
      switch( operatorCommand ) { // Determine the action corresponding to operator command received
    

              // FIRE !!!  
              case '0':   fireGun ( );                      return;              
              
              case '9':   for (i=0; i<3; i++) fireGun( );   return;

              
              /***********************************************************************************
              * The turret will change orientation based on the character received; receiving a  *
              * lower case letter will change the tilt, and receiving an upper case letter will  *
              * change the pan. The user can also select a scanning function, which will         *
              * automatically pan left and right until another command is received.              *
              ***********************************************************************************/
             
              // auto panning 
              case '+':   autoPan ( );                      return;
              
                             
          
              // tilt cases a thru z calculate 65 - 115 degrees in steps of 2
              
              case 'a': case 'b': case 'c': case 'd': case 'e': case 'f': case 'g': case 'h': case 'i': case 'j': case 'k': case 'l': case 'm':
              case 'n': case 'o': case 'p': case 'q': case 'r': case 's': case 't': case 'u': case 'v': case 'w': case 'x': case 'y': case 'z':

                           tiltAngleNEW = 65 + ( ( operatorCommand - 'a' ) * 2 );
                           break;

                               
              
              // pan cases A thru Z calculate 52 - 152 degrees in steps of 4 

              case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G': case 'H': case 'I': case 'J': case 'K': case 'L': case 'M':
              case 'N': case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U': case 'V': case 'W': case 'X': case 'Y': case 'Z':

                           panAngleNEW = 52 + ( ( operatorCommand - 'A' ) * 4 );
                           break;
                               
             
      } // end switch 
    
      // New position requested
      if ( ( tiltAngle != tiltAngleNEW ) || ( panAngle != panAngleNEW ) ) changeGunPosition ( );

   

} // end turretAction function


void changeGunPosition ( void ) {

  // Incoming parameters : Nothing.
  // Global variables    : Uses    -   tiltAngleNEW, panAngleNEW
  //                       Changes -   tiltAngle, panAngle
  // Responsibilities    : Move gun to desired tilt/pan angles.
  // Returns             : Nothing.

      // output to indicate new values for debug purposes
      #ifdef WIFICOMM
      commChannel.print   ("Tilt at ");
      commChannel.print   (tiltAngleNEW);
      commChannel.println (" degrees.");
      commChannel.print   ("Pan at ");
      commChannel.print   (panAngleNEW);
      commChannel.println (" degrees.");
      #endif
          
      #ifdef SERIALCOMM
      Serial.print        ("Tilt at ");
      Serial.print        (tiltAngleNEW);
      Serial.println      (" degrees.");    
      Serial.print        ("Pan at ");
      Serial.print        (panAngleNEW);
      Serial.println      (" degrees.");
      #endif
      
      
      
      
      // ************************************************************************************
      // *                                                                                  * 
      // *  Due to the size and weight of the turret, quick changes in orientation          *
      // *  can cause a lot of inertial pressure on the servomotors. In order to prevent    *
      // *  unnecessary stress and breakage, we will change orientation through a loop      *
      // *  which will instead quickly step up or down to the new degree rotation.          *
      // *                                                                                  *
      // *  This is the recommended method for controlling the action of your gun.          *
      // *                                                                                  *
      // ************************************************************************************
  
      while ( tiltAngle != tiltAngleNEW ) {
       
           if ( tiltAngle < tiltAngleNEW )      // when tilt is lower than desired, we will step up by 2 degrees
             tiltAngle += 2;
           else if ( tiltAngle > tiltAngleNEW ) // when tilt is higher than desired, we will step down by 2 degrees
             tiltAngle -= 2;
             
           tilt.write(tiltAngle);               // moves the turret to the new tilt position
           delay(100);                          // Change this to alter speed
  
      }
          
       
      while ( panAngle != panAngleNEW ) { 
        
           if ( panAngle < panAngleNEW )        // when pan is lower than desired, we will step up by 4 degrees
             panAngle += 4;
           else if ( panAngle > panAngleNEW )   // when pan is higher than desired, we will step down by 4 degrees
             panAngle -= 4;       
           
           pan.write(panAngle);                 // moves the turret to the new pan position
           delay(100);                          // Change this to alter speed
  
       } 
       
       

      /*
      // **************************************************************************************
      // *                                                                                    * 
      // *  Activate the code below if you want the servos to respond directly without delays.*
      // *  You will get maximum speed, but be aware that the gun may be rather "violent"     *
      // *  in its actions.                                                                   *
      // *                                                                                    *
      // *  Only the code above, OR the code below should be active, not both. Ensure that    *
      // *  one set of code is commented out.                                                 *
      // *                                                                                    *
      // **************************************************************************************
       
       tilt.write(tiltAngle = tiltAngleNEW);
       pan.write(panAngle = panAngleNEW); 
      */ 
       
} // end changeGunPosition function


void fireGun ( void ) {

  // Incoming parameters : Nothing.
  // Global variables    : Uses    -   
  //                       Changes -   
  // Responsibilities    : Fire a single round.
  // Returns             : Nothing.
  
      digitalWrite ( SERVO_TRIGGER, HIGH );
      delay(325);                               // delay allows firing of one round per command
      digitalWrite ( SERVO_TRIGGER, LOW  );
      
      // output to console that a round has been fired
      #ifdef WIFICOMM
      commChannel.println ("Round fired.");   
      #endif                   
      #ifdef SERIALCOMM
      Serial.println      ("Round fired.");
      #endif
     
                
} // end fireGun function


void autoPan ( void ) {
  
  // Incoming parameters : Nothing.
  // Global variables    : Uses    -   panAngleMAX, panAngleMIN
  //                       Changes -   panAngle, panAngleNEW
  // Responsibilities    : Automatically pans turret from left to right, and vice versa
  // Returns             : Nothing.
      
       int angle = 4;
       
       // Pan back and forth until a new command detected
       
       while ( commChannel.available ( ) == 0 ) {

          panAngle += angle;
          
          if ( panAngle > panAngleMIN && panAngle < panAngleMAX ) {
            
            pan.write ( panAngle );
            delay(150);
            
          }
          else {
            
            angle *= -1;
            
          } 
          
          
       }

       panAngleNEW = panAngle;


} // end autoPan function



#ifdef SERIALCOMM
void WiFiStatus( void ) {
  
  // Incoming parameters : Nothing.
      
  // Global variables    : Uses    -   
  //                       Changes -  
  
  // Responsibilities    : Display network info on Serial.
  
  // Returns             : Nothing.
  
  // Print the SSID of the network you're attached to
  Serial.print("SSID: ");
  Serial.println(WiFi.SSID());

  // Print your WiFi shield's IP address
  IPAddress ip = WiFi.localIP();
  Serial.print("IP Address: ");
  Serial.println(ip);

  // Print the received signal strength
  long rssi = WiFi.RSSI();
  Serial.print("signal strength (RSSI):");
  Serial.print(rssi);
  Serial.println(" dBm");
  
}
#endif

