/* Weather Station Instrumentation & Communication
  This program collects data from the 
  * DHT22 Digital Temperature and Humidity Sensor (https://www.adafruit.com/product/385)
  * SEN-08942 Analog Rain Gauge / Anemometer / Wind Vane (https://www.sparkfun.com/products/8942)
  * RG-11 Analog Rain Sensor (http://rainsensors.com/)
  * MPL3115A2 - I2C Addr 0x60 Barometric Pressure/Altitude/Temperature Sensor (https://www.adafruit.com/products/1893)
  * TSL2561 - I2C Addr 0x39 Luminosity Sensor (LUX) Sensor (https://www.adafruit.com/product/439)
  * MOD-1016 - I2C Addr  0x03 Lightning Detector (http://www.embeddedadventures.com/MOD1016_lightning_sensor_module_mod-1016.html)
  and will do the following:
    1) transmit the data to a local database
    2) publish the data to a local MQTT server
    3) display the data through a locally addressable webpage.
  
  This project uses a Mega and a SeeedStudio ethernet shield and the components listed above.

  From these devices, the following measurements and calculations can be observed:
  * Temperature, humidity, pressure, LUX, Rain State and wind direction are point in time measurments.
  * Rainfall, and windspeed are time measurments as shown below:
  * Rainfall: volume of rain per unit time. However, this rainguage will record tip events where the amount of each event can be calculated as 0.012"
  * Windspeed: volume of air per unit time.

  NOTES
  - The anemometer will record an average number of contact closures per second. 1 per sec equates to 1.492 MPH (2.4 km/h). 
  - Average calculations for windspeed, and wind direction are calculated every 10 seconds during a two minute period. 
  - Moment in time calculations for Humidity, Pressure, and Is It Currently Raining are recorded just prior to transmitting the data to DB
  - Event counts for rainfall, and lightning strikes are recorded and reset during the two minute reporting period.
  - Wind Gusts are calculated during the 2 minute observational period where if a wind speed measurment is higher than the previous
    measurement then it replaces the highest. At the time of recording the data, the max gust is rest to zero.
*/

// Includes
// __________________________________________________________________________________________________

#include "DHT.h"
#include <SPI.h>
#include <Ethernet.h>
#include <MySQL_Connection.h>
#include <MySQL_Cursor.h>
#include <Dns.h>
#include <Wire.h>
#include "RTClib.h"
#include <Adafruit_Sensor.h>
#include <BME280I2C.h>
#include "Adafruit_TSL2591.h"
#include "Adafruit_VEML6075.h"
#include <AS3935.h>
#include <PubSubClient.h>
#include "UserConfig.h"

// Global Variables
// __________________________________________________________________________________________________
// Tri-color operational status LED pins
#define ciLedRedPin 46
#define ciLedGreenPin 48
#define ciLedBluePin 50

// Analog wind direction pin
#define ciWindDirPin 0

// Wind & rain interrupt pins
#define ciWindSpeedPin 3
#define ciRainGaugePin 19
#define ciRainDetectPin 18

// DHT22 sensor pins
#define ciDHTPin 7

// MOD1016 IRQ Pin
#define ciLightningIRQPin 2
const int ciLightningAddress = 03;

// Device status variables
bool bLightingSensorError= false;           //The Lightning sensor has entered an error state
bool bTHPSensorError = false;               //The temp/humidity/pressure sensor has entered an error state
bool bLUXSensorError = false;               //The light sensor has entered an error state
bool bDHTSensorError = false;               //The temp/humidity sensor has entered an error state
bool bUVSensorError = false;                //The UV sensor has entered an error state
bool bRTCError = false;                     //The Realtime Clock has entered an error state
int iDBConnectionFailures = 0;              //The current number of failed consecutive attempts to connect to the database
int iDBConnectionFailuresSinceBoot = 0;     //The total number of errors counted to the database since startup
const int ciMaxDBFailures = 5;              //Maximum database failures before the system is restarted
const long clDBReconnectPause = 10000;      //Database connection retry interval
int iMQTTErrorCount = 0;                    //The current number of failed consecutive attempts to connect to the MQTT broker
bool bUsingFallbackAddress = false;         //Status if the weather station is using DHCP or the fallback static address
int iLightningDisturber = 0;                //Number of non-lightning signals detected since last update
#ifdef __arm__
// should use uinstd.h to define sbrk but Due causes a conflict
extern "C" char* sbrk(int incr);
#else  // __ARM__
extern char *__brkval;
#endif  // __arm__
unsigned int iMinMem = 0;
unsigned long iAvgMem = 0;
unsigned long iMemCount = 0;
unsigned int iUserCount = 0;

// Global working variables
float fWindSpeed = 0.0;
float fAvgWindSpeed = 0.0;
int iWindSpeedCount = 0;
float fWindDirection = 0.0;
float fAvgWindDir = 0.0;
int iWindDirCount = 0;
float fRainfall = 0.0;
float fMaxGust = 0.0;
unsigned int iLightningCount = 0;
float fWindMatrix[16][2]={
                          {292.5,0},
                          {247.5,0},
                          {270,0},
                          {337.5,0},
                          {315,0},
                          {0,0},
                          {225,0},
                          {67.5,0},
                          {45,0},
                          {157.5,0},
                          {180,0},
                          {112.5,0},
                          {135,0},
                          {292.5,0},
                          {90,0},
                          {22.5,0}
};
const String fWindOrdinal[16] PROGMEM ={
                          "WNW",
                          "WSW",
                          "W",
                          "NNW",
                          "NW",
                          "N",
                          "SW",
                          "ENE",
                          "NE",
                          "SSE",
                          "S",
                          "ESE",
                          "SE",
                          "WNW",
                          "E",
                          "NNE"
};

// strings to hold previous values for web display
String sLastTemperature = "";
String sLastHumidity = "";
String sLastDewPoint = "";
String sLastFeelsLike = "";
String sLastLightning = "";
String sLastLightningDisturber = "";
String sLastRaining = "";
String sLastWindSpeed = "";
String sLastGusts = "";
String sLastWindDirection = "";
String sLastWindOrdinalDirection = "";
String sLastBarometricPressure = "";
String sLastRainfall = "";
String sLastLux = "";
String sUVA = "";
String sUVB = "";
String sUVIndex = "";
String sLastUpdate = "";
String sLastInternalTemperature = "";
String sLastInternalHumidity = "";
String sLastInternalDewPoint = "";
String startupdtstamp = "";

String da = "";
String ti = "";

// volatiles that are subject to modification by IRQs
volatile unsigned long windtime, windlast, windbounce, windinterval, zerotime; 
volatile unsigned char windintcount;
volatile boolean gotwspeed;
volatile unsigned long raintime, rainlast, raininterval, rain;
volatile boolean rainDetected = false;
volatile long debounceDelay = 200;    // the debounce time; increase if the output flickers
volatile int MOD1016IrqTriggered;

// Create global timer variables
unsigned long pollTimer = 0;
unsigned long dataRecordTimer = 0;
unsigned long lastDataRecorded = 0;

// Create global status variable
int systemStatus = 1;

// Create global temp sensor parameter constants
#define DHTTYPE DHT22

// Create Object Instances
// __________________________________________________________________________________________________
EthernetServer ethServer(80);
EthernetClient ethClient;
MySQL_Connection mysqlClient((Client *)&ethClient);
DNSClient dnsClient;
PubSubClient mqttClient(ethClient);
BME280I2C sensorTHP;
Adafruit_TSL2591 sensorTSL = Adafruit_TSL2591(2591);
//Adafruit_TSL2561_Unified sensorTSL = Adafruit_TSL2561_Unified(TSL2561_ADDR_FLOAT, 12345);
Adafruit_VEML6075 uv = Adafruit_VEML6075();
DHT sensorDHT(ciDHTPin, DHTTYPE);
DS3231 rtc;

// Primary Functions
// __________________________________________________________________________________________________

void setup () {

  Serial.begin(115200);
  
  // initialize digital pins
  pinMode (ciLedRedPin, OUTPUT); 
  pinMode (ciLedGreenPin, OUTPUT);
  pinMode (ciLedBluePin, OUTPUT);
  pinMode(ciLightningIRQPin, INPUT);
  pinMode (ciWindSpeedPin,INPUT);           // input from wind meters windspeed sensor
  digitalWrite (ciWindSpeedPin,HIGH);       // turn on pullup
  pinMode (ciRainGaugePin,INPUT);           // input from wind meters rain gauge sensor
  digitalWrite (ciRainGaugePin,HIGH);       // turn on pullup
  pinMode (ciRainDetectPin,INPUT);          // input from rain detection sensor
  digitalWrite (ciRainDetectPin,HIGH);      // turn on pullup  
  
  initLEDs();
  
  displayStatus(1);                         // Set system status to 1 - initializing

  // attach external interrupt pins to IRQ functions
  attachInterrupt(digitalPinToInterrupt(ciWindSpeedPin),wspeedIRQ,FALLING);
  attachInterrupt(digitalPinToInterrupt(ciRainGaugePin),rainIRQ,FALLING);
  attachInterrupt(digitalPinToInterrupt(ciRainDetectPin),rainDetectIRQ,CHANGE);

  // turn on interrupts
  interrupts();

  // Startup objects
  mqttClient.setServer(mqtt_server, 1883);
  mqttClient.setCallback(callback);

  //Test the cards and the connection to the server and return pass/no-pass
  Serial.println("Beginning boot up device tests...");
  
  //Serial.print("******** dht sensor: ");
  bDHTSensorError = initDHT();
  //Serial.println(bDHTSensorError); //Serial.println("----------------------------------------------------------------"); Serial.println("");
  
  //Serial.print("******** lux sensor: ");
  bLUXSensorError = initTSL2561();
  //Serial.println(bLUXSensorError); //Serial.println("----------------------------------------------------------------"); Serial.println("");
  if (bLUXSensorError){
    Wire.begin();
  }
  //Serial.print("******** ligntning sensor: ");
  bLightingSensorError = initMOD1016();
  //Serial.println(bLightingSensorError); //Serial.println("----------------------------------------------------------------"); Serial.println("");
  /*bBPSensorError = initBP();
  Serial.print("******** pressure sensor: ");
  Serial.println(bBPSensorError); Serial.println("----------------------------------------------------------------"); Serial.println(""); */
  
  //Serial.print("******** temp/humidity/pressure sensor: ");
  bTHPSensorError = initTHP();
  //Serial.println(bTHPSensorError); //Serial.println("----------------------------------------------------------------"); Serial.println(""); 
  
  //Serial.print("******** UV sensor: ");
  bUVSensorError = initVEML6075();
  //Serial.println(bUVSensorError); //Serial.println("----------------------------------------------------------------"); Serial.println("");
  
  //Serial.print("******** Realtime Clock: ");
  bRTCError = initRTC();
  Serial.println(bRTCError); //Serial.println("----------------------------------------------------------------"); Serial.println("");

  displayStatus(initEthernet());            // If Ethernet fails, hard stop bootup.

  // init wind speed interrupt global variables
  DateTime now = rtc.now();
  startupdtstamp = String(now.year(), DEC) + "/" + String(now.month(), DEC) + "/" + String(now.day(), DEC) + " " + String(now.hour(), DEC) + ":" + String(now.minute(), DEC)  + ":" + String(now.second(), DEC);
  gotwspeed = false; 
  windintcount = 0;
  zerotime = 0;
  rain = 0;  
  MOD1016IrqTriggered = 0;
  iLightningCount = 0;
  pollTimer = millis() + pollPeriod;
  dataRecordTimer = millis() + dataRecordPeriod;
  lastDataRecorded = millis();

  if (reconnect()){
    //Serial.println("");
  }
  
  displayStatus(2);


  //Serial.println("");
  Serial.println("boot complete");
}

void loop () {

  float fGreatestDirection = 0.0;

  handleClient();

  // If an interrupt occurred, handle it now
  if (gotwspeed == true){
    handleWindSpeed();
  }
  
  if ((millis() - zerotime) > 8000) {
    fWindSpeed = 0.00;
  }

  if(MOD1016IrqTriggered){
    handleLightning();
  }
  
  if (millis() > pollTimer) {

    // get wind direction
    fWindDirection = get_wind_direction();
    //Serial.println(((dataRecordTimer - millis())/1000));
    pollTimer = millis() + pollPeriod;
    getMemory();
    //Serial.println();
  }
  
  if (millis() > dataRecordTimer) {
    TransmitData(fGreatestDirection);
  }
  
  mqttClient.loop();
  
}
