/******************************************************************************/
/* Files to Include                                                           */
/******************************************************************************/

#include <xc.h>         /* XC8 General Include File */

#include <stdint.h>        /* For uint8_t definition */
#include <stdbool.h>       /* For true/false definition */

#include "system.h"        /* System funct/params, like osc/peripheral config */
#include "user.h"          /* User funct/params, such as InitApp */

// Function prototypes
void readclock( void );
void i2c_start( void );
void i2c_send( char data );
void i2c_repeat_start( void );
char i2c_read_byte_ack( void );
char i2c_read_byte_nack( void );
bool i2c_read_bit( void );
void i2c_stop( void );
void send2digit( char digit );
void config_MAX(char addr, char data );
void send2LEDs( void );
void sendbyte2MAX( char data );
void toggle_monitor_mode( void );
char monitor_mode( void );
void assign_hour ( unsigned char ahour );
void light_up( void );
void wr_clock( char addr, char data );
void update_clock_display( void );
void light_sensor( void );
void act_on_hour_sw( void );
void act_on_min_sw( void );
unsigned char read_hour_register ( void );

/******************************************************************************/
/* User Global Variable Declaration                                           */
/******************************************************************************/

/* i.e. uint8_t <variable_name>; */
char LED_addr[3];
char LED_data[3];
unsigned char ADC_result, old_ADC;
unsigned char hour, min, sec, oldmin, temphr;
char temp, index;
char L[9];
bool hoursw, minsw;
bool hoursw_1, minsw_1;
bool acted_on_hoursw, acted_on_minsw;
volatile char family_sw, oclock_sw, brightness_sw;
char old_family, old_oclock, old_brightness;

/******************************************************************************/
/* Main Program                                                               */
/******************************************************************************/
void main(void)
{
   /* Configure the oscillator for the device */
   ConfigureOscillator();

   /* Initialize I/O and Peripherals for application */
   InitApp();

   // Configure MAX7912 chips.
   // Set chips to normal operation (Display Test Mode off).
   config_MAX( 0x0F, 0 );
   // Set chips to "No Decode" mode.
   config_MAX( 0x09, 0 );
   // Set Scan Limit to scan 3 "digits."
   config_MAX( 0x0B, 2 );
   // Set Intensity to half-way on.
   config_MAX( 0x0A, 2 );
   // Set Shutdown to off. Turn on the LEDs!
   config_MAX( 0x0C, 1 );

   // If necessary, put the clock into 12 hour mode;
   temphr = read_hour_register();
   temp = temphr & 0x40;  // mask 12/24* bit
   if( temp == 0 )
   {
       temphr |= 0x40;     // set the 12/24* bit
       wr_clock( 0x02, temphr );
   }
   oldmin = 62;   // Force the first LED update. Later comparison will fail.

   //  Set the clock to a specifi time, if desired, here:
//   wr_clock( 0x02, 0x48 );
//   wr_clock( 0x01, 0x40 );
   __delay_ms(100);

   while(1)
   {

      readclock();    // Results deposited into the hour and min variables.

      if( oldmin != min )
      {
          oldmin = min;
          update_clock_display();
      }
      
      family_sw = RC5;  // sw up = low = do "family" text at 4 o'clock
      oclock_sw = RA5;  // sw up = "o'clock" ON at "quarter" and "half"
      brightness_sw = RA4;  // up = add +2 to brightness
          
      if( old_family != family_sw )
      {
          old_family = family_sw;
          update_clock_display();
      }
      if( old_oclock != oclock_sw )
      {
          old_oclock = oclock_sw;
          update_clock_display();
      }

      light_sensor();

      for( int i=0; i<= 10; i++)
      {      
          hoursw = RC3;
          if( !hoursw )   // hour switch IS pressed
          {
              if( (hoursw_1 == 1) && !acted_on_hoursw )
              {
                  act_on_hour_sw();   // Act on hour button press
                  acted_on_hoursw = 1;
              }
              else  // switch not pressed last time through.
              {
                  hoursw_1 = 1;  // debounce. must see it twice.
              }
          }
          else  // hour switch NOT pressed
          {
              hoursw_1 = 0;
              acted_on_hoursw = 0;
          }

          minsw = RC4;
          if( !minsw )   // minute switch IS pressed
          {
              if( (minsw_1 == 1) && !acted_on_minsw )
              {
                  act_on_min_sw();    // Act on minute button press
                  acted_on_minsw = 1;
              }
              else  // switch not pressed last time through.
              {
                  minsw_1 = 1;  // debounce. must see it twice.
              }
          }
          else  // minute switch NOT pressed
          {
              minsw_1 = 0;
              acted_on_minsw = 0;
          }
          __delay_ms(15);
      }  // End of "loop ten times"
   }  // End of "while(1)"
}  // End of "main"
//---------- Helper functions ------------
void act_on_hour_sw( void )
{
    readclock();
    hour += 1;
    if( hour >= 13 ) hour = 1;
    temp = 0x40;  // Set the 12/24* bit.
    if( hour > 9 )
    {
        temp |= 0x10;
        hour -= 10;
    }
    temp |= hour;
    wr_clock( 0x02, temp);   // Write it to the RTC chip.
    __delay_ms(5);
    readclock();
    update_clock_display();  // Send the new time to the clock display.
}
//----------------------------------------
void act_on_min_sw( void )
{
    readclock();
    // Move to the next 5-minute value: 0, 5, 10, 15, etc...
    temp = min / 5;
    temp += 1;
    min = temp * 5;
    if( min >= 60 ) min = 0;
    temp = 0;
    while( min > 9 )
    {
        temp += 1;
        min -= 10;
    }
    temp = temp << 4;
    temp |= min;
    wr_clock( 0x01, temp );
    __delay_ms(5);
    wr_clock( 0x00, 0x00 );  // Set seconds to 0 after setting minutes.
    __delay_ms(5);
    readclock();
    update_clock_display();
}
//----------------------------------------
void light_sensor( void )
{
    // Read the light sensor.
    // Low light levels = high ohms = low readings.
    // Low readings means set the LED brightness low.
    ADCON0bits.GO = 1; //start the ADC conversion
    while( ADCON0bits.GO == 1 ) {}; //wait for the conversion to end
    ADC_result = ADRESH >> 4;
    if( brightness_sw == 0 )
    {
        ADC_result += 2;
        if( ADC_result > 0x0F ) ADC_result = 0x0F;
    }
    if( old_ADC != ADC_result )
    {
        old_ADC = ADC_result;
        config_MAX( 0x0A, ADC_result );
    }
}
//----------------------------------------
void update_clock_display( void )
{
    // Before calling this, readclock() should have been called recently.
    for( index=0; index<= 8; index++ )
    {
        L[index] = 0;
    }

    if ( ( hour == 4 ) && ( min <= 2) && (family_sw == 0) )
    {
        // Clear display data, then light up "Family is four ever"
        LED_data[0] = 0;  LED_data[1] = 0x3D;  LED_data[2] = 0;
        send2digit( 1 );
        LED_data[0] = 0;  LED_data[1] = 0x81;  LED_data[2] = 0x07;
        send2digit( 2 );
        LED_data[0] = 0;  LED_data[1] = 0x40;  LED_data[2] = 0;
        send2digit( 3 );
    }
    else
    {
     // Turn on "IT" and "IS" lights.
     // Do not turn on "O'CLOCK" for "HALF" and "QUARTER" lights.
        L[0] |= IT;
        L[0] |= IS1;
        if( family_sw == 0 ) L[8] |= OCLOCK;  // S3 up = turn on O'CLOCK always.
     // If minutes between 0 and 4 hour light = HOUR.
        if( (min==0) || (min==1) || (min==2) || (min==3) || (min==4) )
        {
            L[8] |= OCLOCK;
            assign_hour( hour );
            light_up();
        }
     // If minutes between 5 and 34 (inclusive) hour light = HOUR.
     // Also, turn on the "PAST" light.
        if( (min>4) && (min<35) )
        {
            assign_hour( hour );
            L[2] |= PASTA;
            L[3] |= PASTB;
            if( (min>4) && (min<10) )
            {
                L[2] |= MYFIVE;
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            if( (min>9) && (min<15) )
            {
                L[0] |= TEN;
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            if( (min>14) && (min<20) )
            {
                L[0] |= QUARTERA;
                L[1] |= QUARTERB;
            }
            if( (min>19) && (min<30) )
            {
                L[1] |= TWENTY;
                if( (min>24) && (min<30) )
                {
                    L[2] |= MYFIVE;
                }
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            if( min>29 ) L[0] |= HALF;
               light_up();
        }
        else if( min>34 )
        {
         // We now know that minutes are between 35 and 59 (inclusive)
         // Turn on "TO" light
         // hour light = HOUR + 1.
         // If min between 34 and 39 turn on "TWENTY" "FIVE" etc...
            if( min<40 )
            {
                L[1] |= TWENTY;
                L[2] |= MYFIVE;
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            if( (min<45) && (min>39) )
            {
                L[1] |= TWENTY;
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            if( (min<50) && (min>44) )
            {
                L[0] |= QUARTERA;
                L[1] |= QUARTERB;
            }
            if( (min<55) && (min>49) )
            {
                L[0] |= TEN;
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            if( min>54 )
            {
                L[2] |= MYFIVE;
                L[2] |= MINUTES;
                L[8] |= OCLOCK;
            }
            L[3] |= TO;
            temp = hour + 1;
            if( temp == 13 ) temp = 1;
            assign_hour( temp );
            light_up();
        }
    }    
    return;
}
//----------------------------------------
void wr_clock( char addr, char data )
{
    i2c_start();
    i2c_send( 0xD0 );   // Write to the device.
    i2c_send( addr );   // Address of register to write to.
    i2c_send( data );   // Data that gets written.
    i2c_stop();
}
//----------------------------------------
void light_up( void )
{
    LED_data[0] = L[0];
    LED_data[1] = L[3];
    LED_data[2] = L[6];
    send2digit( 1 );
    LED_data[0] = L[1];
    LED_data[1] = L[4];
    LED_data[2] = L[7];
    send2digit( 2 );
    LED_data[0] = L[2];
    LED_data[1] = L[5];
    LED_data[2] = L[8];
    send2digit( 3 );
}
//----------------------------------------
void assign_hour ( unsigned char ahour )
{
    if( ahour == 1 )
    {
        L[3] |= ONEA;
        L[4] |= ONEB;
    }
    if( ahour == 2 ) L[4] |= TWO;
    if( ahour == 3 ) L[4] |= THREE;
    if( ahour == 4 )
    {
        L[4] = FOURA;
        L[5] = FOURB;
    }
    if( ahour == 5 ) L[5] |= FIVE;
    if( ahour == 6 ) L[5] |= SIX;
    if( ahour == 7 ) L[5] |= SEVEN;
    if( ahour == 8 ) L[6] |= EIGHT;
    if( ahour == 9 ) L[6] |= NINE;
    if( ahour == 10 ) L[6] |= MYTEN;
    if( ahour == 11 ) L[7] |= ELEVEN;
    if( (ahour == 12) || (ahour == 0) )
    {
        L[7] |= TWELVEA;
        L[8] |= TWELVEB;
    }
}
//----------------------------------------
unsigned char read_hour_register ( void )
{
    unsigned char hr;
    
    i2c_start();
    i2c_send( 0xD0 );   // Write to the device.
    i2c_send( 0x02 );   // Address of the hour register.
    i2c_repeat_start();
    i2c_send( 0xD1 );   // Read from device.
    hr = i2c_read_byte_nack();
    i2c_stop();
    return( hr );
}
//----------------------------------------
void readclock( void )
{
    // To read seconds, uncomment all of the commented code lines.
    i2c_start();
    i2c_send( 0xD0 );   // Write to the device.
//    i2c_send( 0x00 );   // Address of seconds register. Comment-out for no secs.
    i2c_send( 0x01 );   // Address of the minutes register.
    i2c_repeat_start();
    i2c_send( 0xD1 );   // Read from device.
//    sec = i2c_read_byte_ack();  // Comment-out for no secs.
    min = i2c_read_byte_ack();
    hour = i2c_read_byte_nack();
    i2c_stop();

//      temp = sec;
//      sec = sec & 0b00001111;    // mask "seconds"
//      temp = temp & 0b01110000;  // mask "10 seconds"
//      temp = temp >> 4;
//      sec = ( 10 * temp ) + sec;
    
    // Now store the hour and minutes as integers (chars).
    temp = min;
    min = min & 0b00001111;    // mask "minutes"
    temp = temp & 0b01110000;  // mask "10 minutes"
    temp = temp >> 4;
    min = ( 10 * temp ) + min;

    temp = hour;
    hour = hour & 0x0F;
    if ( ( temp & 0x10 ) == 0x10 ) hour = hour + 10;
    if ( hour == 0 ) hour = 12;
    return;
}
//------------------------------------------
void i2c_stop( void )
{
    TRISC1 = PIN_IS_OUT;    // Set SDA as an output.
    SDA = 0;
    __delay_us(2);
    SCL = 1;
    __delay_us(2);
    SDA = 1;
    __delay_us(2);
}
//------------------------------------------
char i2c_read_byte_ack( void )
{
    char i, byte;

    TRISC1 = PIN_IS_IN;    // Set SDA as an input.
    __delay_us(2);
    for( i = 0; i < 8; i++ )
    {
        byte = (byte << 1) | i2c_read_bit();
    }
    // Now send the ACK bit to the clock chip.
    TRISC1 = PIN_IS_OUT;    // Set SDA as an output.
    SDA = 0;  // Low = ACK
    __delay_us(2);
    SCL = 1;
    __delay_us(2);
    SCL = 0;
    __delay_us(2);
    TRISC1 = PIN_IS_IN;    // Set SDA as an input.
    __delay_us(2);
    return byte;
}
//------------------------------------------
char i2c_read_byte_nack( void )
{
    char i, byte;

    TRISC1 = PIN_IS_IN;    // Set SDA as an input.
    __delay_us(2);
    for( i = 0; i < 8; i++ )
    {
        byte = (byte << 1) | i2c_read_bit();
    }
    // Now send the ACK bit to the clock chip.
    TRISC1 = PIN_IS_OUT;    // Set SDA as an output.
    SDA = 1;  // High = NACK
    __delay_us(2);
    SCL = 1;
    __delay_us(2);
    SCL = 0;
    __delay_us(2);
    TRISC1 = PIN_IS_IN;    // Set SDA as an input.
    __delay_us(2);
    return byte;
}
//------------------------------------------
bool i2c_read_bit( void )
{
    // Presumes that the SDA pin has already been set as an INPUT pin.
    bool read_bit;

    SCL = 1;
    __delay_us(2);
    read_bit = SDA;
    __delay_us(2);
    SCL = 0;
    __delay_us(2);
    return read_bit;
}
//------------------------------------------
void i2c_send( char data )
{
    // Enter with SDA and SCL low, i.e., after the i2c_start().
    // Exits with SDA and SCL low.
    unsigned char mask, i;

    mask = 0x80;
    for( i=0; i<8; i++)
    {
       SDA = ( ( (data & mask) == 0) ? 0 : 1 );
       __delay_us(1);
       SCL = 1;
       __delay_us(2);
       SCL = 0;
       __delay_us(2);
       mask = mask >> 1;
    }
    TRISC1 = PIN_IS_IN;    // Set SDA as an input.
    __delay_us(2);
    SCL = 1;
    __delay_us(2);
    // Ignore state of SDA, i.e., do not read the ACK bit.
    SCL = 0;
    __delay_us(2);
    TRISC1 = PIN_IS_OUT;
    SDA = 0;
    return;
}
//------------------------------------------
void i2c_repeat_start( void )
{
    SDA = 1;
    __delay_us(2);
    SCL = 1;
    __delay_us(2);
    SDA = 0;
    __delay_us(2);
    SCL = 0;
    __delay_us(2);
}
//-------------------------------------------
void i2c_start( void )
{
    SDA = 0;
    __delay_us(2);
    SCL = 0;
    __delay_us(3);
}
//------------------------------------------
void send2digit( char digit )
{
    // Call after setting up the LED_data[3] array.
    // Useful for sending various data to the same register of all 3 MAX chips.
    LED_addr[0] = digit;
    LED_addr[1] = digit;
    LED_addr[2] = digit;
    send2LEDs();
}
//----------------------------------------
void config_MAX (char addr, char data )
{
    // Sends the same data to the same register of all 3 MAX chips.
    // Useful for configuring the chips, setting brightness, etc...
    LED_data[0] = data;
    LED_data[1] = data;
    LED_data[2] = data;
    LED_addr[0] = addr;
    LED_addr[1] = addr;
    LED_addr[2] = addr;
    send2LEDs();
    return;
}
//----------------------------------------
void send2LEDs( void )
{
    CLK_PIN = 0;
    __delay_us(0);
    LD_PIN = 0;
    __delay_us(0);
    sendbyte2MAX( LED_addr[2] );
    sendbyte2MAX( LED_data[2] );
    sendbyte2MAX( LED_addr[1] );
    sendbyte2MAX( LED_data[1] );
    sendbyte2MAX( LED_addr[0] );
    sendbyte2MAX( LED_data[0] );
    LD_PIN = 1;
    __delay_us(1);
    CLK_PIN = 0;
    return;
}
//-----------------------------------------
void sendbyte2MAX( char data )
{
    unsigned char mask, i;
    // Enter with LOAD pin low. Exits with LOAD pin unchanged.
    // Enter with CLK_PIN low, the first time. Exits with CLK_PIN high.

    mask = 0x80;

    for( i=0; i<8; i++)
    {
       CLK_PIN = 0;
       __delay_us(1);
       DATA_PIN = ( ( (data & mask) == 0) ? 0 : 1 );
       __delay_us(0);
       CLK_PIN = 1;
       __delay_us(1);
       mask = mask >> 1;
    }

    return;
}
