/* 
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

Title : WiFly.cpp
Author : http://www.instructables.com/member/RandomMatrix/

Description : A wrapper for initialising the WiFly GSX 802.11G Module, communicating with it via its SPI-to-UART chip 
              connecting to a wifi network and making simple GET HttpWebRequests.

 The WiFly Shield equips your Arduino the ability to connect to 802.11b/g wireless networks. 
 The featured components of the shield are a Roving Network's RN-131G wireless module and an SC16IS750 SPI-to-UART chip. 
 The SPI-to-UART bridge is used to allow for faster transmission speed and to free up the Arduino's UART.

 code and comments based on the following sources:
    http://www.arduino.cc/en/Tutorial/SPIEEPROM
    http://www.lammertbies.nl/comm/info/serial-uart.html
    http://www.tinyclr.com/downloads/Shield/FEZ_Shields_WiFly.cs
    http://www.sparkfun.com/commerce/tutorial_info.php?tutorials_id=158
    http://www.sparkfun.com/datasheets/Components/SMD/sc16is750.pdf
    http://www.sparkfun.com/datasheets/Wireless/WiFi/WiFlyGSX-um.pdf
    http://www.sparkfun.com/datasheets/Wireless/WiFi/rn-131G-ds.pdf
    http://www.societyofrobots.com/microcontroller_uart.shtml

*** Warning ***
The code presented here is in no way a library for the WiFly GSX 802.11G Module.
It does what I need it to do, but writing a full library is beyond the intended scope of the project.

Use at your own risk.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
*/


#include "WiFly.h"
#include "HtmlParser.h"
#include "WProgram.h"
#include <string.h>

// SC16IS750 Register Definitions
// see http://www.lammertbies.nl/comm/info/serial-uart.html
#define THR        0x00 << 3  // THR, Transmitter holding register (WO)
#define RHR        0x00 << 3  // RHR, Receiver holding register (RO)
#define IER        0x01 << 3  // IER, interrupt enable register (R/W)
#define FCR        0x02 << 3  // FCR, FIFO control  (WO)
#define IIR        0x02 << 3  // IIR, interrupt identification register  (RO)
#define LCR        0x03 << 3  // LCR, line control (R/W)
#define MCR        0x04 << 3  // MCR, modem control (R/W)
#define LSR        0x05 << 3  // LSR, line status register (RO)
#define MSR        0x06 << 3  // MSR, modem status register (RO)
#define SPR        0x07 << 3  // SPR, scratchpad register (R/W)

// see http://www.sparkfun.com/datasheets/Components/SMD/sc16is750.pdf
#define TXFIFO     0x08 << 3  // Transmit FIFO Level Register
#define RXFIFO     0x09 << 3  // Receive FIFO Level Register
#define DLAB       0x80 << 3
#define IODIR      0x0A << 3  // I/O pin Direction Register
#define IOSTATE    0x0B << 3  // I/O pin States Register
#define IOINTMSK   0x0C << 3  // I/O Interrupt Enable Register?
#define IOCTRL     0x0E << 3  // I/O pins Control Register
#define EFCR       0x0F << 3  // Extra Features Register

#define DLL        0x00 << 3 // divisor latch LSB
#define DLM        0x01 << 3 // divisor latch MSB?
#define EFR        0x02 << 3 // Enhanced Feature Register
#define XON1       0x04 << 3  
#define XON2       0x05 << 3
#define XOFF1      0x06 << 3
#define XOFF2      0x07 << 3

// SPI pin definitions
#define CS         10 // Slave Select pin - allocated on each device which the master can use to enable and disable 
                      // specific devices and avoid false transmissions due to line noise. 
#define MOSI       11 // Master In Slave Out (MISO) - The Slave line for sending data to the master, 
#define MISO       12 // Master Out Slave In (MOSI) - The Master line for sending data to the peripherals, 
#define SCK        13 // Serial Clock (SCK) - The clock pulses which synchronize data transmission generated by the master

#define ASSOCIATE_TIMEOUT 5000

// These times are taken from Boot-up Timing Values from the datasheet, 
// http://www.sparkfun.com/datasheets/Wireless/WiFi/WiFlyGSX-um.pdf
#define DEFAULT_TIME_TO_READY (600)
#define DEFAULT_TIME_TO_JOIN (800)
#define DEFAULT_TIME_TO_AUTHENTICATE (300)
#define DEFAULT_TIME_TO_WAIT (200)
#define DEFAULT_TIME_UNTIL_TIMEOUT (5000)

#define COMM_OPEN "*OPEN*"
#define COMM_CLOSE "*CLOS*"
#define AOK "AOK"
#define ERR "ERR"

//The THR is actually a 64-byte FIFO accordning to http://www.sparkfun.com/datasheets/Components/SMD/sc16is750.pdf
#define THR_MAX_BYTES (64)

// SC16IS750 communication parameters
struct SPI_UART_cfg
{
  char DivL;
  char DivM;
  char DataFormat;
  char Flow;
};

struct SPI_UART_cfg SPI_Uart_config = {0x50,0x00,0x03,0x10};

// Wifi parameters
char auth_level[] = "3";
char port_listen[] = "80";

WiFly::WiFly(const char* ssid, 
             const char* auth_phrase,
             int sleepTime,
             HardwareSerial& print):
m_network(ssid),
m_password(auth_phrase),
m_sleepTime(sleepTime),
m_printer(&print)
{
}

bool WiFly::Reset()
{
  // SPI pin initialization
  pinMode(MOSI, OUTPUT);
  pinMode(MISO, INPUT);
  pinMode(SCK,OUTPUT);
  pinMode(CS,OUTPUT);

  deselect(); //disable device 

  // see http://www.arduino.cc/en/Tutorial/SPIEEPROM
  // http://www.sparkfun.com/datasheets/Components/SMD/sc16is750.pdf
  /* 
  Now we set the SPI Control register (SPCR) to the binary value 01010000. 
  In the control register each bit sets a different functionality.
  The eighth bit disables the SPI interrupt, 
  the seventh bit enables the SPI, 
  the sixth bit chooses transmission with the most significant bit going first, 
  the fifth bit puts the Arduino in Master mode, 
  the fourth bit sets the data clock idle when it is low, 
  the third bit sets the SPI to sample data on the rising edge of the data clock, 
  and the second and first bits set the speed of the SPI to system speed / 4 (the fastest).
  After setting our control register up we read the SPI status register (SPSR) and data register (SPDR)
  in to the junk clr variable to clear out any spurious data from past runs:*/

  SPCR = (1<<SPE)|(1<<MSTR)|(1<<SPR1)|(1<<SPR0);
  
  char clr = SPSR;
  clr = SPDR;
  delay(10); 

  // Data registers simply hold bytes. For example, the SPI data register (SPDR) 
  // holds the byte which is about to be shifted out the MOSI line, and the data which has just been shifted in the MISO line.

  m_printer->println("WiFly Shield");
  m_printer->println("initializing SC16IS750 bridge..."); 

  SPI_UART_Init();

  // Test SC16IS750 communication
  const bool ok = TestSPI_UART_Bridge();
  

  if(!ok)
  { 
    m_printer->println("Could not initialize SC16IS750 bridge"); 
    return false;
  }

  m_printer->println("success!"); 
  m_printer->println(""); 

  AutoConnect();

  return true;
}


// Test if the SPI<->UART bridge has been set up correctly by writing a test 
// character via SPI and reading it back.
// returns true if success

bool WiFly::TestSPI_UART_Bridge()
{
  // Perform read/write test to check if SPI<->UART bridge is working

  // write a character to the scratchpad register.
  WriteByteToRegister(SPR, 0x55);

  char data = ReadCharFromWiFly(SPR);

  if(data == 0x55)
  { 
    return true; 
  }
  else
  { 
    m_printer->println("Failed to init SPI<->UART chip");
    return false; 
  }
}

// A series of register writes to initialize the SC16IS750 SPI-UART bridge chip
// see http://www.tinyclr.com/downloads/Shield/FEZ_Shields_WiFly.cs

void WiFly::SPI_UART_Init(void)
{
  WriteByteToRegister(LCR,0x80); // 0x80 to program baudrate
  WriteByteToRegister(DLL,SPI_Uart_config.DivL); //0x50 = 9600 with Xtal = 12.288MHz
  WriteByteToRegister(DLM,SPI_Uart_config.DivM); 

  WriteByteToRegister(LCR, 0xBF); // access EFR register
  WriteByteToRegister(EFR, SPI_Uart_config.Flow); // enable enhanced registers
  WriteByteToRegister(LCR, SPI_Uart_config.DataFormat); // 8 data bit, 1 stop bit, no parity
  WriteByteToRegister(FCR, 0x06); // reset TXFIFO, reset RXFIFO, non FIFO mode
  WriteByteToRegister(FCR, 0x01); // enable FIFO mode
}

/*
  Send the correct commands to connect to a wireless network using the parameters used on construction
*/
void WiFly::AutoConnect()
{
  m_printer->println("connecting to network..."); 

  delay(DEFAULT_TIME_TO_READY);

  FlushRX();

  // Enter command mode
  EnterCommandMode();

  // Reboot to get device into known state
  WriteToWiFlyCR("reboot");
  WaitUntilReceived("*Reboot*");
  WaitUntilReceived("*READY*");
  
  FlushRX();

  // Enter command mode
  EnterCommandMode();

  // turn off auto joining
  WriteToWiFlyCR("set wlan join 0");
  WaitUntilReceived(AOK, ERR);

  // Set authentication level to <auth_level>
  WriteToWiFly("set w a ");
  WriteToWiFlyCR(auth_level); 
  WaitUntilReceived(AOK, ERR);

  // Set authentication phrase to <auth_phrase>
  WriteToWiFly("set w p ");
  WriteToWiFlyCR(m_password);
  WaitUntilReceived(AOK, ERR);

  // Set localport to <port_listen>
  WriteToWiFly("set i l ");
  WriteToWiFlyCR(port_listen);
  WaitUntilReceived(AOK, ERR);

  // Deactivate remote connection automatic message
  WriteToWiFlyCR("set comm remote 0");
  WaitUntilReceived(AOK, ERR);

  // Join wireless network <ssid>
  WriteToWiFly("join ");
  WriteToWiFlyCR(m_network);  
  delay(DEFAULT_TIME_TO_JOIN);

  bool ok = WaitUntilReceived("IP=");
  delay(DEFAULT_TIME_TO_WAIT);

  FlushRX();
  
  if(ok == false)
  {
    m_printer->print("Failed to associate with '");
    m_printer->print(m_network);
    m_printer->println("'\n\rRetrying...");
    FlushRX();
    AutoConnect();
  }
  else
  {
    m_printer->println("connected!");
    ExitCommandMode();
  }

  // TODO save this configuration
}

/*
  Parameters: The server to telnet into, the get command that needs to be sent, a custom HtmlParser that
  is called every time a character is received. The parser is responsible for processing the HTML
  that is returned.

  Http is just TCP/IP on port 80

  "Open <ipaddress> 80"

  or using DNS,
  "Open www.google.com 80"

  Twitter requires more of the Http protocol than other sites.
  For example, the "Host" field is required in case there's more than one
  domain name mapped to the server's IP address so it can tell which
  website you actually want.

  "GET / HTTP/1.1\n"
  "Host: server\r\n"
  "\r\n"

*/
bool WiFly::HttpWebRequest(const char* server, const char* getCommand, HtmlParser* parser)
{
  m_printer->println(getCommand);
  FlushRX(); 
  FlushRX(); 

  // Enter command mode
  EnterCommandMode();
  FlushRX(); 

  // open a TCP connection, port 80 for HTTP
  WriteToWiFly("open ");
  WriteToWiFly(server);
  WriteToWiFlyCR(" 80");

  bool openOK = WaitUntilReceived(COMM_OPEN);

  if (openOK == false)
  {
    m_printer->println("open port failed!");
    delay(1000);
    WriteToWiFlyCR("close");
    WaitUntilReceived(COMM_CLOSE);

    ExitCommandMode();
    return false;
  }

  // eg. "GET /search.json?q=foo HTTP/1.1\r\n"
  WriteToWiFlyCRLF(getCommand);

  // eg. "Host: search.twitter.com\r\n"
  WriteToWiFly("Host: ");
  WriteToWiFlyCRLF(server);

  // "\r\n"
  WriteToWiFlyCRLF("");

  // now wait for the response

  int timeOut = 0;
  bool ok = false;

  while(timeOut < 5000)// timeout after 5 seconds
  {
    if((ReadCharFromWiFly(LSR) & 0x01))
    {
      char incoming_data = ReadCharFromWiFly(RHR);
      m_printer->print(incoming_data,BYTE);

      bool done = parser->Parse(incoming_data);
      if (done)
      {
        ok = true;
        break;
      }

      timeOut = 0; //reset the timeout
    }
    else
    {
      delay(1);
      timeOut++;
    }
  }

  FlushRX(); 

  // disconnect TCP connection.
  WriteToWiFlyCR("close");
  WaitUntilReceived(COMM_CLOSE);

  ExitCommandMode();

  return ok;
}

/*
  send the WiFly module to sleep to put into "Ultra-low power" mode (4uA sleep)
*/
void WiFly::Sleep()
{
  WriteToWiFlyCR("set sys sleep 1");
  
  WriteToWiFly("set sys wake ");
  char buffer[10];
  itoa(m_sleepTime, buffer, 10);
  WriteToWiFlyCR(buffer);
}

void WiFly::select()
{
  digitalWrite(CS,LOW);
}

void WiFly::deselect()
{
  digitalWrite(CS,HIGH);
}

// send data to the Wifly and append "\n"
void WiFly::WriteToWiFlyCR(const char* data)
{
  WriteToWiFly(data);
  WriteByteToRegister(THR, 0x0d); // CR
}

// send data to the Wifly and append "\r\n"
void WiFly::WriteToWiFlyCRLF(const char* data)
{
  WriteToWiFly(data);
  WriteByteToRegister(THR, 0x0d); // CR
  WriteByteToRegister(THR, 0x0a); // LF
}

/*
  wrapper for WriteChunk
  see below
*/
void WiFly::WriteToWiFly(const char* data)
{
  int length = strlen(data);

  while (length > 0)
  {
     int written = WriteChunk(data, length);
     data += written;
     length -= written;
  }
}

/*
  The transmitter section consists of the Transmit Holding Register (THR) and the Transmit
  Shift Register (TSR). The THR is actually a 64-byte FIFO. The THR receives data and
  shifts it into the TSR, where it is converted to serial data and moved out on the TX pin. If
  the FIFO is disabled, the FIFO is still used to store the byte. Characters are lost if overflow
  occurs.
*/
int WiFly::WriteChunk(const char* data, int length)
{
  select();
  spi_transfer(THR);

  length = min(length, THR_MAX_BYTES);

  for(int i = 0; i < length; ++i)
  {
    spi_transfer(data[i]);
    delay(1); // This seems to decrease error characters
  }

  deselect();

  return length;
}

// Write <data> to SC16IS750 register at <address>
void WiFly::WriteByteToRegister(char address, char data)
{
  select();
  spi_transfer(address);
  spi_transfer(data);
  deselect();
}


/*
  The ReadCharFromWiFly function allows us to read data back out of the WiFly. 
*/
char WiFly::ReadCharFromWiFly(char address)
{
  address = (address | 0x80);

  select();
  spi_transfer(address);
  char data = spi_transfer(0xFF);
  deselect();

  return data;  
}

// Flush characters from SC16IS750
void WiFly::FlushRX()
{
  while((ReadCharFromWiFly(LSR) & 0x01))
  {
    char _flushed = ReadCharFromWiFly(RHR);
  }  
}

/*
  The spi_transfer function loads the output data into the data transmission register, 
  thus starting the SPI transmission. It polls a bit to the SPI Status register (SPSR) 
  to detect when the transmission is complete using a bit mask, SPIF. 
  It then returns any data that has been shifted in to the data register
*/
char WiFly::spi_transfer(volatile char data)
{
  SPDR = data;                    // Start the transmission
  while (!(SPSR & (1<<SPIF)))     // Wait for the end of the transmission
  {
  };
  return SPDR;                    // return the received byte
}


/*
  Continue reading data from the wifly until either we receive the OKresults or the errorResult or we timeout
  There is some capability to ignore noise by just requiring that the sequence of characters are received in the 
  correct order, but with possibility other characters arriving too.
  returns true if success
*/
bool WiFly::WaitUntilReceived(const char* OKresult, const char* errorResult /* = 0 */)
{
  // find the length of the strings
  int OKLength = strlen(OKresult);
  int errorLength = strlen(errorResult);

  // keep track of when to timeout
  int timeOut = 0;

  // keep track of how many correct characters have been received
  int OKCounter = 0;
  int errorCounter = 0;

  while(timeOut < DEFAULT_TIME_UNTIL_TIMEOUT)// timeout after 5 seconds
  {
    // read from the wifil
    if((ReadCharFromWiFly(LSR) & 0x01))
    {
      char incoming = ReadCharFromWiFly(RHR);

      if (incoming == OKresult[OKCounter])
      {
        // found the next character in the success string
        OKCounter++;
      }

      if (OKCounter == OKLength)
      {
        // done
        return true;
      }

      if (errorResult != 0)
      {
        if (incoming == errorResult[errorCounter])
        {
          // found the next character in the fail string
          errorCounter++;
        }

        if (errorCounter == errorLength)
        {
          m_printer->print("warning: found ");
          m_printer->println(errorResult);
          return false;
        }
      }
    }
    else
    {
      delay(1);
      timeOut++;
    }
  }

  m_printer->print("warning: failed to find ");
  m_printer->println(OKresult);
        
  return false;
}


/*
  Enter command mode by sending: $$$ 
  Characters are passed until this exact sequence is seen. If any bytes are seen before these chars, or
  after these chars, in a 1 second window, command mode will not be entered and these bytes will be passed 
  on to other side.
*/
void WiFly::EnterCommandMode()
{
  FlushRX();
  delay(1000); // wait 1s as instructed above
  m_printer->println("Entering command mode.");
  WriteToWiFly("$$$");
  WaitUntilReceived("CMD");
}


/*
  exit command mode 
  send the "exit" command and await the confirmation result "EXIT"
*/
void WiFly::ExitCommandMode()
{
  WriteToWiFlyCR("exit");
  WaitUntilReceived("EXIT");
}

