;;
;; rc_servo_adapter - Copyright (C) 2014 Ian Davis
;;
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
;;

;;
;; rc_servo_adapeter perform the following...
;;   1) Wait for rising edge (keep code loop same length as generate loop)
;;   2) Measure inputs until all zero
;;   3) Perform noise filter to remove single tick movements.
;;   4) Validate measurements
;;   5) Perform update/scaling/mods on measured values
;;   6) Serially transmit samples on TXSERIAL pin
;;   7) If SWITCH servo input asserted, revert to previous servo values.
;;   8) Generate outputs on local pins.
;;   9) Repeat
;;

#define TXCHIP          ;; tank base - sample receiver, generate bitstream
;;#define RXCHIP        ;; tank turrent - sample bitstream

;;
;;================================================================================================
;;
;; Inputs:
;;   input0 = PORTA,5 = Pin 2 = Receiver AUX  (right switch)
;;   input1 = PORTA,4 = Pin 3 = Receiver GEAR (left switch)
;;   input2 = PORTA,3 = Pin 4 = Receiver RUD  (left horz)
;;   input3 = PORTC,5 = Pin 5 = Receiver ELE  (right vert)
;;   input4 = PORTC,4 = Pin 6 = Receiver AILE (right horz)
;;   input5 = PORTC,3 = Pin 7 = Receiver THR  (left vert)
;;
;; Outputs:
;;   output0/led0 = PORTA,0
;;   output1/led1 = PORTA,1
;;   output2/led2 = PORTA,2
;;   output3/led3 = PORTC,0
;;   output4/led4 = PORTC,1
;;   output5/led5 = PORTC,2
;;
;;================================================================================================
;;


;;
;;================================================================================================
;;
;; SWITCH specifies input to enable switch mode.   In switch mode, either SERVO's or LED's are updated
;;   (one or the other).  If switch input is above threshold, LED's update.  Otherwise servos update.
;; 
;; Example:
;;    #define SWITCH _input0   ;; if transmitter right switch on, lock servo outputs & update LED output's instead.
;;
;;================================================================================================
;;


;;
;;================================================================================================
;;
;; DEFINES for Servo outputs:
;;   OUTPUT* input         Output tracks input (for servos)
;;   IDLECENTER* tolerance Inhibit pulse outputs if servo within center zone (center +/- tolerance) & 64 loops elapse.
;;   IDLEINPUT* outmask    Inhibit pulse on specified output(s) if input hasent moved within a second.
;;   MIN*_PULSE value      Scales servo output range if input less than default CENTER_PULSE.
;;                           Used to change servo ranges.
;;   MAX*_PULSE value      Scales servo output range if input more than default CENTER_PULSE.
;;                           Used to change servo ranges.
;;   CENTER*_PULSE value   New servo center point.
;;   DEFAULT* value        Force servo to default value during powerup.  Default is CENTER_PULSE if not specified.
;;   DELTAINPUT* delay     Update sampled "input" value using delta increments, ever
;;
;; Note:
;;   IDLECENTER* and IDLEINPUT* defines cannot be combined if the same value.  
;;   ie: IDLECENTER0 and IDLEINPUT0 cannot be combined.
;;
;; Example:
;;   #define OUTPUT0   _input2  ;; output0 = input2 = r/c left horz (gun elevation)
;;   #define IDLECENTER0 8      ;; Don't output pulses when OUTPUT0 in center zone 
;;   #define DEFAULT2 80        ;;   Force "_input2" to measurement of 80 during powerup
;;   #define MIN0_PULSE 90      ;; Reduce movement range of vertical turret servo.
;;   #define MAX0_PULSE 122
;;   #define CENTER0_PULSE 102
;; 
;;================================================================================================
;;


;;
;;================================================================================================
;;
;; DEFINES for LED/Relay/Solenoid outputs:
;;   LED* input            Turn output on, when input moved above up threshold.  Off when below down threshold. 
;;   TOGGLEUP* input       Toggle output on/off when input moved above up threshold
;;   TOGGLEDN* input       Toggle output on/off when input moved below down threshold 
;;   PULSEUP* input,delay  Pulse output for "delay" 20msec ticks when input moved above up threshold
;;   PULSEDN* input,delay  Pulse output for "delay" 20msec ticks when input moved below down threshold 
;;
;; LED*, TOGGLEUP*, TOGGLEDN*, PULSEUP*, PULSEDN* defines cannot be combined on a single output.
;;   ie: cannot use LED4 and TOGGLEUP4 at the same time.
;;
;; Example:
;;   #define OUTPUT0   _input3  ; Output0 follows input3
;;   #define TOGGLEUP1 _input2  ; Output1 toggles when input2 moved above ON threshold
;;   #define OUTPUT2   _input1  ; Output2 follows input1 (multiple outputs can follow same input)
;;   #define OUTPUT3   _input1  ; Output3 follows input1 (   "        "     "    "     "     "  )
;;   #define TOGGLEDN4 _input0  ; Output4 toggles when input0 moved below OFF threshold
;;   #define LED5      _input1  ; Output5 turns on LED if input1>on_threshold, and OFF if input1<off_threshold.
;;
;;================================================================================================
;;


	radix	dec
	processor 16lf1825
#include p16lf1825.inc
#include macros.inc

	__config _CONFIG1, (_FOSC_INTOSC & _MCLRE_OFF & _WDTE_OFF & _PWRTE_OFF & _CP_OFF & _CPD_OFF & _BOREN_OFF & _CLKOUTEN_OFF & _IESO_OFF & _FCMEN_OFF)
	__config _CONFIG2, (_WRT_OFF & _PLLEN_OFF & _STVREN_OFF & _BORV_LO)

;;#define USE_UART              ;; debug


#define OUTMASK0 0x10
#define OUTMASK1 0x20
#define OUTMASK2 0x40
#define OUTMASK3 0x01
#define OUTMASK4 0x02
#define OUTMASK5 0x04


;;
;; rx_adapter_base header layout (top-view):
;;   TX1  TX3  TX5
;;   TX0  TX2  TX4
;;
#ifdef TXCHIP
#define SWITCH _input0          ;; if transmitter right switch on, lock servo outputs & update LED output's instead.

#define OUTPUT5 _input5		;; output5 = input5 = r/c left vert = left motor
#define MIN5_PULSE 30  
#define MAX5_PULSE 150 
#define IDLEINPUT5 OUTMASK5	;; Don't output pulses if input5 hasent changed within last second
#define DEFAULT5 106

#define OUTPUT4 _input3		;; output4 = input3 = r/c right vert = right motor
#define MIN4_PULSE 50  
#define MAX4_PULSE 172 
#define IDLEINPUT3 OUTMASK4 	;; Don't output pulses if input3 hasent changed within last second

#define TOGGLEDN3 _input5	;; Treat PORTA,1 as LED driver.  Toggle if "left vert" below down threshold.
#define TOGGLEUP2 _input5	;; Treat PORTA,2 as LED driver.  Toggle if "left vert" above up threshold.
#define TXSERIAL 		;; output1 transmits state of all servo inputs.

#define OUTPUT0 _input1		;; output0 = input1 = r/c left switch = whistle
#define DEFAULT1 83  		;; Force "_input1" to measurement of 80 during powerup
#define MIN0_PULSE 83
#define MAX0_PULSE 120
#define CENTER0_PULSE 100
#endif


;;
;; rx_adapter_turret header layout (top-view):
;;   RX1  RX3  RX5
;;   RX0  RX2  RX4
;;
;; RX0 Output = Vertical Gun Elevation Servo
;; RX1 Output = Horizontal Turret Rotate Servo
;; RX2 Output = Sound Trigger
;; RX3 Output = Search Light
;; RX4 Output = Gun LED
;; RX5 Output = Solenoid
;;
#ifdef RXCHIP
#define RXSERIAL                ;; Obtains servo positions from serial input on "PORTA,5" instead of measuring.
#define SWITCH    _input0       ;; Lock servo outputs if SWITCH input above threshold & update LED outputs instead.

#define OUTPUT0   _input2       ;; output0 = input2 = r/c left horz (gun elevation)
#define MIN0_PULSE 90           ;; Reduce movement range of vertical turret servo.
#define MAX0_PULSE 122
#define CENTER0_PULSE 102
#define DELTAINPUT2 6           ;; Delta adjust input2 every specified number of 20msec loops

#define OUTPUT1   _input4       ;; output1 = input4 = r/c right horz (turret rotate)
#define IDLECENTER1 8           ;; Don't output pulses when OUTPUT1 in center zone 

#define PULSEDN2  _input3,50    ;; output2 = Pulse output if "right vert" below down threshold.  Pulse = 1sec
#define TOGGLEUP3 _input3       ;; output3 = LED driver.  Toggle if "right vert" above ON threshold.
#define PULSEDN4  _input3,25    ;; output4 = Pulse output if "right vert" below down threshold.  Pulse = 1/2sec
#define PULSEDN5  _input3,12    ;; output5 = Pulse output if "right vert" below down threshold.  Pulse = 1/4sec
#endif


;;
;; Thresholds...
;;
#define ON_THRESHOLD_HI 125     ;; Thresholds for LED toggles
#define ON_THRESHOLD_LO 120
#define OFF_THRESHOLD_HI 90
#define OFF_THRESHOLD_LO 85
#define IDLE_THRESHOLD_HI 109   ;; Thresholds for IDLECENTER pulse disable 
#define IDLE_THRESHOLD_LO 104


;;
;; Nominal max/min range of inputs...
;;
#define MIN_PULSE 64            ;; min pulse = 0.9msecs
#define MAX_PULSE 148           ;; max pulse = 2.1msecs
#define CENTER_PULSE 0x70       ;; 112


#ifndef CENTER0_PULSE
#define CENTER0_PULSE CENTER_PULSE
#endif
#ifndef CENTER1_PULSE
#define CENTER1_PULSE CENTER_PULSE
#endif
#ifndef CENTER2_PULSE
#define CENTER2_PULSE CENTER_PULSE
#endif
#ifndef CENTER3_PULSE
#define CENTER3_PULSE CENTER_PULSE
#endif
#ifndef CENTER4_PULSE
#define CENTER4_PULSE CENTER_PULSE
#endif
#ifndef CENTER5_PULSE
#define CENTER5_PULSE CENTER_PULSE
#endif

#ifndef DEFAULT0                ;; Default values to output during power up
#define DEFAULT0 CENTER0_PULSE
#endif
#ifndef DEFAULT1
#define DEFAULT1 CENTER1_PULSE
#endif
#ifndef DEFAULT2
#define DEFAULT2 CENTER2_PULSE
#endif
#ifndef DEFAULT3
#define DEFAULT3 CENTER3_PULSE
#endif
#ifndef DEFAULT4
#define DEFAULT4 CENTER4_PULSE
#endif
#ifndef DEFAULT5
#define DEFAULT5 CENTER5_PULSE
#endif


;;
;; Standard register/flag defines...
;;
indf	equ	0
indf0	equ	0
pc	equ	2
pcl	equ	2
status	equ	3
c	equ	1
z	equ	0


;; Assign register addresses to variables...

_leda	    equ 0x20
_ledc	    equ 0x21
_pulse      equ 0x22
_startcount equ 0x23
_switch     equ 0x24
_bitcount   equ 0x25

_timer0     equ 0x30            ;Timers used for controlling PULSE outputs
_timer1     equ 0x31
_timer2     equ 0x32
_timer3     equ 0x33
_timer4     equ 0x34
_timer5     equ 0x35

_delta0     equ 0x38            ;Timers used for delta inputs.  When reach zero, inputs are adjusted.
_delta1     equ 0x39
_delta2     equ 0x3A
_delta3     equ 0x3B
_delta4     equ 0x3C
_delta5     equ 0x3D

_cvalue0    equ 0x40            ;Detected center values
_cvalue1    equ 0x41
_cvalue2    equ 0x42
_cvalue3    equ 0x43
_cvalue4    equ 0x44
_cvalue5    equ 0x45

_ccount0    equ 0x48            ;Number of back-to-back samples of unchanged input
_ccount1    equ 0x49
_ccount2    equ 0x4A            ;If input sample unchanged & count>=0x20, then input is idle.
_ccount3    equ 0x4B            ;If IDLEINPUT selected, then output pulses are masked.
_ccount4    equ 0x4C
_ccount5    equ 0x4D 

_last0	    equ 0x50            ;Last four samples captures per input.  
_last1	    equ 0x54            ;Used for input jitter filtering.
_last2	    equ 0x58
_last3	    equ 0x5C
_last4	    equ 0x60
_last5	    equ 0x64

_hold0      equ 0x68            ;Last filtered value (used if LED switch enabled).
_hold1      equ 0x69
_hold2      equ 0x6A
_hold3      equ 0x6B
_hold4      equ 0x6C
_hold5      equ 0x6D

_input0     equ 0x70            ;_input & _idlecenter & _output must be accessible from any bank
_input1     equ 0x71	
_input2     equ 0x72
_input3     equ 0x73
_input4     equ 0x74
_input5     equ 0x75
_idlecenter equ 0x76		
_delay	    equ 0x77            ;Used by delay macro
_output0    equ 0x78
_output1    equ 0x79
_output2    equ 0x7A
_output3    equ 0x7B
_output4    equ 0x7C
_output5    equ 0x7D
_temp	    equ 0x7E
_temp2	    equ 0x7F


RST	code	0x0000
	goto	_main
	goto	_main
	goto	_main
	goto	_main
	goto	_main
	goto	_main
	goto	_main
	goto	_main

PGM	code	0x0008          ;Skip past ISR vectors...


;;
;;================================================================================================
;; SERIAL_RXBYTE:
;;   Wait for a byte of data to arrive on serial port.
;;   Once initial rising edge detected, delay 8 instr, then capture bit every 16 instr.
;;   Takes 153 instructions to receive 1 byte, including CALL instruction.  
;;   Equiv to ~65 kbaud (6500 bytes/sec).  
;;
;; Received byte stored in _temp variable.  
;; On return, W register contains: 0=Success.  1=Error.
;;================================================================================================
;;
#define HALFBITDELAY 2
#define BITDELAY 4

#ifdef RXSERIAL
#define RXPORT PORTA,5
_serial_rxbyte:
	btfss	RXPORT			;Wait for rising edge (start bit).
	goto	_serial_rxbyte
	delay	HALFBITDELAY
	setreg	_bitcount, 8

	btfss	RXPORT			;Make sure still in start bit
	retlw	1
	nop				;Simulate time taken in serial_rxloop between its btfss
	nop				;and looping back.
	nop
	nop
	nop

_serial_rxloop:
	delay	BITDELAY
	setc
	btfss	RXPORT			;If bit=0, clear carry.  If bit=1, set carry.
	clrc
	rlf	_temp,f
	nop				;Make loop 16 instructions.
	decfsz	_bitcount,f
	goto	_serial_rxloop
	nop				;Keep end-loop timing same as in-loop timing

	delay	BITDELAY
	nop
	btfsc	RXPORT			;If bit not 0, then invalid stop bit
	retlw	1
	retlw	0
#endif


;;
;;================================================================================================
;; SERIAL_TXBYTE:
;;   Send byte in W reg out serial port.   
;;   Send start bit, wait 24 instr, then one bit every 24 instr, ending with 0 for stop bit.
;;   One byte every ~256 instructions = ~39kbaud
;;================================================================================================
;;
#ifdef TXSERIAL
#define TXPORT LATA,1
#define TXMASK 0x57
_serial_txbyte:
	movwf	_temp			;Save byte value
	bsf     TXPORT			;Set start bit
	delay	BITDELAY
	nop
	setreg	_bitcount, 8

_serial_txloop:
	rlf	_temp, f
	skipc
	goto	_tx0
	nop

_tx1:	bsf     TXPORT
	delay	BITDELAY
	decfsz	_bitcount,f
	goto	_serial_txloop
	nop				;Keep end-loop timing same as in-loop timing.
	goto	_txdone

_tx0:	bcf     TXPORT
	delay	BITDELAY
	decfsz	_bitcount,f
	goto	_serial_txloop
	nop				;Keep end-loop timing same as in-loop timing.
	goto	_txdone

_txdone:
	nop
	nop
	bcf     TXPORT			;Send stop bit
	delay	2*BITDELAY
	retlw	0
#endif


#ifdef USE_UART
;;
;; Output contents of WREG as two-byte ascii-hex value...
;;
_uart_txhex:				
	movwf	_temp			;Output high nibble...
	swapf	_temp,w
	call	_uart_txnib
	movf	_temp,w			;Output low nibble...
	;; fall through...
;;
;; Output contents of WREG as single-byte ascii-hex value...
;;
_uart_txnib:
	andlw	0xF
	btfss	WREG,3			;If bit3 set (value>=0x8), skip to bit2 check...
	goto	_uart_outhex
	btfsc	WREG,2			;If bit2 clear (value<0xC), skip to bit1 check...
	goto	_uart_addhex
	btfss	WREG,1			;If bit1 set (value>=0xA), adjust for hex digits...
	goto	_uart_outhex
_uart_addhex:	
	addlw	'A'-'9'-1
_uart_outhex:
	addlw	'0'
	;; fall through...
;;
;; Output contents to WREG to UART (stall if UART busy with previous character)...
;;
_uart_txbyte:
	movlb	(TXREG>>7) & 0x1F	;Select UART BANK
_uart_txloop:
	btfss	(TXSTA&0x1F), TRMT	;Skip goto if transmit shift register empty...
	goto	_uart_txloop
	movwf	(TXREG&0x1F)
	movlb	0			;Reset BANK
	return

uartxmit_byte macro VALUE
	movlw	VALUE	
	call	_uart_txbyte
	endm

uartxmit_aschex macro VARNAME
	movf	VARNAME,w	
	call	_uart_txhex
	endm

#endif


;;
;;================================================================================================
;; Initialize R/C Adapter Program...
;;================================================================================================
;;
_main:	
	setreg	-OSCCON, (1<<IRCF3) | (1<<IRCF2) | (1<<IRCF0) ;Select 4MHz
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop

	setreg  -PORTA, 0		;Clear port A
	setreg  -LATA, 0
	setreg  -PORTC, 0		;Clear port C
	setreg  -LATC, 0
	setreg	-ANSELA, 0		;Select digital I/O's
	setreg	-ANSELC, 0
	setreg	-WPUA, 0		;Disable weak pullups
	setreg	-WPUC, 0	
	setreg	-INLVLA, 0		;Select TTL input sampling
	setreg	-INLVLC, 0
	setreg	-TRISA, 0x38		;Set RA[2:0] as outputs
	setreg	-TRISC, 0x38		;Set RC[2:0] as outputs

#ifdef USE_UART
	setreg	-SPBRGH, 0		;Setup UART for 125KBaud = 4MHz/(4*(7+1))
	setreg	-SPBRGL, 7
	setreg	-BAUDCON, (1<<BRG16)
	setreg	-TXSTA, (1<<TXEN) | (1<<BRGH)
	setreg	-RCSTA, (1<<SPEN) | (1<<CREN)
	setreg	-APFCON0, (1<<TXCKSEL) | (1<<RXDTSEL)	;Configure RX/TX on RA1/RA0
	uartxmit_byte 0
	uartxmit_byte 0xD
	uartxmit_byte 0xA
	uartxmit_byte 't'
	uartxmit_byte 'e'
	uartxmit_byte 's'
	uartxmit_byte 't'
#endif

	banksel	0
	clrf	(_leda)
	clrf	(_ledc)

	clrf	(_timer0)		;Clear pulse timer counters
	clrf	(_timer1)
	clrf	(_timer2)
	clrf	(_timer3)
	clrf	(_timer4)
	clrf	(_timer5)

#ifdef DELTAINPUT0
	setreg	_delta0, DELTAINPUT0	;Init delta timer counters
#endif
#ifdef DELTAINPUT1
	setreg	_delta1, DELTAINPUT1
#endif
#ifdef DELTAINPUT2
	setreg	_delta2, DELTAINPUT2
#endif
#ifdef DELTAINPUT3
	setreg	_delta3, DELTAINPUT3
#endif
#ifdef DELTAINPUT4
	setreg	_delta4, DELTAINPUT4
#endif
#ifdef DELTAINPUT5
	setreg	_delta5, DELTAINPUT5
#endif

	clrf	(_ccount0)		;Clear center duration counts
	clrf	(_ccount1)
	clrf	(_ccount2)
	clrf	(_ccount3)
	clrf	(_ccount4)
	clrf	(_ccount5)

	setreg	_input0, DEFAULT0	;Init default values...
	movwf	(_cvalue0)
	movwf	(_last0)
	movwf	(_last0 + 1)
	movwf	(_last0 + 2)
	movwf	(_last0 + 3)
	movwf	(_hold0)
	movwf	(_output0)

	setreg	_input1, DEFAULT1
	movwf	(_cvalue1)
	movwf	(_last1)
	movwf	(_last1 + 1)
	movwf	(_last1 + 2)
	movwf	(_last1 + 3)
	movwf	(_hold1)
	movwf	(_output1)

	setreg	_input2, DEFAULT2
	movwf	(_cvalue2)
	movwf	(_last2)
	movwf	(_last2 + 1)
	movwf	(_last2 + 2)
	movwf	(_last2 + 3)
	movwf	(_hold2)
	movwf	(_output2)

	setreg	_input3, DEFAULT3
	movwf	(_cvalue3)
	movwf	(_last3)
	movwf	(_last3 + 1)
	movwf	(_last3 + 2)
	movwf	(_last3 + 3)
	movwf	(_hold3)
	movwf	(_output3)

	setreg	_input4, DEFAULT4
	movwf	(_cvalue4)
	movwf	(_last4)
	movwf	(_last4 + 1)
	movwf	(_last4 + 2)
	movwf	(_last4 + 3)
	movwf	(_hold4)
	movwf	(_output4)

	setreg	_input5, DEFAULT5
	movwf	(_cvalue5)
	movwf	(_last5)
	movwf	(_last5 + 1)
	movwf	(_last5 + 2)
	movwf	(_last5 + 3)
	movwf	(_hold5)
	movwf	(_output5)

#ifdef TXCHIP
	setreg	_startcount, 50 	;Number of 20msec loops before sampling R/C inputs (~1sec)...
#endif
#ifdef RXCHIP
	setreg	_startcount, 10 	;Number of 20msec loops before sampling serial input (~200ms)...
#endif
	goto	_validate


;;
;;================================================================================================
;; Start of processing loop...
;;================================================================================================
;;
_start: 
	banksel	0
	
;;
;;================================================================================================
;; Wait 1sec after powering up before measuring output of r/c receiver.
;; While waiting output default values to servos to recover from any glitches.
;;================================================================================================
;;
#ifdef TXCHIP
	decf	_startcount,w		;If startup count == 1, startup init done...
	skipnz
	goto	_startup_done
	movwf	_startcount		;Save decremented startup count.

	setreg	_temp, 20		;20 loops * 1000 instruction delay = 20msec
_loop_delay:
	delay	250
	decf	_temp,f
	skipz
	goto	_loop_delay

	goto	_validate

_startup_done:
#endif


;;
;;================================================================================================
;; If RXSERIAL set, then get measured values from serial input port...
;;================================================================================================
;;
#ifdef RXSERIAL
getrx	macro	TARGET
	call	_serial_rxbyte		;Get servo values from serial port.
	movf	_temp,w
	movwf	TARGET
	endm

	getrx	_input0			;Receiving 6 bytes takes ~1.5msecs
	getrx	_input1
	getrx	_input2
	getrx	_input3
	getrx	_input4
	getrx	_input5

;;
;; For ~200ms wait 5msecs between input samples.   The short delay ensures
;; we stall longer than a serial burst, in case receiver wakes up in the
;; middle of one.  Once aligned to transmitter, each loop takes 20msecs 
;; becauses getrx waits for input.
;;
	decf	_startcount,w		;If startup count == 1, startup init done...
	skipnz
	goto	_rxserial_done
	movwf	_startcount		;Save decremented startup count.

	setreg	_temp, 10		;10 loops * 1000 instruction delay = 10msec
_startup_delay:
	delay	250
	decf	_temp,f
	skipz
	goto	_startup_delay
;;	goto	_start

	setreg	_input0, DEFAULT0
	setreg	_input1, DEFAULT1
	setreg	_input2, DEFAULT2
	setreg	_input3, DEFAULT3
	setreg	_input4, DEFAULT4
	setreg	_input5, DEFAULT5
	goto	_validate

_rxserial_done:
#endif


;;
;;================================================================================================
;; Measure width of input pulses.  Loop must be same # clocks as generate loop.
;;
;; Pulse ranges from 0.9 to 2.1msecs.  Typical center value of 1.52msecs.
;; 4Mhz = 1M instr/sec = 1000 instr/msec
;; So can perform 2000 instr/pulse.  
;;
;; Given 14 clocks per loop, can perform (1000/14) = ~71.43 loops/msec.
;;
;; Since servo control pulses range between 1 and 2msecs, measured counts should 
;; range between 64 and 148.  Thus count fits within single byte.
;;================================================================================================
;;
measure	macro	PORT,BIT,REGNUM
	local	mloop,mmeas
mloop:	btfss	PORT,BIT		;Wait until input non-zero...
	goto	mloop
mmeas:	incf	REGNUM,f		;Measure width of pulse.  One sample every 14 instr.
	delay	2			;10 instr delay
	nop
	nop
	btfsc	PORT,BIT
	goto	mmeas
	endm

#ifdef TXCHIP
_wait4zero:
	btfsc	PORTC,3			;Wait until for input clear.  Don't want to start
	goto	_wait4zero		;measuring in middle of a cycle...

	banksel PORTC
	clrf	(_input0)
	clrf	(_input1)
	clrf	(_input2)
	clrf	(_input3)
	clrf	(_input4)
	clrf	(_input5)

	measure	PORTC,3,_input5
	measure	PORTC,4,_input4
	measure	PORTC,5,_input3
	measure	PORTA,3,_input2
	measure	PORTA,4,_input1
	measure	PORTA,5,_input0
#endif ;; TXCHIP


;;
;;================================================================================================
;; Ensure values in range...
;;================================================================================================
;;
_validate:
;;	min _input0,MIN_PULSE	;min pulse = 0.9 msecs
;;	min _input1,MIN_PULSE
	min _input2,MIN_PULSE
	min _input3,MIN_PULSE
	min _input4,MIN_PULSE
	min _input5,MIN_PULSE

;;	max _input0,MAX_PULSE	;max pulse = 2.1 msecs
;;	max _input1,MAX_PULSE
	max _input2,MAX_PULSE
	max _input3,MAX_PULSE
	max _input4,MAX_PULSE
	max _input5,MAX_PULSE


;;
;;================================================================================================
;; Output serial version of all servo measurements to remote adapter of RC2 (pin8)...
;; Sending 6 channels of 8 bits, for total of 48 bits.
;;
;; Send rising edge (start flag).  Wait longdly for a "1".  shortdly for "0".
;; For each new bit toggle RC2.  The delay indicates if a 0 or 1.
;; Worst case takes 12 * 48 bits, or 576 instructions (~0.6msec).
;;================================================================================================
;;
#ifdef TXSERIAL
	movf	_input0,w	
	call	_serial_txbyte
	movf	_input1,w	
	call	_serial_txbyte
	movf	_input2,w	
	call	_serial_txbyte
	movf	_input3,w	
	call	_serial_txbyte
	movf	_input4,w	
	call	_serial_txbyte
	movf	_input5,w	
	call	_serial_txbyte
#endif



;;
;;================================================================================================
;; Report inputs out UART...
;;================================================================================================
;;
#ifdef USE_UART
	uartxmit_byte 0xD
	uartxmit_byte 0xA
	uartxmit_aschex _input0
	uartxmit_byte ' '
	uartxmit_aschex _input1
	uartxmit_byte ' '
	uartxmit_aschex _input2
	uartxmit_byte ' '
	uartxmit_aschex _input3
	uartxmit_byte ' '
	uartxmit_aschex _input4
	uartxmit_byte ' '
	uartxmit_aschex _input5
#endif


;;
;;================================================================================================
;; Noise filter.  Compensate for sample jitter.  Combine last four samples to average out...
;;================================================================================================
;;
filter	macro	CURRENT, LAST
	local	done
	copyreg	LAST+2,LAST+3
	copyreg	LAST+1,LAST+2
	copyreg	LAST,LAST+1
	copyreg	CURRENT,LAST
	;;----
	clrf	_temp
	addwf	(LAST+1),w	;w=LAST[0]+LAST[1]
	rlf	_temp,f		;store carry
	addwf	(LAST+2),w	;w=LAST[0]+LAST[1]+LAST[2]
	skipnc
	incf	_temp,f		;update carry total
	addwf	(LAST+3),w	;w=LAST[0]+LAST[1]+LAST[2]+LAST[3]
	skipnc
	incf	_temp,f		;update carry total
	movwf	CURRENT
	;;----
	rrf	_temp,f 	;Divide by 4
	rrf	CURRENT,f
	rrf	_temp,f
	rrf	CURRENT,f
	;;----
	skipc			 ;If the two bits shifted out from CURRENT are set, inc CURRENT (ie: round up)
	goto	done
	btfsc	_temp,7
	incf	CURRENT,f
done:
	endm

	banksel 0
	filter	_input0, _last0
	filter	_input1, _last1
	filter	_input2, _last2
	filter	_input3, _last3
	filter	_input4, _last4
	filter	_input5, _last5


;;
;;================================================================================================
;; If SWITCH input being used, and switch asserted, revert input's to last values.
;; In switch state only LED values are updated.
;;================================================================================================
;;
_check_switch:

testled macro	INPUT, VAR, BITNUM
	local	teston, done
	skip_lt	INPUT, OFF_THRESHOLD_LO		;Skip if INPUT<OFF_THRESHOLD_LO
	goto	teston
	bcf	VAR, BITNUM
	goto	done
teston:	skip_ge	INPUT, ON_THRESHOLD_HI		;Skip next if INPUT>=ON_THRESHOLD_HI
	goto	done
	bsf	VAR, BITNUM
done:
        endm

;; If toggle enable set, and INPUT > ON_THRESHOLD_HI, then toggle output & clear toggle enable.
;; If INPUT < ON_THRESHOLD_LO, then set toggle enable.
toggle_up macro INPUT, VAR, BITNUM
	local	done
	skip_ge	INPUT, ON_THRESHOLD_LO
	bsf	VAR, (BITNUM+4)			;Set toggle enable if INPUT<ON_THRESHOLD_LO
	skip_ge	INPUT, ON_THRESHOLD_HI		;Do nothing if INPUT<ON_THRESHOLD_HI
	goto	done
	btfss	VAR, (BITNUM+4)			;Do nothing if toggle enable (BITNUM+4) clear
	goto	done
	bcf	VAR, (BITNUM+4)			;Clear toggle enable
	movlw	(1<<BITNUM)			;Toggle output 
	xorwf	VAR,f
done:
        endm

;; If toggle enable set, and INPUT < OFF_THRESHOLD_LO, then toggle output & clear toggle enable.
;; If INPUT >= OFF_THRESHOLD_HI, then set toggle enable.
toggle_down macro INPUT, VAR, BITNUM
	local	done
	skip_lt	INPUT, OFF_THRESHOLD_HI
	bsf	VAR, (BITNUM+4)			;Set toggle enable if INPUT>=OFF_THRESHOLD_HI
	skip_lt	INPUT, OFF_THRESHOLD_LO		;Do nothing if INPUT>=OFF_THRESHOLD_LO
	goto 	done
	btfss	VAR, (BITNUM+4)			;Do nothing if toggle enable (BITNUM+4) clear
	goto	done
	bcf	VAR, (BITNUM+4)			;Clear toggle enable
	movlw	(1<<BITNUM)			;Toggle output
	xorwf	VAR,f
done:
        endm

;; If pulse enable set, and INPUT>ON_THRESHOLD_HI, then set output, init timer, clear pulse enable.
;; If timer==0, and INPUT < ON_THRESHOLD_LO, then set pulse enable.
;; Decrement timer until reaches 0, then clear output.
pulse_up macro INPUT, DELAY, TIMER, VAR, BITNUM
	local	check_timer, timer_zero, done
	skip_lt	INPUT, ON_THRESHOLD_HI		;Do nothing if INPUT<ON_THRESHOLD_HI
	btfss	VAR, (BITNUM+4)			;Do nothing if pulse enable (BITNUM+4) clear
	goto	check_timer
	bcf	VAR, (BITNUM+4)			;Clear pulse enable
	bsf	VAR, BITNUM			;Set output 
	setreg	TIMER, DELAY
check_timer:
	movf	TIMER,f				;Test if TIMER zero
	skipnz	
	goto	timer_zero
	decf	TIMER,f				;If TIMER hits 0, clear output...
	skipz
	goto	done
	bcf	VAR, BITNUM
timer_zero:
	skip_ge	INPUT, ON_THRESHOLD_LO
	bsf	VAR, (BITNUM+4)			;Set pulse enable if INPUT<ON_THRESHOLD_LO
done:	
        endm

;; If pulse enable set, and INPUT < OFF_THRESHOLD_LO, then set output, init timer, clear pulse enable.
;; If timer==0, and INPUT >= OFF_THRESHOLD_HI, then set pulse enable.
;; Decrement timer until reaches 0, then clear output.
pulse_down macro INPUT, DELAY, TIMER, VAR, BITNUM
	local	check_timer, timer_zero, done
	skip_ge	INPUT, OFF_THRESHOLD_LO		;Do nothing if INPUT>=OFF_THRESHOLD_LO
	btfss	VAR, (BITNUM+4)			;Do nothing if pulse enable (BITNUM+4) clear
	goto	check_timer
	bcf	VAR, (BITNUM+4)			;Clear pulse enable
	bsf	VAR, BITNUM			;Set output 
	setreg	TIMER, DELAY
check_timer:
	movf	TIMER,f				;Test if TIMER zero
	skipnz	
	goto	timer_zero
	decf	TIMER,f				;If TIMER hits 0, clear output...
	skipz
	goto	done
	bcf	VAR, BITNUM
timer_zero:
	skip_lt	INPUT, OFF_THRESHOLD_HI
	bsf	VAR, (BITNUM+4)			;Set pulse enable if INPUT>=OFF_THRESHOLD_HI
done:	
        endm


#ifdef SWITCH
	copyreg	SWITCH, _switch
	skip_ge	_switch, ON_THRESHOLD_HI	;Skip to update if SWITCH < ON_THRESHOLD_HI
	goto	_update
#endif


#ifdef LED0
#define LED0_MASK 0x67
	testled	LED0, _leda, 0
#endif
#ifdef TOGGLEUP0
#define LED0_MASK 0x67
	toggle_up TOGGLEUP0, _leda, 0
#endif
#ifdef TOGGLEDN0
#define LED0_MASK 0x67
	toggle_down TOGGLEDN0, _leda, 0
#endif
#ifdef PULSEUP0
#define LED0_MASK 0x67
	pulse_up PULSEUP0, _timer0, _leda, 0
#endif
#ifdef PULSEDN0
#define LED0_MASK 0x67
	pulse_down PULSEDN0, _timer0, _leda, 0
#endif

#ifdef LED1
#define LED1_MASK 0x57
	testled	LED1, _leda, 1
#endif
#ifdef TOGGLEUP1
#define LED1_MASK 0x57
	toggle_up TOGGLEUP1, _leda, 1
#endif
#ifdef TOGGLEDN1
#define LED1_MASK 0x57
	toggle_down TOGGLEDN1, _leda, 1
#endif
#ifdef PULSEUP1
#define LED1_MASK 0x57
	pulse_up PULSEUP1, _timer1, _leda, 1
#endif
#ifdef PULSEDN1
#define LED1_MASK 0x57
	pulse_down PULSEDN1, _timer1, _leda, 1
#endif

#ifdef LED2
#define LED2_MASK 0x37
	testled	LED2, _leda, 2
#endif
#ifdef TOGGLEUP2
#define LED2_MASK 0x37
	toggle_up TOGGLEUP2, _leda, 2
#endif
#ifdef TOGGLEDN2
#define LED2_MASK 0x37
	toggle_down TOGGLEDN2, _leda, 2
#endif
#ifdef PULSEUP2
#define LED2_MASK 0x37
	pulse_up PULSEUP2, _timer2, _leda, 2
#endif
#ifdef PULSEDN2
#define LED2_MASK 0x37
	pulse_down PULSEDN2, _timer2, _leda, 2
#endif

#ifdef LED3
#define LED3_MASK 0x76
	testled	LED3, _ledc, 0
#endif
#ifdef TOGGLEUP3
#define LED3_MASK 0x76
	toggle_up TOGGLEUP3, _ledc, 0
#endif
#ifdef TOGGLEDN3
#define LED3_MASK 0x76
	toggle_down TOGGLEDN3, _ledc, 0
#endif
#ifdef PULSEUP3
#define LED3_MASK 0x76
	pulse_up PULSEUP3, _timer3, _ledc, 0
#endif
#ifdef PULSEDN3
#define LED3_MASK 0x76
	pulse_down PULSEDN3, _timer3, _ledc, 0
#endif

#ifdef LED4
#define LED4_MASK 0x75
	testled	LED4, _ledc, 1
#endif
#ifdef TOGGLEUP4
#define LED4_MASK 0x75
	toggle_up TOGGLEUP4, _ledc, 1
#endif
#ifdef TOGGLEDN4
#define LED4_MASK 0x75
	toggle_down TOGGLEDN4, _ledc, 1
#endif
#ifdef PULSEUP4
#define LED4_MASK 0x75
	pulse_up PULSEUP4, _timer4, _ledc, 1
#endif
#ifdef PULSEDN4
#define LED4_MASK 0x75
	pulse_down PULSEDN4, _timer4, _ledc, 1
#endif

#ifdef LED5
#define LED5_MASK 0x73
	testled	LED5, _ledc, 2
#endif
#ifdef TOGGLEUP5
#define LED5_MASK 0x73
	toggle_up TOGGLEUP5, _ledc, 2
#endif
#ifdef TOGGLEDN5
#define LED5_MASK 0x73
	toggle_down TOGGLEDN5, _ledc, 2
#endif
#ifdef PULSEUP5
#define LED5_MASK 0x73
	pulse_up PULSEUP5, _timer5, _ledc, 2
#endif
#ifdef PULSEDN5
#define LED5_MASK 0x73
	pulse_down PULSEDN5, _timer5, _ledc, 2
#endif


#ifdef SWITCH
	banksel 0
	copyreg	_hold0, _input0
	copyreg	_hold1, _input1
	copyreg	_hold2, _input2
	copyreg	_hold3, _input3
	copyreg	_hold4, _input4
	copyreg	_hold5, _input5
	goto	_setup_outputs
#endif 


;;
;;================================================================================================
;; Update inputs...   
;;
;; If delta input,
;;   if _deltax == 0, 
;;     set _deltax = DELTAINPUTx
;;     If input above CENTERx_PULSE+5, add tick to "_holdx" (if _holdx below MAXx_PULSE).
;;     If input below CENTERx_PULSE-5, subtract tick from "_holdx" (if _holdx above MINx_PULSE)..
;;   else decrement _deltax
;;
;; If normal input,
;;   Copy input to hold.
;;================================================================================================
;;
evaldelta macro	INPUTVAR, HOLDVAR, DELTAVAR, DEFAULT_COUNT
	local	not_above, not_zero, done
	decf	DELTAVAR,f
	jumpnz	done			;; jump if delta not zero after decrement

	movlw	DEFAULT_COUNT
	movwf	DELTAVAR		;; set _deltax = default tick count (DELTAINPUTx)

	skip_ge	INPUTVAR, (CENTER_PULSE+6) ;; input > (CENTER_PULSE+5) ?
	goto	not_above		;; jump if not
	skip_ge	HOLDVAR, MAX_PULSE	;; holdvar >= MAX_PULSE ?
	incf	HOLDVAR,f		;; increment holdvar if not
	goto	done

not_above:
	skip_lt	INPUTVAR, (CENTER_PULSE-5) ;; input < (CENTER_PULSE-5) ?
	goto	done			;; jump if not
	skip_lt	HOLDVAR, (MIN_PULSE+1)	;; holdvar <= MIN_PULSE ?
	decf	HOLDVAR,f		;; decrement holdvar if not
done:
	copyreg	HOLDVAR, INPUTVAR
	endm

_update:
	banksel	0

#ifdef DELTAINPUT0
	evaldelta _input0, _hold0, _delta0, DELTAINPUT0
#else
	copyreg _input0, _hold0	
#endif

#ifdef DELTAINPUT1
	evaldelta _input1, _hold1, _delta1, DELTAINPUT1
#else
	copyreg _input1, _hold1	
#endif

#ifdef DELTAINPUT2
	evaldelta _input2, _hold2, _delta2, DELTAINPUT2
#else
	copyreg _input2, _hold2	
#endif

#ifdef DELTAINPUT3
	evaldelta _input3, _hold3, _delta3, DELTAINPUT3
#else
	copyreg _input3, _hold3	
#endif

#ifdef DELTAINPUT4
	evaldelta _input4, _hold4, _delta4, DELTAINPUT4
#else
	copyreg _input4, _hold4	
#endif

#ifdef DELTAINPUT5
	evaldelta _input5, _hole5, _delta5, DELTAINPUT5
#else
	copyreg _input5, _hold5	
#endif



;;
;;================================================================================================
;; Assert LED outputs...
;;================================================================================================
;;
setled	macro	LEDVAR, PORTVAR, BITNUM
	local	ledon, done
	btfsc	LEDVAR, BITNUM	; skip next if ledvar bit clear
	goto	ledon
	banksel	PORTVAR
	bcf	PORTVAR, BITNUM
	goto	done
ledon:	
	banksel	PORTVAR
	bsf	PORTVAR, BITNUM
done:
	banksel	0
	endm

_setup_outputs:

	banksel	0
#ifdef LED0_MASK
	setled	_leda, LATA, 0
#endif
#ifdef LED1_MASK
	setled	_leda, LATA, 1
#endif
#ifdef LED2_MASK
	setled	_leda, LATA, 2
#endif
#ifdef LED3_MASK
	setled	_ledc, LATC, 0
#endif
#ifdef LED4_MASK
	setled	_ledc, LATC, 1
#endif
#ifdef LED5_MASK
	setled	_ledc, LATC, 2
#endif

#ifndef OUTPUT0
#define OUTPUT0 _input0
#endif
#ifndef OUTPUT1
#define OUTPUT1 _input1
#endif
#ifndef OUTPUT2
#define OUTPUT2 _input2
#endif
#ifndef OUTPUT3
#define OUTPUT3 _input3
#endif
#ifndef OUTPUT4
#define OUTPUT4 _input4
#endif
#ifndef OUTPUT5
#define OUTPUT5 _input5
#endif

#ifndef LED0_MASK
#define LED0_MASK 0x77
#endif
#ifndef LED1_MASK
#define LED1_MASK 0x77
#endif
#ifndef LED2_MASK
#define LED2_MASK 0x77
#endif
#ifndef LED3_MASK
#define LED3_MASK 0x77
#endif
#ifndef LED4_MASK
#define LED4_MASK 0x77
#endif
#ifndef LED5_MASK
#define LED5_MASK 0x77
#endif
#ifndef TXMASK
#define TXMASK 0x77
#endif

#ifndef USE_UART
#define UARTMASK 0x77
#else
#define UARTMASK 0x47
#endif


;;
;;================================================================================================
;; Setup to generate servo output pulses.  Perform scaling of input range if needed.
;;================================================================================================
;;
#define SERVO_MASK (LED0_MASK & LED1_MASK & LED2_MASK & LED3_MASK & LED4_MASK & LED5_MASK & TXMASK & UARTMASK)
	messg "SERVO_MASK="SERVO_MASK

;; outreg = NEWCENTER - (OLDCENTER - srcreg)*(NEWCENTER-NEW_MIN)/(OLDCENTER-OLD_MIN)
scalemin macro	SRCREG,NEWMIN,OLDCENTER,NEWCENTER,OUTREG
	setreg	_temp, OLDCENTER
	movf	SRCREG,w		; w = SRCREG
	subwf	_temp,w			; w = CENTER - SRCREG
	multC16xW ((256*(NEWCENTER-NEWMIN))/(OLDCENTER-MIN_PULSE)), _temp, OUTREG, _temp2
	comf	OUTREG,f		; OUTREG = CENTER-OUTREG
	incf	OUTREG,f
	movlw	NEWCENTER
	addwf	OUTREG,f
	endm

;; outreg = NEWCENTER + (srcreg - OLDCENTER)*(NEW_MAX-NEWCENTER)/(OLD_MAX-OLDCENTER)
scalemax macro  SRCREG,NEWMAX,OLDCENTER,NEWCENTER,OUTREG
	movlw	OLDCENTER
	subwf	(SRCREG),w		;W = (SRCREG-OLDCENTER)
	multC16xW ((256*(NEWMAX-NEWCENTER))/(MAX_PULSE-OLDCENTER)), _temp, OUTREG, _temp2
	movlw	NEWCENTER
	addwf	(OUTREG),f
	endm

;; Use different scaling, depending if above or below CENTER...
;; if srcreg < CENTER,
;;    outreg = CENTER - (CENTER - srcreg)*(CENTER-NEW_MIN)/(CENTER-OLD_MIN)
;; else outreg = CENTER + (srcreg - CENTER)*(NEW_MAX-CENTER)/(OLD_MAX-CENTER)
scale2	macro SRCREG,NEWMIN,NEWMAX,OLDCENTER,NEWCENTER,OUTREG
	local smin,smax,done
	skip_lt	SRCREG,OLDCENTER
	goto	smax
smin:	
	if NEWMIN != MIN_PULSE
	scalemin SRCREG,NEWMIN,OLDCENTER,NEWCENTER,OUTREG
	else
	copyreg	SRCREG, OUTREG
	endif
	goto	done
smax:	
	if NEWMAX != MAX_PULSE
	scalemax SRCREG,NEWMAX,OLDCENTER,NEWCENTER,OUTREG
	else
	copyreg	SRCREG, OUTREG
	endif
done:
	endm

	banksel	0
#ifdef MIN0_PULSE
	scale2	OUTPUT0, MIN0_PULSE, MAX0_PULSE, CENTER_PULSE, CENTER0_PULSE, _output0
#else
	copyreg	OUTPUT0, _output0
#endif
#ifdef MIN1_PULSE
	scale2	OUTPUT1, MIN1_PULSE, MAX1_PULSE, CENTER_PULSE, CENTER1_PULSE, _output1
#else
	copyreg	OUTPUT1, _output1
#endif
#ifdef MIN2_PULSE
	scale2	OUTPUT2, MIN2_PULSE, MAX2_PULSE, CENTER_PULSE, CENTER2_PULSE, _output2
#else
	copyreg	OUTPUT2, _output2
#endif
#ifdef MIN3_PULSE
	scale2	OUTPUT3, MIN3_PULSE, MAX3_PULSE, CENTER_PULSE, CENTER3_PULSE, _output3
#else
	copyreg	OUTPUT3, _output3
#endif
#ifdef MIN4_PULSE
	scale2	OUTPUT4, MIN4_PULSE, MAX4_PULSE, CENTER_PULSE, CENTER4_PULSE, _output4
#else
	copyreg	OUTPUT4, _output4
#endif
#ifdef MIN5_PULSE
	scale2	OUTPUT5, MIN5_PULSE, MAX5_PULSE, CENTER_PULSE, CENTER5_PULSE, _output5
#else
	copyreg	OUTPUT5, _output5
#endif
	


;;
;;================================================================================================
;; See if any servo outputs should be idled because input within center range.
;; If a servo output should be idled, set bit in _idlecenter variable.
;;================================================================================================
;;
#define CCOUNT_THRES 6	
centermask macro OUTVAR, CVALUE, CCOUNT, MASKVAR, MASKVALUE, CENTER_TOL
	local	changed, nochange, done
	movf	OUTVAR,w	
	subwf	CVALUE,w		; W = CVALUE-OUTVAR
	addlw	CENTER_TOL		; Adjust by min acceptable value
	movwf	_temp
	skip_lt	_temp, (CENTER_TOL*2)+1		
	goto	changed			; Jump if (CVALUE-OUTVAR)>=3, or (CVALUE-OUTVAR)<-2

nochange:
	movlw	MASKVALUE		; Mask if input unchanged...
	xorwf	MASKVAR,f
	btfsc	CCOUNT,CCOUNT_THRES	; If CCOUNT>=0x40, then locked, so no more changes...
	goto	done
	movf	OUTVAR,w
	movwf	CVALUE	
	incf	CCOUNT,f	
	goto	done

changed:
	btfsc	CCOUNT,CCOUNT_THRES	; If CCOUNT>=0x40, then locked, so no more changes...
	goto	done
	movf	OUTVAR,w		; If OUTVAR changes before CCOUNT reaches 0x40,
	movwf	CVALUE			; then reset CVALUE...
	clrf	CCOUNT
done:
	endm

	banksel	0
	setreg	_idlecenter, 0x77

#ifdef IDLECENTER0
	centermask _output0, _cvalue0, _ccount0, _idlecenter, 0x10, IDLECENTER0
#endif
#ifdef IDLECENTER1
	centermask _output1, _cvalue1, _ccount1, _idlecenter, 0x20, IDLECENTER1
#endif
#ifdef IDLECENTER2
	centermask _output2, _cvalue2, _ccount2, _idlecenter, 0x40, IDLECENTER2
#endif
#ifdef IDLECENTER3
	centermask _output3, _cvalue3, _ccount3, _idlecenter, 0x01, IDLECENTER3
#endif
#ifdef IDLECENTER4
	centermask _output4, _cvalue4, _ccount4, _idlecenter, 0x02, IDLECENTER4
#endif
#ifdef IDLECENTER5
	centermask _output5, _cvalue5, _ccount5, _idlecenter, 0x04, IDLECENTER5
#endif


;;
;;================================================================================================
;; See if any servo outputs should be idled because position hasent changed in last second.
;; If a servo output should be idled, set bit in _idlecenter variable.
;;================================================================================================
;;
#define DIFFTOL 2
countmask macro OUTVAR, CVALUE, CCOUNT, MASKVAR, MASKVALUE
	local	nochange, changed, inc, done
	movf	OUTVAR,w	
	subwf	CVALUE,w
	movwf	_temp			; _temp = diff of last two samplesd
	movlw	DIFFTOL
	addwf	_temp,f			; _temp = _temp + 2
	movlw	(DIFFTOL*2)+1
	subwf	_temp,f			; _temp = _temp - 5 (original diff values 0xFE to 0x02 will set borrow flag)
	skipb
	goto	changed			; Jump if _temp>=3, or _temp<-2

nochange:
	btfss	CCOUNT,5		; If CCOUNT>=0x20, then mask it...
	goto	inc
	movlw	MASKVALUE		; Mask if input unchanged...
	xorwf	MASKVAR,f
	goto	done

changed:
	movf	OUTVAR,w
	movwf	CVALUE
	clrf	CCOUNT
inc:
	incf	CCOUNT,f		; If CCOUNT<0x20, increment count...
done:
	endm

#ifdef IDLEINPUT0
	countmask _input0, _cvalue0, _ccount0, _idlecenter, IDLEINPUT0
#endif
#ifdef IDLEINPUT1
	countmask _input1, _cvalue1, _ccount1, _idlecenter, IDLEINPUT1
#endif
#ifdef IDLEINPUT2
	countmask _input2, _cvalue2, _ccount2, _idlecenter, IDLEINPUT2
#endif
#ifdef IDLEINPUT3
	countmask _input3, _cvalue3, _ccount3, _idlecenter, IDLEINPUT3
#endif
#ifdef IDLEINPUT4
	countmask _input4, _cvalue4, _ccount4, _idlecenter, IDLEINPUT4
#endif
#ifdef IDLEINPUT5
	countmask _input5, _cvalue5, _ccount5, _idlecenter, IDLEINPUT5
#endif


;;
;;================================================================================================
;; Debug - Dump current output values to UART...
;;================================================================================================

;; debug
#ifdef USE_UART
	banksel	0
	uartxmit_byte ' '
	uartxmit_byte '-'
	uartxmit_byte ' '
	uartxmit_aschex _output0
	uartxmit_byte ' '
	uartxmit_aschex _output1
	uartxmit_byte ' '
	uartxmit_aschex _output2
	uartxmit_byte ' '
	uartxmit_aschex _output3
	uartxmit_byte ' '
	uartxmit_aschex _output4
	uartxmit_byte ' '
	uartxmit_aschex _output5
	uartxmit_byte ' '
	uartxmit_byte '-'
	uartxmit_byte ' '
	uartxmit_aschex _idlecenter
#endif


;;
;;================================================================================================
;; Generate PORTA outputs...
;;================================================================================================
;;
	errorlevel -302			;Disable register banksel warnings
	banksel LATA

	movf	_idlecenter,w		;Mask off outputs being used for LED's...
	andlw	SERVO_MASK
	movwf	_idlecenter
	
	swapf	_idlecenter,w		;Get mask for LATA.
	andlw	0x7			;Check if LATA mask all zero.
	skipnz				;If nothing to output, skip to LATC...
	goto	_setupgenc

	iorwf	LATA,f			;Turn on LATA outputs.

_generateb:
#if ((UARTMASK & LED0_MASK) != 0x77)
	nop
	nop
	nop
#else
	decf	(_output0),f		;Clear RA0 (pin13) if count reaches zero
	skipnz
	bcf	LATA,0
#endif

#if ((UARTMASK & TXMASK & LED1_MASK) != 0x77)
	nop
	nop
	nop
#else
	decf	(_output1),f		;Clear RA1 (pin12) if count reaches zero
	skipnz
	bcf	LATA,1
#endif

#if (LED2_MASK != 0x77)
	nop
	nop
	nop
#else
	decf	(_output2),f		;Clear RA2 (pin11) if count reaches zero
	skipnz
	bcf	LATA,2
#endif
	movf	LATA,w			;W = LATA
	andlw	(SERVO_MASK>>4)&7	;W = LATA & SERVO_MASK
	skipz				;Loop until all outputs zero
	goto	_generateb


;;
;;================================================================================================
;; Generate PORTC outputs...
;;================================================================================================
;;
_setupgenc:
	movf	_idlecenter,w		;Get mask for LATC.
	andlw	0x7			;Check if LATC mask all zero.
	skipnz				;If nothing to output, skip to start...
	goto	_start

	iorwf	LATC,f			;Turn on LATC outputs.

_generatec:
#if (LED3_MASK != 0x77)
	nop
	nop
	nop
#else
	decf	(_output3),f		;Clear RC0 (pin10) if count reaches zero
	skipnz
	bcf	LATC,0
#endif

#if (LED4_MASK != 0x77)
	nop
	nop
	nop
#else
	decf	(_output4),f		;Clear RC1 (pin9) if count reaches zero
	skipnz
	bcf	LATC,1
#endif

#if (LED5_MASK != 0x77)
	nop
	nop
	nop
#else
	decf	(_output5),f		;Clear RC2 (pin8) if count reaches zero
	skipnz
	bcf	LATC,2
#endif
	movf	LATC,w			;W = LATC latch outputs
	andlw	(SERVO_MASK)&7		;W = LATC & SERVO_MASK
	skipz				;Loop until all outputs zero
	goto	_generatec

	goto	_start

	end

