/* -*- mode: c++; c-basic-offset: 4; indent-tabs-mode: nil; -*-
 * (c) h.zeller@acm.org. Free Software. GNU Public License v3.0 and above
 */

/*
  TODO PnP
   - Instead of a one-size-fits-all 10mm, have a shallow hover height (1mm).
     Essentially the clearance we want above obstructions in the path.
   - determine actual hover depending on component height
   - Pick:
      - go hover height over tape.
      - go down, suck it.
      - pull up top-of-tape + tape-thickness + 1mm to get the component, that is
        supposedly as high as the tape, safely out of it
   - Place:
      - move it towards the board. Z should be
        top-of-board + component-height + 1mm. That way multiple of the same
        component are not knocked over
      - (thinner components come first in the order, so this is always safe)
   - Result: best travel times (less Z movement), but guaranteed non-knockover
     result.
   - TODO: These height decisions should not be made in the machine. They
     should happen outside whoever sends the plan to the machine.
     That way, it allows for easy testing the results by implementing a mock
     machine in the test.

  Also FIXME:
  Current assumption of tape is that is sits somewhere on the build-platform.
  (i.e. that tape-height == component-height).
  That needs to change, as it could well be elevated on a tray or something and
  then we would start dropping components from some height on the board :)
*/

#include "machine.h"

#include <assert.h>
#include <stdio.h>
#include <math.h>

#include "tape.h"
#include "board.h"

#include "pnp-config.h"

// TODO: most of these constants should be configurable or deduced from board/
// configuration.

// Hovering while transporting a component.
// TODO: calculate that to not knock over already placed components.
#define PNP_Z_HOVERING 10

// Components are a bit higher as they are resting on some card-board. Let's
// assume some value here.
// Ultimately, we want the placement operation be a bit spring-loaded.
#define PNP_TAPE_THICK 0.0

// Multiplication to get 360 degrees mapped to one turn. This is specific to
// our stepper motor.
#define PNP_ANGLE_FACTOR (50.34965 / 360)

// Speeds in mm/s
#define PNP_TO_TAPE_SPEED 1000          // moving needle to tape
#define PNP_TO_BOARD_SPEED 100          // moving component from tape to board

#define DISP_MOVE_SPEED 20              // move dispensing unit to next pad
#define DISP_DISPENSE_SPEED 1           // speed when doing the dispensing down/up
#define DISP_DISPENSE_DIV 3.0           // dispensing area divider
#define DISP_DISPENSE_STRIPPED 1.0      // generate stripped gcode

#define DISP_Z_DISPENSING_ABOVE 0.3     // Above board when dispensing
#define DISP_Z_HOVER_ABOVE 2            // Above board when moving around
#define DISP_Z_SEPARATE_DROPLET_ABOVE 5 // Above board right after dispensing.

// All templates should be in a separate file somewhere so that we don't
// have to compile.

// param: moving needle up.
static const char *const gcode_preamble = R"(
G21        ; set to mm
G91        ; Use relative positions in general.
M790 P4 "DOW1___" ; enter dispensing mode
)";

// param: moving needle up.
static const char *const gcode_preamble_stripped = R"(G21
G91
M790 P4 "DOW1___")";

// param: name, move-speed, x, y, zup, zdown, a, zup
static const char *const gcode_pick = R"(
;; -- Pick %s
G0 F%d X%.3f Y%.3f Z%.3f E%.3f ; Move over component to pick.
G1 Z%-6.2f   F4000 ; move down on tape.
G4           ; flush buffer
M42 P6 S255  ; turn on suckage
G1 Z%-6.3f   ; Move up a bit for travelling
)";

// param: name, place-speed, x, y, zup, a, zdown, zup
static const char *const gcode_place = R"(
;; -- Place %s
G0 F%d X%.3f Y%.3f Z%.3f E%.3f ; Move component to place on board.
G1 Z%-6.3f F4000 ; move down over board thickness.
G4            ; flush buffer.
M42 P6 S0     ; turn off suckage
G4            ; flush buffer.
M42 P8 S255   ; blow
G4 P40        ; .. for 40ms
M42 P8 S0     ; done.
G1 Z%-6.2f    ; Move up
)";

// move to new position, above board.
// param: component-name, pad-name, x, y, hover-z
static const char *const gcode_dispense_move = R"(
;; -- component %s, pad %s
G0 F%d X%.3f Y%.3f Z%.3f ; move there.)";

// move to new position, above board.
// stripped command from comments.
// param: component-name, pad-name, x, y, hover-z
static const char *const gcode_dispense_move_stripped = R"(
G0 F%d X%.3f Y%.3f Z%.3f)";

// Dispense paste.
// param: z-dispense-height, wait-time-ms, area, z-separate-droplet
static const char *const gcode_dispense_paste = R"(
G1 F%d Z%.2f ; Go down to dispense
G4 P500.0 ; pause for 0.5 sec
M790 P4 "STE%d___" ; perform steps dependent on area %.2f mm^2
G4 P999.0 ; Wait time
G1 Z%.2f ; high above to have paste separated
)";

// Dispense paste.
// param: z-dispense-height, wait-time-ms, area, z-separate-droplet
// stripped command from comments
static const char *const gcode_dispense_paste_stripped = R"(
G1 F%d Z%.2f
G4 P500.0
M790 P4 "STE%d___"
G4 P999.0
G1 Z%.2f)";

static const char *const gcode_finish = R"(
)";

GCodeMachine::GCodeMachine(float init_ms, float area_ms)
    : init_ms_(init_ms), area_ms_(area_ms), config_(NULL) {}

bool GCodeMachine::Init(const PnPConfig *config,
                        const std::string &init_comment,
                        const Dimension& dim) {
    config_ = config;
    if (config_ == NULL) {
        fprintf(stderr, "Need configuration\n");
        return false;
    }
    fprintf(stderr, "Board-thickness = %.1fmm\n",
            config_->board.top - config_->bed_level);
    printf("; %s\n", init_comment.c_str());
    float highest_tape = config_->board.top;
    for (const auto &t : config_->tape_for_component) {
        highest_tape = std::max(highest_tape, t.second->height());
    }
    if(DISP_DISPENSE_STRIPPED==0.0){
    	printf(gcode_preamble/*, highest_tape + 10*/);
    }else{
    	printf(gcode_preamble_stripped/*, highest_tape + 10*/);
    }
    return true;
}

void GCodeMachine::PickPart(const Part &part, const Tape *tape) {
    if (tape == NULL) return;
    float px, py;
    if (!tape->GetPos(&px, &py)) {
        fprintf(stderr, "We are out of components for %s %s\n",
                part.footprint.c_str(), part.value.c_str());
        return;
    }

    const float board_thick = config_->board.top - config_->bed_level;
    const float travel_height = tape->height() + board_thick + PNP_Z_HOVERING;
    const std::string print_name = part.component_name + " ("
        + part.footprint + "@" + part.value + ")";

    // param: name, x, y, zdown, a, zup
    printf(gcode_pick,
           print_name.c_str(),
           60 * PNP_TO_TAPE_SPEED,
           px, py, tape->height() + PNP_Z_HOVERING,     // component pos.
           PNP_ANGLE_FACTOR * fmod(tape->angle(), 360.0),   // pickup angle
           tape->height(),                              // down to component
           travel_height);                              // up for travel.
}

void GCodeMachine::PlacePart(const Part &part, const Tape *tape) {
    if (tape == NULL) return;
    const float board_thick = config_->board.top - config_->bed_level;
    const float travel_height = tape->height() + board_thick + PNP_Z_HOVERING;
    const std::string print_name = part.component_name + " ("
        + part.footprint + "@" + part.value + ")";

    // param: name, x, y, zup, a, zdown, zup
    printf(gcode_place,
           print_name.c_str(),
           60 * PNP_TO_BOARD_SPEED,
           part.pos.x + config_->board.origin.x,
           part.pos.y + config_->board.origin.y,
           travel_height,
           PNP_ANGLE_FACTOR * fmod(part.angle - tape->angle() + 360, 360.0),
           tape->height() + board_thick - PNP_TAPE_THICK,
           travel_height);
}

PreviousPad GCodeMachine::Dispense(const Part &part, const Pad &pad, int count, float prev_x, float prev_y) {
     // TODO: so this part looks like we shouldn't have to do it here.
     //const float angle = 2 * M_PI * part.angle / 360.0;
     PreviousPad previousPad;
     const float angle = 2 * 3.14159265359 * part.angle / 360.0;
     const float part_x = config_->board.origin.x + part.pos.x;
     const float part_y = config_->board.origin.y + part.pos.y;
     const float area = pad.size.w * pad.size.h;
     const float pad_x = pad.pos.x;
     const float pad_y = pad.pos.y;
     const float x = part_x - prev_x + pad_x * cos(angle) - pad_y * sin(angle);
     const float y = part_y - prev_y + pad_x * sin(angle) + pad_y * cos(angle);
     previousPad.x = part_x + pad_x * cos(angle) - pad_y * sin(angle);;
     previousPad.y = part_y + pad_x * sin(angle) + pad_y * cos(angle);;
     if(count>0){
     	if(DISP_DISPENSE_STRIPPED==0.0){
        printf(gcode_dispense_move,
               part.component_name.c_str(), pad.name.c_str(),
               DISP_MOVE_SPEED * 60,
               x, y, -5.0);
        printf(gcode_dispense_paste, DISP_DISPENSE_SPEED * 60,
               -1.9,
               (int)((init_ms_ + (area * area_ms_))/DISP_DISPENSE_DIV), area,
               6.9);
      }else{
      	printf(gcode_dispense_move_stripped,
               DISP_MOVE_SPEED * 60,
               x, y, -5.0);
        printf(gcode_dispense_paste_stripped, DISP_DISPENSE_SPEED * 60,
               -1.9,
               (int)((init_ms_ + (area * area_ms_))/DISP_DISPENSE_DIV),
               6.9);
      }
     }else{
     	if(DISP_DISPENSE_STRIPPED==0.0){
     	  printf(gcode_dispense_paste, DISP_DISPENSE_SPEED * 60,
               0.1,
               (int)((init_ms_ + (area * area_ms_))/DISP_DISPENSE_DIV), area,
               6.9);
      }else{
      	printf(gcode_dispense_paste_stripped, DISP_DISPENSE_SPEED * 60,
               0.1,
               (int)((init_ms_ + (area * area_ms_))/DISP_DISPENSE_DIV),
               6.9);
      }
     }
     return previousPad;
}

void GCodeMachine::Finish() {
    printf(gcode_finish);
}
