   PROCESSOR 16F1823
#INCLUDE <P16F1823.INC>
	RADIX	dec
	LIST	b=4

;===============================================================================
	TITLE	Universal LED driver
;	by Akimitsu Sadoi - aki@theLEDart.com
;	Commercial use of this code is prohibited.
;-------------------------------------------------------------------------------
;	Version history
;	0.0 ported from 16F616 version
;	0.1 use of DAC for comparator 1 (V_SENSE) reference voltage
;	0.2 IR remote (Sony) receive function
;	0.3 main loop revised to remove glitches
;===============================================================================

#INCLUDE <SONY_TV.INC>	; Sony TV remote codes

;--- Device Configuration Bits -------------------------------------------------
	__CONFIG _CONFIG1, _FOSC_INTOSC & _WDTE_ON & _PWRTE_ON & _BOREN_ON & _CLKOUTEN_OFF & _MCLRE_ON & _CPD_OFF
	__CONFIG _CONFIG2, _WRT_OFF & _PLLEN_ON & _STVREN_ON & _BORV_25 & _LVP_ON
;===============================================================================
;	constants
;
option_param	equ		b'00001000'		; pull-up enable, TMR0 no prescale
wdtcon_param	equ		b'00000001'		; WDT prescale = 1:32 (1ms), WDT enable

ccpcon_pram		equ		b'00001100'		; LSBs of duty cycle(bit 5:4), PWM active-high
prescale		equ		1				; 1: prescale 1, 2: 4, 3: 16
postscale		equ		1				; between 1 to 16
t2com_param		equ		((postscale-1)<<3)|4|(prescale-1)

; maximum PWM period value (PR2) - lowest frequency
pwm_pr_max		equ		255				; 31.25kHz - lowest possible

; mininum PWM period value (PR2) - highest frequency
;pwm_pr_min		equ		159				; 50kHz
;pwm_pr_min		equ		127				; 62.5kHz
;pwm_pr_min		equ		91				; 87kHz
;pwm_pr_min		equ		81				; 97.6kHz
;pwm_pr_min		equ		79				; 100kHz
;pwm_pr_min		equ		63				; 125kHz
;pwm_pr_min		equ		52				; 150kHz
;pwm_pr_min		equ		47				; 166kHz
pwm_pr_min		equ		45				; 174kHz
;pwm_pr_min		equ		42				; 186kHz
;pwm_pr_min		equ		39				; 200kHz

; mininum off-time (x125ns)
;min_off_time	equ		16				; 2us
min_off_time	equ		12				; 1.5us
;min_off_time	equ		8				; 1us

; output over-voltage shutdown voltage (2.944V steps)
shutdown_vol	equ		61

; supply over-voltage shutdown voltage (V steps)
shutdown_vol2	equ		24

; Comparator parameters
cm1com0_param	equ		1<<C1ON | 1<<C1POL | 1<<C1SP	; output-off, inv-pol, high-speed
cm1com1_param	equ		b'00010011'			; +input:DAC, -input:C12IN3-
cm2com0_param	equ		1<<C2ON | 1<<C2POL | 1<<C2SP	; output-off, inv-pol, high-speed
cm2com1_param	equ		b'10000001'			; int on positive, C2in+, C12IN1-
; FVR (fixed voltage reference) parameter
fvr_param		equ		1<<FVREN | 1<<CDAFVR1 | 1<<CDAFVR0	; 4.096V
; DAC parameters
daccon0_param	equ		1<<DACEN | 1<<DACPSS1
daccon1_param	equ		shutdown_vol*32/(23*4)	; output voltage = 4.096 x DACR/32

ad_threshold	equ		8				; threshold level before getting out of brightness lock/slave
debounce_cnt	equ		10				; button switch debounce count (x4.096ms)

; mininum duty value for software PWM - reference voltage ganaration
ref_min_duty	equ		2				; needs to be 2 to cancel out comparator offset
ref_max_duty	equ		32				; peak inductor current (A)(apox.) = 3*ref_max_duty/255

; brightness control range (between 0 - 255)
min_brightness	equ		1
max_brightness	equ		255

;--- ports for I/O -------------------------------
#define	PWM_OUT	LATC,5
#define	LED_1	LATC,4			; also C2OUT
#define	SPWM_OUT	LATA,2		; SPWM for reference voltage generation
#define	SPI_DAT	PORTA,0			; SPI data port
#define	SPI_CLK	PORTA,1			; SPI clock port
SPI_CLK_N		equ		1		; 
#define	SPI_IOC	IOCAN,SPI_CLK_N	; Interrupt on change port for SPI clock (negative edge)
#define	SW1		PORTA,1
#define	SW2		PORTA,0
IR_PORT_N		equ		5		; port number of the IR receiver
#define	IR_PORT	PORTA,IR_PORT_N	; port & bit of the IR receiver

;--- analog ports ---
#define	AN_IN	6
#define	AN_IN_PT	ANSELC,2
#define	C_SENSE	5
#define	C_SENSE_PT	ANSELC,1
#define	V_SENSE	7			; output voltage sense
#define	V_SENSE_PT	ANSELC,3
#define	S_SENSE	3			; supply voltage sense
#define	S_SENSE_PT	ANSELA,4
#define	V_REF_PT	ANSELC,0

;--- serial communication ------------------------
broadcast		equ		b'01010101'		; broadcast unit ID
end_packet		equ		b'10101010'		; end of packet marker
spi_timeout_val	equ		380				; x125ns
spi_timeout_h	equ		HIGH(0xFFFF-spi_timeout_val)	; high byte
spi_timeout_l	equ		LOW(0xFFFF-spi_timeout_val)		; lower byte

;--- flags ---------------------------------------
B_UPDATE		equ		0				; brightness value updated
SER_RX			equ		1				; data packet received
SER_ERR			equ		2				; error
B_LOCKED		equ		3				; brightness control locked (remote data received)
;--- A/D flags ---------------
AD_SAMP			equ		4				; A/D sample requested
SP_VOL			equ		5				; supply voltage sampling

;--- button status flags -------------------------
SW1_PUSH		equ		0
SW1_LONG		equ		1
SW2_PUSH		equ		2
SW2_LONG		equ		3
SW_CHECK		equ		4
;--- IR flags ----------------
IR_RCV			equ		5				; IR receiving flag
IR_DEV			equ		6				; IR device code receiving flag
IR_RCD			equ		7				; IR received flag

; --- IR receiver ------------------------------------------
start_bit		equ		2080/32			; minimum duration of start bit (in micro second)
bit_threshold	equ		1040/32			; threshold value used to determin the bit value(long/short pulse)(in micro second)


;===============================================================================
;	variables
;
				cblock	0x20
	; --- bank 0 (80 bytes) ------------------------------------------
	pr_value							; PWM period value
	pwm_cycles							; PWM cycle counter
	duty_h								; duty level high byte
	duty_l								; duty level low byte
	max_duty							; maximum duty/on-time
	set_brightness						; brightness value target
	brightness							; current brightness value
	ad_value							; analog value (brightness control VR)
	ad_value1							; analog value (prior value for averaging)
	sp_voltage							; 
	spwm_cnt							; software PWM counter
	ref_pwm_duty						; software PWM duty value
	rcv_data							; SPI received data
	;--- IR receiver -----------------------------
	IR_keycode							; IR key code received
	IR_device							; IR device code received
	_IR_keycode							; IR command being received (internal)
	_IR_device							; IR device code being received (internal)
	IR_timeout							; IR command timeout counter (decremented by Timer1 int)
	IR_last_key							; last key received (needs to be initialized with 0xFF)
	IR_keycount							; entered key count (count kept, but not used)
	IR_enter_cmd						; command entered
	IR_enter_val_h						; value entered by ten keys (2 bytes = 4 digits)
	IR_enter_val_l
				endc
				cblock	0x70
	; --- common memory (16 bytes) -----------------------------------
	flags								; global flags
	buttons								; button/IR status
	sw1_count							; button SW debouce counter
	sw2_count
	rcv_temp							; serial receive temp buffer
	tx_data								; serial send data
	tx_bit:0							; serial send bit (used when data is sent)
	_temp								; temporary val
				endc
				cblock	0xA0
	; --- bank 1 (32 bytes) ------------------------------------------

				endc
;
;===============================================================================
;
				org		0				; reset vector
				nop
				movlw   HIGH startup
				movwf   PCLATH
				goto	startup
;
;===============================================================================
;
				org		4				; INT vector
;				--------------------------------------------
				banksel	0					; select bank0
				btfss	INTCON,T0IF			; is it timer0 interrupt?
				goto	port_change			; otherwise port_change

				;-----------------------------------------------------
				;--- software PWM service (timer0) -------------------
				incf	spwm_cnt,f			; increment PWM phase counter
				incfsz	spwm_cnt,f			; increment PWM phase counter twice -> 244.1Hz
				goto	$+2
				bsf		buttons,SW_CHECK	; SW_CHECK flag is set when spwm_cnt=0 (4.096ms)
				movfw	ref_pwm_duty		; compare ref_pwm_duty : spwm_cnt
				subwf	spwm_cnt,w
				banksel	LATA
				btfsc	STATUS,C			; if (ref_pwm_duty < spwm_cnt)
				bcf		SPWM_OUT			; 	turn off SPWM_OUT -> reference voltage
				btfss	STATUS,C			; if (ref_pwm_duty > spwm_cnt)
				bsf		SPWM_OUT			; 	turn on SPWM_OUT -> reference voltage

				bcf		INTCON,T0IF			; clear timer0 INT flag
				goto	finish_int

				;=====================================================
port_change
				banksel	IOCAF
				btfss	IOCAF,SPI_CLK_N		; is it SPI_CLK?
				goto	IR_service			; IR interrupt otherwise
				;-----------------------------------------------------
				;--- receive SPI data --------------------------------
spi_service
				banksel	0
				movlw	b'00000001'
				movwf	rcv_temp			; initialize buffer with LSB set
				movlw	spi_timeout_h		; reset the timer for the timeout
				movwf	TMR1H
				movlw	spi_timeout_l
				movwf	TMR1L
				bcf		PIR1,TMR1IF			; clear timer1 overflow flag
				bsf		T1CON,TMR1ON		; start timer1

				;--- wait for raising edge -----------------
spi_wait
				clrwdt
				btfsc	PIR1,TMR1IF			; check if timeout has occurred
				goto	spi_timeout			; 
				btfss	SPI_CLK				; CLK high?
				goto	spi_wait

				;--- receive a data bit --------------------
				bcf		STATUS,C			; clear carry flag
				btfsc	SPI_DAT				; sample data pin
				bsf		STATUS,C			; set carry flag if the data = 1
				rlf		rcv_temp,f			; put the bit into the byte buffer
				btfsc	STATUS,C			; 8 bits received if carry set
				goto	spi_received
											; otherwise receive more bits
spi_wait_l
				clrwdt
				btfsc	PIR1,TMR1IF			; check if timeout has occurred
				goto	spi_timeout			; 
				btfsc	SPI_CLK				; CLK low?
				goto	spi_wait_l
				goto	spi_wait

spi_timeout		;--- timeout - set the error flag ----------
				bsf		flags,SER_ERR
				goto	spi_end

spi_received	;--- 8 bits received -----------------------
				movfw	rcv_temp			; copy the data to the buffer
				movwf	rcv_data
				bsf		flags,SER_RX		; set data received flag

spi_end
				bcf		T1CON,TMR1ON		; stop timer1
				banksel	IOCAF
				bcf		IOCAF,SPI_CLK_N		; clear port change bits
				banksel	INTCON
				bcf		INTCON,IOCIF		; clear INT flag

				goto	finish_int

				;-----------------------------------------------------
				;--- receive IR remote -------------------------------
IR_service
				banksel	0
				btfsc	IR_PORT				; is the IR port high?
				goto	IR_read_timer		;  IR port high - read the timer
IR_start_timer								;  IR port low - start the timer
				movlw	0xFF-start_bit		; set timer1 period to start_bit
				banksel	TMR1L
				clrf	TMR1L
				movwf	TMR1H
				bcf		PIR1,TMR1IF			; clear timer1 overflow flag
				bsf		T1CON,TMR1ON		; start timer1
				goto	IR_end
IR_read_timer
				bcf		T1CON,TMR1ON		; stop timer1
				btfsc	PIR1,TMR1IF			; timer overflow?
				goto	IR_start_bit		; then it's a start bit

				btfsc	buttons,IR_RCV		; otherwise check if start bit has already received
				goto	IR_long_short		; yes, move on to data bit

				goto	IR_end				; not a start bit, skip till start bit comes in

IR_long_short								; long or short pulse(1 or 0)?
				movlw	0xFF-start_bit+bit_threshold
				subwf	TMR1H,W				; compare timer with bit threshold
											; data bit is in the C flag
				btfsc	buttons,IR_DEV		; receiving command?
				goto	IR_dev_bit
				rrf		_IR_keycode,f		; shift into the command buffer
				goto	IR_data_end

IR_dev_bit		rrf		_IR_device,f		; shift into the device code buffer
IR_data_end
				btfss	STATUS,C			; if the C flag set, all bits in
				goto	IR_end				; else exit

				btfsc	buttons,IR_DEV		; received device code?
				goto	IR_data_done		; yes, whole packet received
				bsf		buttons,IR_DEV		; else set the flag so that device code will be received
				goto	IR_end
IR_data_done								; process the received data
				bcf		STATUS,C			; clear the C flag
				rrf		_IR_keycode,W		; shift command one more bit
				movwf	IR_keycode

				bcf		STATUS,C			; clear the C flag
				rrf		_IR_device,f		; shift code 3 more bits
				rrf		_IR_device,f
				rrf		_IR_device,W
				movwf	IR_device			; update IR_device

				bcf		buttons,IR_RCV		; clear IR receiving flag
				bcf		buttons,IR_DEV		; clear IR device code receiving flag
				bsf		buttons,IR_RCD		; set IR received flag
				goto	IR_end
IR_start_bit								; start bit came in
				movlw	b'01000000'
				movwf	_IR_keycode			; clear the command buffer (with 7th bit set)
				movlw	b'00010000'
				movwf	_IR_device			; clear the device code buffer (with 5th bit set)
				bsf		buttons,IR_RCV		; set IR receiving flag
				bcf		buttons,IR_DEV		; clear IR device receiving flag
IR_end
				banksel	IOCAF
				bcf		IOCAF,IR_PORT_N		; clear port change bits
				banksel	INTCON
				bcf		INTCON,IOCIF		; clear INT flag

finish_int
;				--------------------------------------------
				retfie
;
;===============================================================================
;	startup
;
startup
				;--- initialization --------------------------------------------
				;--- check for brown-out -----------------------------
				banksel	PCON
				btfsc	PCON,NOT_POR
				goto	not_POR
				bsf		PCON,NOT_POR		; set POR flag
				bsf		PCON,NOT_BOR		; set BOR flag
not_POR
;				btfss	PCON,NOT_BOR
;				goto	shutdown_2

				;--- check for WDT timeout ---------------------------
				btfss	STATUS,NOT_TO
				goto	shutdown_1

				clrwdt						; clear watchdog timer

				;--- initialize system clock -------------------------
				movlw	b'11110000'			; 32MHz internal oscillator
				banksel	OSCCON
				movwf	OSCCON

				;--- clear RAM ---------------------------------------
				banksel	0
				movlw	0x20				; starting RAM addr
				movwf	FSR0L				; indirect addressing
next_ram		clrf	INDF0
				incf	FSR0L,f
				btfss	FSR0L,7				; done if the bit7 on (0x80)
				goto	next_ram

				;--- set port analog/digital input -------------------
				banksel	ANSELA
				clrf	ANSELA
				clrf	ANSELC
				bsf		AN_IN_PT			; set AN_IN as analog input
				bsf		C_SENSE_PT			; set C_SENSE as analog input
				bsf		V_SENSE_PT			; set V_SENSE as analog input
				bsf		S_SENSE_PT			; set S_SENSE as analog input
				bsf		V_REF_PT			; set V_REF_PT as analog input
											; other ports are digital

				;--- initialize I/O ports ----------------------------
				banksel	LATA
				clrf	LATA
				clrf	LATC
				banksel	TRISA
				bcf		SPWM_OUT			; enable SPWM port
				bcf		LED_1				; set LED_1 port output
											; other ports are input

				;--- initialize options & WDT ------------------------
				banksel	WPUA				; configure pull-ups
				clrf	WPUA				; disable PORTA pull-ups
				clrf	WPUC				; disable PORTC pull-ups
				bsf		SW1					; enable individual pull-ups
				bsf		SW2
				bsf		IR_PORT				; enable pull-up for IR port

				clrwdt						; clear watchdog timer
				movlw	option_param		; set watchdog timer prescaler among other options
				banksel	OPTION_REG
				movwf	OPTION_REG
				movlw	wdtcon_param
				banksel	WDTCON
				movwf	WDTCON

				;--- initialize interrupts ---------------------------
				banksel	TMR0
				clrf	TMR0				; clear timer0
				bcf		INTCON,T0IF			; clear timer0 int flag
				bsf		INTCON,T0IE			; enable timer0 interrupt

				;--- interrupt for SPI communication ---
;				banksel	0
;				clrwdt						; clear watchdog timer
;				btfss	SPI_CLK				; wait till port goes HIGH
;				goto	$-2
				banksel	IOCAN
				bsf		IOCAN,SPI_CLK_N		; enable port change INT on SPI clock

				;--- initialize IR receiver ---
				banksel	0
				clrwdt						; clear watchdog timer
				btfss	IR_PORT				; wait until the receiver module stablizes to high
				goto	$-2

				banksel	IOCAN				; enable port change INT for IR port
				bsf		IOCAN,IR_PORT_N		; enable negavie edge
				banksel	IOCAP
				bsf		IOCAP,IR_PORT_N		; enable positive edge

				banksel	IOCAF
				clrf	IOCAF				; clear port change indivisual flags
				banksel	INTCON
				bcf		INTCON,IOCIF		; clear port change int flag
				bsf		INTCON,IOCIE		; enable port change interrupt
				bsf		INTCON,GIE			; enable all interrupts

				;--- set up A/D converter ----------------------------
				banksel	ADCON0
				movlw	b'00011001'			; AN6 as input, ADC enable
				movwf	ADCON0
;				movlw	1<<ADCS2|1<<ADCS1|1<<ADCS0		; left justify, A/D clock: Frc, Vdd as ref
				movlw	b'00100000'			; left justify, A/D clock: Fosc/32, Vdd as ref
				movwf	ADCON1

				;--- set up comparators ------------------------------
				banksel	FVRCON
				movlw	fvr_param			; set up voltage ref (FVR)
				movwf	FVRCON
				banksel	DACCON0
				movlw	daccon0_param		; set up DAC
				movwf	DACCON0
				movlw	daccon1_param		; set up DAC
				movwf	DACCON1
				banksel	CM1CON0
				movlw	cm1com0_param		; set up comparator 1
				movwf	CM1CON0
				movlw	cm1com1_param
				movwf	CM1CON1
				movlw	cm2com0_param		; set up comparator 2
				movwf	CM2CON0
				movlw	cm2com1_param
				movwf	CM2CON1

				clrwdt						; clear watchdog timer
				btfsc	CMOUT,MC1OUT		; wait until the C1 output is low
				goto	$-2

				banksel	PIR2
				bcf		PIR2,C2IF			; clear C2 interrupt flag

				;--- initialize PWM ----------------------------------
				banksel	PR2					; set PWM period
				movlw	pwm_pr_max
				movwf	PR2
				banksel	CCP1CON
				movlw	ccpcon_pram			; LSBs of duty cycle(bit 5:4), PWM mode set
				movwf	CCP1CON
				clrf	CCPR1L				; clear duty cycle
				movlw	t2com_param			; set postscale, timer on, prescale
				banksel	T2CON
				movwf	T2CON
				banksel	PIR1
				bcf		PIR1,TMR2IF			; clear timer2 overflow flag

				clrwdt						; clear watchdog timer
				btfss	PIR1,TMR2IF			; wait for timer2 overflow
				goto	$-1
				bcf		PIR1,TMR2IF			; clear timer2 overflow flag

				banksel	CCP1AS
				bcf		CCP1AS,CCP1ASE		; clear PWM auto-shutdown status
				bsf		CCP1AS,CCP1AS0		; enable C1 auto-shutdown

				banksel	TRISA
				bcf		PWM_OUT				; set PWM port output

				;--- startup -----------------------------------------
				banksel	0
				movlw	ref_min_duty		; set SPWM duty to mininum
				movwf	ref_pwm_duty
				movlw	pwm_pr_max			; initialize pr_value
				movwf	pr_value
				movlw	min_brightness
				movwf	set_brightness
				call	startup_status		; wait for the circuit to stabilize

				banksel	ADCON0
				bsf		ADCON0,GO			; start the first AD conversion

				;---------------------------------------------------------------
				;--- main loop -------------------------------------------------
main_loop
				banksel	CCP1AS
				btfsc	CCP1AS,CCP1ASE		; if PWM auto-shutdown occurred
				goto	shutdown_3			; 	shutdown the oparation

				;--- check the analog in ---------------------------------------
				btfss	flags,AD_SAMP		; analog sample done?
				goto	ad_convert
				banksel	ADCON0
				bsf		ADCON0,GO			; start AD conversion
				bcf		flags,AD_SAMP		; clear the flag
				goto	ad_skip
ad_convert
				banksel	ADCON0
				btfsc	ADCON0,GO			; is AD conversion done?
				goto	ad_skip
											; conversion done
				btfsc	flags,SP_VOL		; brightness control or supply voltage?
				goto	check_supply

				;--- sample brightness control -----------------------
				btfss	flags,B_LOCKED		; disable the control if brightness locked
				goto	ad_notLocked
				; check if the value change is within the threshold
				banksel	ADRESH
				movfw	ADRESH				; W = ADRESH (high 8 bits of AD value)
				banksel	0
				subwf	ad_value,w			; W = ad_value - W
				btfss	STATUS,C			; negate the result if negative
				sublw	0
				sublw	ad_threshold		; W = ad_threshold - W
				btfsc	STATUS,C
				goto	ad_noChange			; threshold not reached
				bcf		flags,B_LOCKED		; threshold reached - unlock control
ad_notLocked
				movfw	ad_value1			; ad_value = ad_value1
				movwf	ad_value
				banksel	ADRESH
				movfw	ADRESH				; W = ADRESH (high 8 bits of AD value)
				banksel	0
				movwf	ad_value1			; keep the new value in ad_value1
				addwf	ad_value,f			; avarage with the last value
				rrf		ad_value,f
				movf	ad_value,f			; test ad_value
				btfsc	STATUS,Z			; if (ad_value = 0)
				incf	ad_value,f			;   ++ad_value

				movfw	ad_value			; if (ad_value != brightness)
				subwf	set_brightness,w
				btfsc	STATUS,Z
				goto	ad_noChange			; {
				movfw	ad_value			; 	update brightness
				movwf	set_brightness
				bsf		flags,B_UPDATE		;	set the flag
ad_noChange									; }
				bsf		flags,SP_VOL		; start sampling supply voltage
				movlw	b'00001101'			; input: AN3, ADC enable
				banksel	ADCON0
				movwf	ADCON0
				banksel	0
				bsf		flags,AD_SAMP		; set the analog sample flag

				goto	ad_skip

check_supply	;--- sample supply voltage ---------------------------
				banksel	ADRESH
				movfw	ADRESH				; read the high byte
				banksel	0
				addwf	sp_voltage,f		; avarage with the last value
				rrf		sp_voltage,f
				movfw	sp_voltage

				bcf		flags,SP_VOL		; start sampling brightness control
				movlw	b'00011001'			; AN6 as input, ADC enable
				banksel	ADCON0
				movwf	ADCON0
				bsf		flags,AD_SAMP		; set the analog sample flag
ad_skip
				banksel	0

				;--- count PWM cycles ------------------------------------------
				incf	pwm_cycles,f

				;--- ramp up/down brightness -----------------------------------
				btfss	pwm_cycles,0		; update brightness every other PWM cycles
				goto	brightness_upd
				movfw	set_brightness
				subwf	brightness,w		; W = brightness - set_brightness
				btfsc	STATUS,Z
				goto	brightness_upd		; no change
				btfss	STATUS,C			; if (brightness > set_brightness)
				goto	inc_brightness		
				decf	brightness,f		;   --brightness
				goto	brightness_upd
inc_brightness								; else
				incf	brightness,f		;   ++brightness
brightness_upd
				call	get_PR_value		; look up PR value -> w

				movwf	pr_value			; set next PWM period

				addlw	0-min_off_time		; set maximum on-time/duty
				movwf	max_duty

				call	get_duty_value		; look up duty for software PWM
				movwf	ref_pwm_duty

				;--- wait for the new PWM cycle --------------------------------
				banksel	PIR1
				bcf		PIR1,TMR2IF			; clear timer2 overflow flag

				clrwdt						; clear watchdog timer
				btfss	PIR1,TMR2IF			; wait for timer2 overflow
				goto	$-1
				bcf		PIR1,TMR2IF			; clear timer2 overflow flag

				;--- adjust duty cycle based on comparator output --------------
				movf	brightness,f		; if (brightness = 0)
				btfsc	STATUS,Z
				goto	tripped				;   force the mininum duration

				btfss	PIR2,C2IF			; if comparator tripped {
				goto	no_trip
tripped
				bcf		PIR2,C2IF			;   clear C2 interrupt flag
				banksel	LATC
				bsf		LED_1				;   DEBUG - LED_1 on
				banksel	0
				movf	duty_h,f
				btfss	STATUS,Z			;   if (duty_h != 0)
				goto	dec_duty
				movf	duty_l,f			;   or (duty_l != 0) {
				btfsc	STATUS,Z
				goto	end_comparator
dec_duty
				decf	duty_l,f			; 	  decrease duty value/on time
				btfss	duty_l,7			; 	  if (duty_l < 0)
				goto	end_comparator
				movlw	3
				movwf	duty_l				;		duty_l = 3
				decf	duty_h,f			;		duty_h--
				goto	end_comparator		;	  }
no_trip										;   } else {
				banksel	LATC
				bcf		LED_1				;   DEBUG - LED_1 off
				banksel	0
				movfw	duty_h				;   if (duty_h < max_duty) {
				subwf	max_duty,w
				btfss	STATUS,C
				goto	max_duty_reached
				incf	duty_l,f			;     increase duty value/on time
				btfss	duty_l,2			;     if (duty_l > 3) {
				goto	end_comparator
				clrf	duty_l				; 		duty_l = 0
				incf	duty_h,f			; 		duty_h++
				goto	end_comparator		;	  }
max_duty_reached							;   } else {
				movfw	max_duty
				movwf	duty_h
				clrf	duty_l				;   }
											; }
end_comparator

				;--- set the next duty cycle for PWM ---------------------------
				movfw	duty_h
				banksel	CCPR1L
				movwf	CCPR1L				; CCPR1L = high 8 bits of actual duty value

				banksel	0
				swapf	duty_l,w			; set last 2 bits of duty value
				andlw	0x30
				iorlw	ccpcon_pram			; OR with the parameters
				banksel	CCP1CON
				movwf	CCP1CON				; set CCP1CON

				;--- set the new period for PWM --------------------------------
				banksel	PIR1
				bcf		PIR1,TMR2IF			; clear timer2 overflow flag

				clrwdt						; clear watchdog timer
				btfss	PIR1,TMR2IF			; wait for timer2 overflow
				goto	$-1

				movfw	pr_value
				banksel	PR2					; set PWM period
				movwf	PR2
				banksel	0

				;--- check IR status -------------------------------------------
				btfsc	buttons,IR_RCD		; check IR flag
				call	key_scan

				;--- check if data received ------------------------------------
				btfss	flags,SER_RX		; if data received flag set
				goto	no_data
				movfw	rcv_data			; set the brightness level
				movwf	set_brightness
				bsf		flags,B_LOCKED		; lock the brightness controll
				bcf		flags,SER_RX		; clear data received flag
no_data
				;--- check button status ---------------------------------------
				btfss	buttons,SW_CHECK	; check buttons if the flag set
				goto	main_loop
				call	check_buttons		; check buttons at specified interval

				;--- send data as needed ---------------------------------------
				btfss	flags,B_UPDATE		; if the update flag set
				goto	main_loop			; {

;				banksel	LATC				;   DEBUG
;				bsf		LED_1				;   DEBUG - LED_1 on
;				banksel	0					;   DEBUG

				; check if SPI lines are high level (button not pushed)
				btfss	SPI_DAT				;   skip if SPI_DAT is low
				goto	main_loop
				btfss	SPI_CLK				;   skip if SPI_CLK is low
				goto	main_loop

				; disable the port change interrupt (serial receive)
				bcf		INTCON,IOCIE		;   disable port change interrupt

				banksel	LATA
				bsf		SPI_DAT				;   set serial data high
				bcf		SPI_CLK				;   set serial clock low
				banksel	TRISA
				bcf		SPI_DAT				;   set serial port to output
				bcf		SPI_CLK				;   set serial port to output
				banksel	0

				movfw	set_brightness		;   send brightness data
				call	send_data

				banksel	TRISA
				bsf		SPI_DAT				;   set serial port back to input
				bsf		SPI_CLK				;   set serial port back to input

				; enable the port change interrupt (serial receive)
				banksel	IOCAF
				clrf	IOCAF				;   clear individual flags
				bcf		INTCON,IOCIF		;   clear port change int flag
				bsf		INTCON,IOCIE		;   enable port change interrupt

				bcf		flags,B_UPDATE		;   clear the flag

;				banksel	LATC				;   DEBUG
;				bcf		LED_1				;   DEBUG - LED_1 off
;				banksel	0					;   DEBUG

				goto	main_loop			; }

;-------------------------------------------------------------------------------
;

brightness_down
				; lower the set_brightness
				movfw	set_brightness
				btfsc	STATUS,Z			; if (set_brightness = 0)
				return						;   return
				addlw	0-0x10				; w = set_brightness - 0x10
				btfss	STATUS,C			; if (set_brightness < 0)
				movlw	min_brightness		;     set_brightness = min_brightness
				btfsc	STATUS,Z			; or (set_brightness = 0)
				movlw	min_brightness		;     set_brightness = min_brightness
				movwf	set_brightness
				bsf		flags,B_LOCKED		; lock the brightness controll
				return						; 

brightness_up
				; up the set_brightness
				comf	set_brightness,w
				btfsc	STATUS,Z			; if (set_brightness = 0xFF)
				return						;   return
				movfw	set_brightness
				addlw	0x10
				btfsc	STATUS,C			; if (set_brightness > 0xFF)
				movlw	max_brightness		;   set_brightness = max_brightness
				movwf	set_brightness
				bsf		flags,B_LOCKED		; lock the brightness controll
				return

brightness_max
				; maximum brightness
				movlw	max_brightness		; set_brightness = max_brightness
				movwf	set_brightness
				bsf		flags,B_LOCKED		; lock the brightness controll
				return

brightness_off
				; turn off - zero brightness
				clrf	set_brightness
				bsf		flags,B_LOCKED		; lock the brightness controll
				return

;-------------------------------------------------------------------------------
;	PR value lookup tables
;
get_PR_value
				;--- look up table via brightness ---
				movlw	HIGH (pwm_pr_tbl)
				movwf	PCLATH
				movfw	brightness
				addlw	LOW (pwm_pr_tbl)
				btfsc	STATUS,C
				incf	PCLATH,f			; increment PCLATH if carry over
				movwf	PCL					; table jump

pwm_pr_tbl		; 
	variable x = 0
	while	x <= 254
				dt		pwm_pr_max-((pwm_pr_max-pwm_pr_min)*x/254)
x ++
	endw
				dt		pwm_pr_min

;-------------------------------------------------------------------------------
;	PWM value lookup tables for reference voltage (software PWM)
;
get_duty_value
				;--- look up table via brightness ---
				movlw	HIGH (pwm_duty_tbl)
				movwf	PCLATH
				movfw	brightness
				addlw	LOW (pwm_duty_tbl)
				btfsc	STATUS,C
				incf	PCLATH,f			; increment PCLATH if carry over
				movwf	PCL					; table jump

pwm_duty_tbl
	variable x = 0
	while	x <= 254
						; interpolate between ref_min_duty and ref_max_duty
				dt		ref_min_duty+((ref_max_duty-ref_min_duty)*x/254)
x ++
	endw
				dt		ref_max_duty

;===============================================================================
;	Button switch service
;
check_buttons
				banksel	PORTA
				btfsc	SW1					; if SW1 pushed {
				goto	sw1_released
				incfsz	sw1_count,f			;   increase sw counter
				goto	sw1_end				;   if sw counter rolled over {
				decf	sw1_count,f			;     keep sw1_count at max
				bsf		buttons,SW1_LONG	;     set SW1 long pushed flag
				goto	sw1_end				;   }
											; }
sw1_released
				comf	sw1_count,w			; if (sw count = 0xFF)
				btfsc	STATUS,Z
				goto	sw1_deffer			;   ignore this event
				movlw	debounce_cnt		; if sw count > debounce count
				subwf	sw1_count,w
				btfsc	STATUS,C
				bsf		buttons,SW1_PUSH	; set SW1 pushed flag
sw1_deffer		clrf	sw1_count			; clear sw on counter
sw1_end
				btfsc	SW2					; if SW2 pushed {
				goto	sw2_released
				incfsz	sw2_count,f			;   increase sw counter
				goto	sw2_end				;   if sw counter rolled over {
				decf	sw2_count,f			;     keep sw2_count at max
				bsf		buttons,SW2_LONG	;     set SW2 long pushed flag
				goto	sw2_end				;   }
											; }
sw2_released
				comf	sw1_count,w			; if (sw count = 0xFF)
				btfsc	STATUS,Z
				goto	sw2_deffer			;   ignore this event
				movlw	debounce_cnt		; if sw count > debounce count
				subwf	sw2_count,w
				btfsc	STATUS,C
				bsf		buttons,SW2_PUSH	; set SW2 pushed flag
sw2_deffer		clrf	sw2_count			; clear sw on counter
sw2_end
				bcf		buttons,SW_CHECK	; clear SW check flag

				;--- SW functions dispatch ---
				btfss	buttons,SW1_LONG	; sw1 long pushed?
				goto	sw1_skip
				call	brightness_off
				bcf		buttons,SW1_PUSH
sw1_skip
				btfsc	buttons,SW1_PUSH	; sw1 pushed?
				call	brightness_down

				btfss	buttons,SW2_LONG	; sw2 long pushed?
				goto	sw2_skip
				call	brightness_max
				bcf		buttons,SW2_PUSH	; 
sw2_skip
				btfsc	buttons,SW2_PUSH	; sw2 pushed?
				call	brightness_up
end_buttons
				bcf		buttons,SW1_PUSH	; clear button flags
				bcf		buttons,SW2_PUSH	; 
				bcf		buttons,SW1_LONG	; 
				bcf		buttons,SW2_LONG	; 

				return

;===============================================================================
;	IR key scan
;
key_scan
				bcf		buttons,IR_RCD		; clear the IR received flag

				movlw	IR_key_0 +1			; is it a number key?
				subwf	IR_keycode,W
				btfsc	STATUS,C
				goto	next_key2
ten_keys
				incf	IR_keycode,W
				movwf	_temp				; value: 0-9
											; do something
				goto	key_scan_end
next_key2
				movlw	IR_key_Ch_up		; is it the 'CH+' key?
				subwf	IR_keycode,W
				btfss	STATUS,Z
				goto	next_key3
				call	brightness_up		; brightness up
				goto	key_scan_end
next_key3
				movlw	IR_key_Ch_down		; is it the 'CH-' key?
				subwf	IR_keycode,W
				btfss	STATUS,Z
				goto	next_key4
				call	brightness_down		; brightness down
				goto	key_scan_end
next_key4
				movlw	IR_key_Muting		; is it the 'mute' key?
				subwf	IR_keycode,W
				btfsc	STATUS,Z
				call	brightness_off		; turn off output
key_scan_end
				return

;===============================================================================
;	pause (W changes the loop) pause time is about 1 sec @ W = 255
;
pause
				clrf	_temp
pause_loop
				clrwdt						; clear watchdog timer
				goto	$+1					; take extra time
				goto	$+1					; abjust these lines to calibrate time
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				goto	$+1
				incfsz	_temp,f
				goto	pause_loop
				addlw	-1					; W--
				btfsc	STATUS,C
				goto	pause_loop
				return

;===============================================================================
;	software synchronous serial send - send out tx_data (8 bit)
;
send_data
				movwf	tx_data				; copy data from W

				; prepare for the data send
				movlw	b'10000000'
				movwf	tx_bit				; tx_bit points to the bit being sent
send_data_bit
				andwf	tx_data,W			; test the bit being sent
				btfss	STATUS,Z
				call	_send_bit_high
				btfsc	STATUS,Z
				call	_send_bit_low

				bcf		STATUS,C			; clear carry flag
				rrf		tx_bit,f			; tx_bit points to the bit being sent
				movf	tx_bit,W
				btfss	STATUS,C			; 8 bits done if carry is set
				goto	send_data_bit

				return
;
_send_bit_low	;--- send low bit to the port on timing ---
				banksel	LATA
				bcf		SPI_CLK				; drive the clock low
				bcf		SPI_DAT				; set data port low
				goto	$+1					; give a little time
				goto	$+1
				bsf		SPI_CLK				; drive the clock high
				banksel	0
				return
;
_send_bit_high	;--- send high bit to the port on timing ---
				banksel	LATA
				bcf		SPI_CLK				; drive the clock low
				bsf		SPI_DAT				; set data port high
				goto	$+1					; give a little time
				goto	$+1
				bsf		SPI_CLK				; drive the clock high
				banksel	0
				return

;===============================================================================
;	Startup sequence
;
startup_status
				banksel	LATA
				bsf		LED_1				; turn on LED1

				movlw	80
				call	pause

				bcf		LED_1				; turn off LED1

				movlw	80
				call	pause

				bsf		LED_1				; turn on LED1

				movlw	80
				call	pause

				bcf		LED_1				; turn off LED1
				banksel	0

				return

;===============================================================================
;	Error shurdown
;
shutdown_1
				bcf		INTCON,GIE			; disable all interrupts

				movlw	b'00111010'			; use 500kHz internal oscillator
				banksel	OSCCON
				movwf	OSCCON

				banksel	TRISA
				movlw	0xFF
				movwf	TRISA				; set all ports input/high-impedance
				movwf	TRISC
				bcf		LED_1				; LED1 output

				banksel	LATA
				bcf		LED_1				; turn off LED1
				movlw	8
				call	pause

											;--- flash LED once ---
				bsf		LED_1				; turn on LED1
				movlw	1
				call	pause

				goto	shutdown_1

shutdown_2
				bcf		INTCON,GIE			; disable all interrupts

				movlw	b'00111010'			; use 500kHz internal oscillator
				banksel	OSCCON
				movwf	OSCCON

				banksel	TRISA
				movlw	0xFF
				movwf	TRISA				; set all ports input/high-impedance
				movwf	TRISC
				bcf		LED_1				; LED1 output

				banksel	LATA
				bcf		LED_1				; turn off LED1
				movlw	8
				call	pause
											;--- flash LED twice ---
				bsf		LED_1				; turn on LED1
				movlw	1
				call	pause
				bcf		LED_1				; turn off LED1
				movlw	1
				call	pause

				bsf		LED_1				; turn on LED1
				movlw	1
				call	pause

				goto	shutdown_2

shutdown_3
				bcf		INTCON,GIE			; disable all interrupts

				movlw	b'00111010'			; use 500kHz internal oscillator
				banksel	OSCCON
				movwf	OSCCON

				banksel	TRISA
				movlw	0xFF
				movwf	TRISA				; set all ports input/high-impedance
				movwf	TRISC
				bcf		LED_1				; LED1 output

				banksel	LATA
				bcf		LED_1				; turn off LED1
				movlw	8
				call	pause
											;--- flash LED three times ---
				bsf		LED_1				; turn on LED1
				movlw	1
				call	pause
				bcf		LED_1				; turn off LED1
				movlw	1
				call	pause
				bsf		LED_1				; turn on LED1
				movlw	1
				call	pause
				bcf		LED_1				; turn off LED1
				movlw	1
				call	pause
				bsf		LED_1				; turn on LED1
				movlw	1
				call	pause

				goto	shutdown_3

;===============================================================================
				end
